#include <allegro.h>
#include <alleggl.h>

#include "engine/GLBitmap.h"
#include "engine/TextRenderer.h"
#include "engine/TimeKeeper.h"
#include "misc/Settings.h"
#include "misc/TextHandler.h"
#include "misc/Random.h"
#include "misc/KeyReader.h"
#include "things/Game.h"
#include "things/Ship.h"
#include "things/Panel.h"
#include "things/Planet.h"
#include "things/IWindow.h"
#include "things/Info.h"
#include "things/Rock.h"
#include "things/Sun.h"
#include "things/StarFlare.h"
#include "things/MenuWindow.h"
#include "sound/SoundSystem.h"

/********************************************************/

void initAllegro()
{
	allegro_init();

	Settings::loadSettings();

	install_allegro_gl();
	
	allegro_gl_set(AGL_DOUBLEBUFFER, true);
	allegro_gl_set(AGL_COLOR_DEPTH, 32);
	allegro_gl_set(AGL_RENDERMETHOD, true);
	
	int suggested = AGL_DOUBLEBUFFER | AGL_COLOR_DEPTH | AGL_RENDERMETHOD;
	
	if (Settings::setting[GFX_FULLSCREEN])
	{
		allegro_gl_set(AGL_FULLSCREEN, true);
		suggested |= AGL_FULLSCREEN;
	}
	else
	{
		allegro_gl_set(AGL_WINDOWED, true);
		suggested |= AGL_WINDOWED;
	}
	
	allegro_gl_set(AGL_REQUIRE, suggested);
	
	set_color_depth(32);
	request_refresh_rate(100);
	
	if (set_gfx_mode(GFX_OPENGL, 640, 480, 0, 0) != 0)
	{
		allegro_gl_set(AGL_SUGGEST, suggested);
		if (set_gfx_mode(GFX_OPENGL, 640, 480, 0, 0) != 0)
		{
			allegro_message("Error setting up display: %s", allegro_error);
			exit(-1);
		}
	};
	
	rest(500);
	
	install_timer();
	install_keyboard();
}

/***************************************************/

void setUpGL()
{
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	glOrtho(0, 640, 480, 0, 5, -5);
	glMatrixMode(GL_MODELVIEW);
	
	glEnable(GL_POINT_SMOOTH);
	glClearColor(0.0, 0.0, 0.0, 1.0);
}

/***************************************************/

void init()
{
	initAllegro();
	setUpGL();
	
	set_window_title("Starship Survivor");
	if (set_display_switch_mode(SWITCH_BACKGROUND) == -1)
		set_display_switch_mode(SWITCH_BACKAMNESIA);
	
	set_close_button_callback(NULL);
	
	Ship::shipBMP = GLBitmap::load("gfx/ship.pcx", GLBitmap::TF_MAGIC_PINK);
	Ship::shieldBMP = GLBitmap::load("gfx/shield.pcx", GLBitmap::TF_MAGIC_PINK);
	Panel::panelBMP = GLBitmap::load("gfx/panel.pcx", GLBitmap::TF_MAGIC_PINK);
	Planet::planetBMP[0] = GLBitmap::load("gfx/planet.pcx", GLBitmap::TF_MAGIC_PINK);
	Planet::planetBMP[1] = GLBitmap::load("gfx/planet_el.pcx", GLBitmap::TF_MAGIC_PINK);
	Planet::atmoBMP = GLBitmap::load("gfx/atmosphere.pcx", GLBitmap::TF_MAGIC_PINK);
	IWindow::cornerBMP = GLBitmap::load("gfx/win_corner.pcx", GLBitmap::TF_MAGIC_PINK);
	IWindow::borderBMP = GLBitmap::load("gfx/win_border.pcx", GLBitmap::TF_MAGIC_PINK);
	Rock::rockBMP = GLBitmap::load("gfx/rock.pcx", GLBitmap::TF_MAGIC_PINK);
	Info::infoBMP[0] = GLBitmap::load("gfx/select_to_hyper.pcx", GLBitmap::TF_MAGIC_PINK);
	Info::infoBMP[1] = GLBitmap::load("gfx/a_to_dock.pcx", GLBitmap::TF_MAGIC_PINK);
	Info::infoBMP[2] = GLBitmap::load("gfx/b_to_fuel.pcx", GLBitmap::TF_MAGIC_PINK);
	Sun::sunBMP = GLBitmap::load("gfx/sun.pcx", GLBitmap::TF_MAGIC_PINK);
	Sun::particleBMP = GLBitmap::load("gfx/sun_particle.pcx", GLBitmap::TF_NO_ALPHA);
	StarFlare::flareBMP = GLBitmap::load("gfx/star_flare.pcx", GLBitmap::TF_MAGIC_PINK);
	
	initRandom();
	
	SoundSystem::loadAll();
	soundSys = new SoundSystem();
}

/***************************************************/

void destroy()
{
	Settings::saveSettings();
	
	delete soundSys;
	SoundSystem::freeAll();
	
	TextHandler::freeFonts();
}

/***************************************************/

std::string instructions = "Control your ship with \"z\", \"x\", Space and Enter. Try to survive. In order to do that, you must have food, power, and fuel. "
 "You can obtain them by visiting planets, catching solar flares and sucking up atmosphere (but only the red kind). Also, try to collect "
 "hyperium in your free time. When you have enough hyperium (and power), press Space to warp. This will take you to a harder level. After "
 "three such levels you win the game. For more information read entry.txt.";

/***************************************************/

int main()
{
	init();
	
	soundSys->playSound(SND_BACKLOOP, 128, 1.0, 1.0, true);
	std::string ops[] = {"Start game", "Instructions", "Exit"};
	std::string iOps[] = {"OK"};
	MenuWindow *mWindow = new MenuWindow("", ops, 3, 0, 200);
	MenuWindow *iWindow = new MenuWindow(instructions, iOps, 1, 0, 450);
	iWindow->y = 240.0 - iWindow->height * 0.5;
	
	TimeKeeper tk(0.01, 0.025, 2, true);
	
	bool instrMode = false;
	
	do
	{
		bool end = false;
		mWindow->selected = -1;
		iWindow->selected = -1;
		tk.reset();
		KeyReader::clearBuffer();
		
		MenuWindow *theWindow = (instrMode) ? iWindow : mWindow;
		
		while(!end)
		{
			double dt = tk.getDT();
			
			do
			{
				theWindow->step(dt);
				
				if (theWindow->selected > -1)
					end = true;
	
				KeyReader::update();
			}
			while(tk.getFramesToSkip() > 0);
			
			glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
			
			glLoadIdentity();
			
			TextRenderer *tr = TextHandler::getInstance()->getRenderer("wet");
			tr->setSize(1.0, 1.0);
			tr->render(320.0, 50.0, 0.5, 0.5, GLColor(1.0, 1.0, 1.0), "S t a r s h i p    S u r v i v o r");
			tr->setSize(0.3, 0.3);
			tr->render(320.0, 90.0, 0.5, 0.5, GLColor(0.8, 0.8, 0.8), "A SpeedHack effort by Jakub Wasilewski");
			
			theWindow->render();
			
			allegro_gl_flip();
		}
		
		if (!instrMode) 
		{
			if (mWindow->selected == 0)
			{
				Game game;
				game.play();
			}
			else if (mWindow->selected == 1)
				instrMode = true;
		}
		else
			instrMode = false;
		
	} while (mWindow->selected != 2);
	
	delete mWindow; delete iWindow;
	
	destroy();
	
	return 0;
}
END_OF_MAIN();
