#include <allegro.h>
#include <cmath>

#include "../engine/GLBitmap.h"
#include "../entity/GameWorld.h"
#include "../things/Planet.h"
#include "../things/IWindow.h"
#include "../things/StarFlare.h"
#include "../misc/KeyReader.h"
#include "../sound/SoundSystem.h"

#include "Ship.h"

/***************************************************/

GLBitmap *Ship::shipBMP = NULL, *Ship::shieldBMP = NULL;

/***************************************************/

Ship::Ship(double x, double y)
	: Mover(x, y)
{
	vx = vy = angle = 0.0;
	foodCap = 100.0; powerCap = 100.0; fuelCap = 200.0; hyper = 0.0;
	fuel = fuelCap; food = foodCap * 0.5; power = powerCap * 0.7; hyperCap = 100.0;
	aliveFlag = true; canDock = false; canFuel = canHyper = goHyper = hypered = false;
	
	colModel.quads.push_back(Quad(-5.0, -13.5, 5.0, 13.5));
	colModel.quads.push_back(Quad(-18.0, -1.5, 18.0, 8.5));
	colModel.scale = 0.5;
	colModel.XYLimit = 35.0;
	
	colModel.rotation = 0.0;
	colModel.modelX = x;
	colModel.modelY = y;
	
	attachedTo = NULL;
	
	for (int i = 0; i < ARTIFACT_COUNT; i++)
		hasArtifact[i] = false;
	
	thrustSound = -1;
}

Ship::~Ship()
{
	if (thrustSound != -1)
		FSOUND_StopSound(thrustSound);
}
/***************************************************/

void Ship::step(double dt)
{
	// keys
	
	if (key[KEY_LEFT])
		angle += 4.0 * dt;
	if (key[KEY_RIGHT])
		angle -= 4.0 * dt;
	if (key[KEY_UP] && fuel > 0.0 && power > 0.0)
	{
		vx -= std::sin(angle) * 200.0 * dt;
		vy -= std::cos(angle) * 200.0 * dt;
		
		fuel -= 10.0 * dt;
		if (!hasArtifact[BATTERY])
			power -= 0.5 * dt;
		if (hasArtifact[DEVICE])
		{
			hyper += 0.7 * dt;
			if (hyper > hyperCap)
				hyper = hyperCap;
		}
		
		
		if (fuel < 0.0)
			fuel = 0.0;
		if (power < 0.0)
			power = 0.0;
		
		attachedTo = NULL;
		
		if (thrustSound == -1)
			thrustSound = soundSys->playSound(SND_THRUST, 128, 0.5, 1.0, true);
	}
	else if (thrustSound != -1)
	{
		FSOUND_StopSound(thrustSound);
		thrustSound = -1;
	}
	
	// docking
	canDock = false;
	canFuel = false;
	
	for (EL_ITERATOR it = planets.begin(); it != planets.end(); it++)
	{
		Planet *p = (Planet*)(*it);
		if (p->checkDocking(this, dt))
		{
			canDock = true;
			if (key[KEY_Z])
				p->dock(this, panel);
		}
		
		canFuel |= p->canFuel;
	}
	
		
	
	// actual update
	
	if (!attachedTo)
	{
		x += vx * dt;
		y += vy * dt;
	}
	else
	{
		vx = (attachedTo->x + relX - x) / dt;
		vy = (attachedTo->y + relY - y) / dt;
		x = attachedTo->x + relX;
		y = attachedTo->y + relY;
		
		power -= 2.0 * dt;
		if (power < 0.0) power = 0.0;
	}
	
	food -= 0.5 * dt;

	if (food < 0.0)
		food = 0.0;

	if (food * power * fuel == 0.0)
	{
		std::string message;
		if (food == 0.0)
			message = "You ran out of food supplies and shortly after that, started feeling weaker and weaker. Eventually, you starved to your death onboard your own ship.";
		else if (power == 0.0)
			message = "With the last drop of power drained from the batteries of your ship, the life support systems fail. You die.";
		else
			message = "Without fuel all that is left is hopeless drifting through space, without the ability to land or dock. Game over.";
		
		IWindow *newWindow = new IWindow(message, 350);
		parent->paused = true;
		parent->add(newWindow, true);
		aliveFlag = false;
	}
	
	canHyper = (hyper == hyperCap) && (power > 40.0);
	if (key[KEY_SPACE] && canHyper)
	{
		attachedTo = NULL;
		
		hyper = 0.0;
		power -= 40.0;
		
		goHyper = true;
		hyperCounter = 1.82;
		
		soundSys->playSound(SND_WARP, 128, 0.4, 0.8);
	}
	if (goHyper)
	{
		vx -= std::sin(angle) * 1200.0 * dt;
		vy -= std::cos(angle) * 1200.0 * dt;
		
		hyperCounter -= dt;
		if (hyperCounter <= 0.0)
		{
			hypered = true;
			goHyper = false;
		}
	}
	
	// collision update
	
	colModel.rotation = 0.0;
	colModel.modelX = x;
	colModel.modelY = y;
	
	// pause check
	
	for(KEY_ITERATOR k = KeyReader::scancodes.begin(); k != KeyReader::scancodes.end(); k++)
	{
		int scancode = (*k);
		if (scancode == KEY_ENTER)
		{
			showArtifacts();
			KeyReader::clearBuffer();
		}
	}
}

/**************************************************/

void Ship::render()
{
	shipBMP->setScale(0.5, 0.5);
	shipBMP->setOrigin(0.5, 0.5);
	shipBMP->setRotation(angle * 180 / M_PI);
	shipBMP->draw(x, y);
	
	if (attachedTo || goHyper)
	{
		shieldBMP->setScale(0.5, 0.5);
		shieldBMP->setOrigin(0.5, 0.5);
		shieldBMP->setTransparencyMode(GLBitmap::TM_ALPHA_BLEND);
		shieldBMP->setTint(GLColor(1.0, 1.0, 1.0, 0.7));
		shieldBMP->draw(x, y);
	}
}

/**************************************************/

void Ship::renewPlanets()
{
	planets = parent->getEntitiesOfType(ETYPE_PLANET); 
};

/**************************************************/

bool Ship::contains(double xx, double yy)
{
	return colModel.pointInside(xx, yy);
}

/**************************************************/

void Ship::processCollision(Entity *area, Entity *collider)
{
	if (!this->isAlive())
		return;
	if (attachedTo || goHyper)
		return;
	
	if (area == this)
	{
		switch(collider->type())
		{
			case ETYPE_ROCK:
				
			//!!!!!!!!!
			//return;
			
			parent->add(new IWindow("The hull of your ship was breached by a small floating asteroid. A shame, really.", 250), true);
			parent->paused = true;
			aliveFlag = false;
			break;
			
			case ETYPE_FLARE:
			power += (hasArtifact[CRYSTAL]) ? 10.0 : 7.0;
			if (power > powerCap)
				power = powerCap;
			((StarFlare*)collider)->age = 1000.0;
			soundSys->playSound(SND_PICKFLARE);
			break;
		}
	}
	else 
	{
		//!!!!!!!!!
		//	return;
		
		if (area->type() == ETYPE_PLANET && this->isAlive())
		{
			parent->add(new IWindow("You smashed into a planet... so much for survival. Game over.", 250), true);
			parent->paused = true;
			aliveFlag = false;
		}
		else if (area->type() == ETYPE_SUN && this->isAlive())
		{
			parent->add(new IWindow("Your ship got melted in the hot innards of the sun. May the light shine on your poor soul.", 250), true);
			parent->paused = true;
			aliveFlag = false;
		}
	}
}

/****************************************************/

void Ship::showArtifacts()
{
	std::string message = "Possessed artifacts: \n \n ";
	
	for (int i = 0; i < ARTIFACT_COUNT; i++)
		if (hasArtifact[i])
			message += artifactData[i][0] + string(" \n ");
	
	if (message == "Possessed artifacts: \n \n ")
		message += "None";
	
	IWindow *win = new IWindow(message, 300);
	win->y = 240.0 - win->height * 0.5;
	
	parent->add(win, true);
	parent->paused = true;
	
	soundSys->playSound(SND_SELECT);
}
