#include <cmath>
#include <string>

#include "../entity/GameWorld.h"
#include "../engine/GLBitmap.h"
#include "../engine/TextRenderer.h"
#include "../misc/TextHandler.h"
#include "../misc/Random.h"
#include "../things/IWindow.h"
#include "../things/Ship.h"
#include "../things/PlanetMenu.h"

#include "Planet.h"

/*********************************************/

GLBitmap *Planet::planetBMP[2], *Planet::atmoBMP = NULL;
Planet   *Planet::lastVisited = NULL;

/*********************************************/

Planet::Planet(Entity *anchor, double selfRadius, double mrX, double mrY, double speed, double angle, 
		   GLColor color, int kind, double foodValue, double hyperValue, double eventValue, double atmoSize, double atmoType, std::string name)
	: anchor(anchor), sRadius(selfRadius), mRadiusX(mrX), mRadiusY(mrY), speed(speed), angle(angle), color(color), name(name), atmoSize(atmoSize), atmoType(atmoType), foodValue(foodValue), hyperValue(hyperValue), eventValue(eventValue), kind(kind)
{
	dockCounter = 1.0;
	dockRadius = (sRadius + 35.0) * (sRadius + 35.0);
	fuelRadius = (sRadius + atmoSize) * (sRadius + atmoSize);
	canFuel = false;
}

/*********************************************/

void Planet::step(double dt)
{
	angle += speed * dt;
	
	if (anchor)
		{ x = anchor->getX(); y = anchor->getY(); }
	else
		{ x = y = 0.0; }
	
	x += std::cos(angle) * mRadiusX;
	y += -std::sin(angle) * mRadiusY;
}

/*********************************************/

void Planet::render()
{
	if (atmoSize > 0.0)
	{
		double scale = (sRadius + atmoSize) * (1.0 / 27.0);
		atmoBMP->setScale(scale, scale);
		atmoBMP->setTint(GLColor(atmoType, (1.0 - atmoType) * 0.5, (1.0 - atmoType)));
		atmoBMP->setOrigin(0.5, 0.5);
		atmoBMP->draw(x, y);
	}
	planetBMP[kind]->setScale(sRadius * (1.0 / 64.0), sRadius * (1.0 / 64.0));
	planetBMP[kind]->setTint(color);
	planetBMP[kind]->setOrigin(0.5, 0.5);
	
	planetBMP[kind]->draw(x, y);
	
	TextRenderer *tr = TextHandler::getInstance()->getRenderer("wet");
	tr->setSize(0.3, 0.3);
	tr->printf(x + sRadius * 0.5 + 2, y + sRadius * 0.5 + 2, 0.0, 0.0, GLColor(0.0, 0.0, 0.0), name.c_str());
	if (lastVisited == this)
		tr->printf(x + sRadius * 0.5, y + sRadius * 0.5, 0.0, 0.0, GLColor(0.6, 0.6, 0.6), name.c_str());
	else
		tr->printf(x + sRadius * 0.5, y + sRadius * 0.5, 0.0, 0.0, GLColor(1.0, 1.0, 1.0), name.c_str());
}

/*********************************************/

bool Planet::contains(double xx, double yy)
{
	double distSquared = (xx - x) * (xx - x) + (yy - y) * (yy - y);
	
	return (distSquared < (sRadius + 5.0) * (sRadius + 5.0));
}

/*********************************************/

bool Planet::checkDocking(Ship *ship, double dt)
{
	if (anchor) return false;

	canFuel = false;

	if (ship->attachedTo == this) return false;

	double distSquared = (ship->getX() - x) * (ship->getX() - x) + (ship->getY() - y) * (ship->getY() - y);
	if (atmoType > 0.5)
	{
		if (distSquared < fuelRadius)
		{
			canFuel = true;
			if (key[KEY_X])
			{
				ship->fuel += atmoType * 40.0 * dt * (ship->hasArtifact[SUCKER] ? 2.0 : 1.0);
				if (ship->fuel > ship->fuelCap)
					ship->fuel = ship->fuelCap;
			}
		}
	}
	
	if (lastVisited == this) return false;
	
	if (distSquared > dockRadius)
		dockCounter = 1.0;
	else
		dockCounter -= 1.3 * dt;
	
	
	return (dockCounter <= 0.0);
}

/**********************************************/

void Planet::dock(Ship *ship, Panel *panel)
{
	PlanetMenu *pm = new PlanetMenu(this, ship, panel);
	pm->play();
	delete pm;
	
	ship->attachedTo = this;
	ship->relX = ship->x - x;
	ship->relY = ship->y - y;
	
	double dist = std::sqrt(ship->relX * ship->relX + ship->relY * ship->relY);
	ship->relX = (sRadius + 12.0) * ship->relX / dist;
	ship->relY = (sRadius + 12.0) * ship->relY / dist;
	
	lastVisited = this;
}
