#define EL_ITERATOR list<Entity*>::iterator

/***********************************************/

#include <list>
#include <algorithm>

#include "../engine/GLColor.h"
#include "../misc/Random.h"
#include "../misc/Settings.h"
#include "../entity/Entity.h"
#include "../entity/Sorters.h"

#include "GameWorld.h"

using namespace std;

/************************************************/

GameWorld::GameWorld()
{
	paused = false;
}

GameWorld::~GameWorld()
{
	EL_ITERATOR it;
	
	for (it = autoDestroyList.begin(); it != autoDestroyList.end(); it++)
		delete *it;
}

/************************************************/

void GameWorld::insertIntoList(list<Entity*>& elist, Entity *e, bool (*greaterFunc)(Entity*, Entity*) )
{
	EL_ITERATOR it;
	
	for (it = elist.begin(); it != elist.end(); it++)
		if (greaterFunc(e, (*it)))
			break;
	
	elist.insert(it, e);
}

/************************************************/

void GameWorld::add(Entity *e, bool autoDestroy)
{
	masterList.push_back(e);
	
	if (e->type() != ETYPE_UNKNOWN)
		typedList.push_back(e);
	
	if (autoDestroy)
		autoDestroyList.push_back(e);
	
	e->setParent(this);
	
	if (e->isArea())
		areas.push_back(e);

	if (e->renderPriority() >= 0)
		insertIntoList(renderSorted, e, renderPriGT);
	if (e->stepPriority() >= 0)
		insertIntoList(stepSorted, e, stepPriGT);
	if (e->collisionPriority() >= 0)
		insertIntoList(colSorted, e, collisionPriGT);
}

/************************************************/

void GameWorld::remove(Entity *e)
{
	masterList.remove(e);
	
	if (e->type() != ETYPE_UNKNOWN)
		typedList.remove(e);
	
	e->setParent(NULL);
	
	if (e->renderPriority() >= 0)
		renderSorted.remove(e);
	if (e->stepPriority() >= 0)
		stepSorted.remove(e);
	if (e->collisionPriority() >= 0)
		colSorted.remove(e);
	if (e->isArea())
		areas.remove(e);
	
	if (find(autoDestroyList.begin(), autoDestroyList.end(), e) != autoDestroyList.end())
	{
		delete e;
		autoDestroyList.remove(e);
	}
}

/*************************************************/

list<Entity*> GameWorld::getEntities()
{
	return masterList;
}

list<Entity*> GameWorld::getEntitiesOfType(int type)
{
	EL_ITERATOR it;
	list<Entity*> newList;
	
	for (it = typedList.begin(); it != typedList.end(); it++)
		if ((*it)->type() == type)
			newList.push_back(*it);
	
	return newList;
}

int GameWorld::countEntitiesOfType(int type)
{
	int count = 0;
	EL_ITERATOR it;
	
	for (it = typedList.begin(); it != typedList.end(); it++)
		if ((*it)->type() == type)
			count++;
	
	return count;
}

/*************************************************/

void GameWorld::processRender()
{
	//glPushMatrix();
	
	EL_ITERATOR it;
	for (it = renderSorted.begin(); it != renderSorted.end(); it++)
	{
		(*it)->render();
	}
	
	//glPopMatrix();
}

/*************************************************/

void GameWorld::processStep(double dt)
{
	EL_ITERATOR it;
	
	for (it = stepSorted.begin(); it != stepSorted.end(); it++)
	{
		if (!paused || (*it)->type() > 128)
		{
			if ((*it)->isAlive())
			{
				(*it)->step(dt);
				
				if (!(*it)->isAlive() && (*it)->type() != ETYPE_SHIP)
					removeQueue.push_back(*it);
			}
			else
				removeQueue.push_back(*it);
		}
	}
	
	for (it = removeQueue.begin(); it != removeQueue.end(); it++)
	{
		remove(*it);
	}
	
	removeQueue.clear();
}

/*************************************************/

void GameWorld::processCollisions()
{
	if (paused)
		return;
	
	EL_ITERATOR objIt, areaIt;
	
	for (objIt = colSorted.begin(); objIt != colSorted.end(); objIt++)
	{
		for (areaIt = areas.begin(); areaIt != areas.end(); areaIt++)
		{
			
			if ((*objIt) != (*areaIt)   &&   (*areaIt)->contains((*objIt)->getX(), (*objIt)->getY()))
			{
				(*areaIt)->processCollision(*areaIt, *objIt);
				(*objIt)->processCollision(*areaIt, *objIt);
			}
		}
	}
}

