#include <cmath>
#include <alleggl.h>
#include <string>

#include "../things/Ship.h"
#include "../things/Panel.h"
#include "../things/Starfield.h"
#include "../things/Planet.h"
#include "../things/IWindow.h"
#include "../things/Info.h"
#include "../things/Rock.h"
#include "../things/Sun.h"
#include "../things/Blanket.h"
#include "../things/MineWindow.h"
#include "../sound/SoundSystem.h"
#include "../misc/Random.h"
#include "../misc/KeyReader.h"
#include "../entity/GameWorld.h"
#include "../engine/TimeKeeper.h"
#include "../engine/GLColor.h"

#include "Game.h"

/***************************************************/

const std::string story[2]
=
{
"The test was a complete bust. \n After waving goodbye to the scientists wishing you good luck, you set up the experimental hyperdrive "
"to target the Outpost system and pushed the button. The last thing you remember is a bright flash. \n "
"After waking up, you immediately checked the navigation computer. Contrary to what the scientists believed, the drive was far from "
"perfect - you were half a galaxy off target! The nearest human outpost is far beyond the reach of the conventional drive, and it "
"seems that you are completely on your own. \n With only a month's worth of fuel, food and energy you realise that your only hope "
"lies in the unfortunate hyperdrive. If you'd somehow manage to survive, and then recharge it with hyperium, maybe it could take you home this "
"time. \n \n Or maybe not.",
	
"You glance at the navigation computer and can't believe what your eyes tell you. You are in Deluron, one of the most crowded human "
"solar systems! You target the nearest planet and dock on the orbital station. \n \n You managed to SURVIVE."
};

/***************************************************/

const int NAMES_COUNT = 30;

const std::string planetNames[NAMES_COUNT] =
{ "Bonanza", "Wesside", "Sayonara", "Wisconsin", "Tarantoga", "Carter",
  "Whiplash", "Sierra", "Cryo", "Spokane", "Muchas Gracias", "San Salvador",
  "Einstein", "Dubya", "Midnight", "Stirrups", "Stallion", "Pothole",
  "Bigmouth", "Doohickey", "Spade", "Luther", "Meow", "Ramen", "Vader",
  "Oatmeal", "Flagpole", "Wrong Way", "Bless You", "Sodoma"};

/***************************************************/

void Game::play()
{
	ship = new Ship(200.0, 200.0);

	bool result;
	for(int diff = 1; diff <= 3; diff++)
	{
		ship->vx = 0;
		ship->vy = 0;
		ship->x = 200.0;
		ship->y = 200.0;
		
		result = playRound(diff);
		if (!result) break;
	}
	
	ship->vx = 0;
	ship->vy = 0;
	ship->x = 2000.0;
	ship->y = 1700.0;
		
	if (result)
		playRound(10);

	delete ship;
}

void Game::generateSystem(GameWorld *gw, int difficulty)
{
	//double aimSum = difficulty * 100.0;
	
	int numPlanets = difficulty + 2;
	int tankerPlanet, fodderPlanet;
	
	do
	{
		tankerPlanet = randomInt (numPlanets - 2, numPlanets - 1);
		fodderPlanet = randomInt (0, numPlanets / 2);
	} while (fodderPlanet == tankerPlanet);
	
	Planet *planets[10];
	
	double ecliptic = 1.0 + randomFloat(0.05, 0.05 + difficulty * 0.1);
	
	for (int i = 0; i < numPlanets; i++)
	{
		
		double radius = randomFloat(25.0, 55.0);
		//double mRadius = std::pow(2.5, i) * 400.0 * randomFloat(0.95, 1.05);
		double mRadius = ((i + 1) * 400.0) * randomFloat(0.95, 1.05);
		double speed = randomFloat(40.0, 60.0) * (1.0 + difficulty * 0.6) / mRadius;
		double angle = randomFloat(0.0, 6.283);
		
		int dominant, secondary, third;
		do
		{
			dominant = randomInt(0, 2);
			secondary = randomInt(0, 2);
			third = randomInt(0, 2);
		} while (dominant == secondary || dominant == third || secondary == third);
		double element[3];
		element[dominant] = randomFloat(0.75, 1.0);
		element[secondary] = randomFloat(0.0, 0.75);
		element[third] = randomFloat(0.0, 0.4);
		GLColor color(element[0], element[1], element[2]);
		
		int kind = (i == fodderPlanet) ? 1 : randomInt(0, 1);
		
		double foodValue = (i != fodderPlanet) ? randomFloat(0.0, 0.6 - difficulty * 0.1) : randomFloat(0.8 - difficulty * 0.1, 1.0 - difficulty * 0.05);
		double hyperValue = 0.0;
		double eventValue = randomFloat(0.5, 1.0) - difficulty * 0.15;
		if (eventValue < 0.0) eventValue = 0.0;
		
		double atmoSize, atmoType;
		if (randomFloat(0.0, 1.0) < 0.4)
			{atmoSize = randomFloat(5.0, 8.0); atmoType = randomFloat(0.0, 1.0);}
		if (i == tankerPlanet)
			{atmoSize = 23.0 - randomFloat(difficulty * 1.0, 0.5 + difficulty * 2.0); atmoType = randomFloat(0.8 - difficulty * 0.09, 1.0 - difficulty * 0.05);}
		if (i == fodderPlanet)
			{atmoSize = randomFloat(5.0, 10.0); atmoType = randomFloat(0.0, 0.4);}
		
		std::string name;
		bool OK = false;
		do
		{
			name = planetNames[randomInt(0, NAMES_COUNT - 1)];
			OK = true;
			for (int j = 0; j < i; j++)
				if (name == planets[j]->name) OK = false;
		}
		while (!OK);
		
		planets[i] = new Planet(NULL, radius, mRadius * ecliptic, mRadius, speed, angle, color, kind, foodValue, hyperValue, eventValue, atmoSize, atmoType, name);
		gw->add(planets[i], true);
		
		for(int moon = 0; moon < 2; moon++)
			if (randomFloat(0.0, 1.0) < difficulty * 0.25)
			{
				double nRadius = randomFloat(8.0, 13.0);
				double nMRadius = planets[i]->sRadius + nRadius + 7.0 + moon * 30.0 + randomFloat(3.0, 6.0);
				double nSpeed = 0.4 + randomFloat(difficulty * 0.2, difficulty * 0.4);
				double nAngle = randomFloat(0.0, 6.283);

				GLColor nColor(color.r + randomFloat(-0.2, 0.2), color.g + randomFloat(-0.2, 0.2), color.b + randomFloat(-0.2, 0.2));
				
				Planet *moon = new Planet(planets[i], nRadius, nMRadius, nMRadius, nSpeed, nAngle, nColor, 0, 0.0, 0.0, 0.0, 0.0, 0.0, "");
				gw->add(moon, true);
			}
		
	}
	
	int hyperPlanets = randomInt(2, numPlanets);
	double hyperEach = 120.0 / hyperPlanets;
	for (int i = 0; i < hyperPlanets; i++)
	{
		int nr;
		do
		{
			nr = randomInt(0, numPlanets - 1);
		} while (planets[nr]->hyperValue > 0.0);
		
		planets[nr]->hyperValue = hyperEach + randomFloat(3.0, 5.0);
	}
	
	double gRadius;
	switch(difficulty)
	{
		case 1:
			// some asteroids, static
			for (int i = 0; i < 40; i++)
				gw->add(new Rock(randomFloat(-20.0, 20.0), randomFloat(-20.0, 20.0), randomFloat(220.0, 1000.0), randomFloat(0.0, 0.1), randomFloat(0, 6.28)), true);
			break;
			
		case 2:
			// asteroid belt
			gRadius = planets[fodderPlanet]->mRadiusX + planets[fodderPlanet]->sRadius + 20.0;
			for (int i = 0; i < 200; i++)
				gw->add(new Rock(randomFloat(-5.0, 5.0), randomFloat(-5.0, 5.0), randomFloat(gRadius, gRadius + 100.0), randomFloat(0.1, 0.3), randomFloat(0, 6.28)), true);
			break;
			
		case 3:
			// both
			for (int i = 0; i < 200; i++)
				gw->add(new Rock(randomFloat(-100.0, 100.0), randomFloat(-100.0, 100.0), randomFloat(150.0, 1500.0), randomFloat(0.05, 0.15), randomFloat(0, 6.28)), true);
			
			gRadius = planets[fodderPlanet]->mRadiusX + planets[fodderPlanet]->sRadius + 20.0;
			for (int i = 0; i < 300; i++)
				gw->add(new Rock(randomFloat(-5.0, 5.0), randomFloat(-5.0, 5.0), randomFloat(gRadius, gRadius + 100.0), randomFloat(0.2, 0.4), randomFloat(0, 6.28)), true);
			break;
			
	}
	//delete [] planets;
}

bool Game::playRound(int difficulty)
{
	bool end = false;
	
	TimeKeeper tk(0.01, 0.025, 2, true);
	GameWorld gw;
	
	Info *dockInfo = new Info(Info::INFO_DOCK), *fuelInfo = new Info(Info::INFO_FUEL);
	Info *hyperInfo = new Info(Info::INFO_WARP);
	
	Sun *sun = new Sun();
	gw.add(sun, true);
	
	/*gw.add(new Planet(NULL, 25.0, 400.0, 400.0, 0.5, 0.0, GLColor(0.5, 0.7, 1.0), 1, 1.0, 0.0, 10.0, 0.0, "New Earth"), true);
	Planet *hell = new Planet(NULL, 50.0, 1100.0, 950.0, 0.2, 0.6, GLColor(1.0, 0.0, 0.0), 0, 0.0, 60.0, 6.0, 0.0, "Hell");
	gw.add(hell, true);
	gw.add(new Planet(hell, 15.0, 120.0, 120.0, 0.6, 1.3, GLColor(1.0, 0.5, 0.0), 0, 0.0, 0.0, 0.0, 0.0, "Era"), true);
	gw.add(new Planet(hell, 8.0, 90.0, 90.0, 1.5, 1.6, GLColor(1.0, 0.9, 0.4), 0, 0.0, 0.0, 0.0, 0.0, "Ara"), true); 
	gw.add(new Planet(NULL, 40.0, 2500.0, 1500.0, 0.05, 0.6, GLColor(0.4, 0.0, 1.0), 0, 0.3, 40.0, 20.0, 0.7, "Gazongas"), true);
	*/
	
	generateSystem(&gw, difficulty);
	
	panel = new Panel(ship, &gw);
	gw.add(panel, false);
	
	gw.add(ship, false);
	gw.add(new Starfield(ship, 400), true);

	// !!!!!!!!!!!
	// ship->hyper = 100.0; ship->power = 80.0;
	
	ship->goHyper = ship->hypered = false;
	ship->panel = panel;
	ship->renewPlanets();
	sun->targetShip();
	panel->figureOutMapScale();
	
	if (difficulty > 1)
		gw.add(new Blanket(GLColor(1.0, 1.0, 1.0)), true);
	//for (int i = 0; i < 200; i++)
	//	gw.add(new Rock(0.0, 0.0, randomFloat(650.0, 750.0), randomFloat(0.16, 0.2), randomFloat(0, 6.28)), true);

	MineWindow::miningDifficulty = difficulty;
	Planet::lastVisited = NULL;
	
	double storyDelay;
	if (difficulty == 1)
		storyDelay = 0.0;
	else
		storyDelay = 1.5;
	bool storyShown = false;
	
	tk.reset();	
	KeyReader::clearBuffer();
	while(!end)
	{
		double dt = tk.getDT();
		
		do
		{
			storyDelay -= dt;
			
			gw.processStep(dt);
			gw.processCollisions();
			
			if (ship->canDock && (!dockInfo->getParent()))
			{
				gw.add(dockInfo, false);
				soundSys->playSound(SND_INFO, 128, 1.0, 0.8);
			}
			else if ((!ship->canDock) && dockInfo->getParent())
				gw.remove(dockInfo);
			if (ship->canFuel && (!fuelInfo->getParent()))
			{
				gw.add(fuelInfo, false);
				soundSys->playSound(SND_INFO, 128, 1.0, 0.8);
			}
			else if ((!ship->canFuel) && fuelInfo->getParent())
				gw.remove(fuelInfo);
			if (ship->canHyper && (!hyperInfo->getParent()))
			{
				gw.add(hyperInfo, false);
				soundSys->playSound(SND_INFO, 128, 1.0, 0.8);
			}
			else if (!ship->canHyper && (hyperInfo->getParent()))
				gw.remove(hyperInfo);
			
			if (storyDelay < 0.0 && (!storyShown))
			{
				showStory(&gw, difficulty);
				storyShown = true;
			}
			
			if (key[KEY_ESC])
				ship->aliveFlag = false;
			
			KeyReader::update();
		}
		while(tk.getFramesToSkip() > 0);
		
		glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
		
		glLoadIdentity();
		glTranslatef(-ship->getX() + 320.0, -ship->getY() + 240.0, 0.0);
		
		gw.processRender();
		
		allegro_gl_flip();
		
		if (!gw.paused)
			end = (!ship->isAlive() || ship->hypered);
	}
	
	bool retValue = ship->isAlive();
	
	delete dockInfo; delete hyperInfo; delete fuelInfo; delete panel;
	return retValue;
}

void Game::showStory(GameWorld *gw, int difficulty)
{
	IWindow *win = NULL;
	
	if (difficulty == 1)
		win = new IWindow(story[0], 450);
	if (difficulty == 10)
	{
		win = new IWindow(story[1], 450);
		((Ship*)(*(gw->getEntitiesOfType(ETYPE_SHIP).begin())))->aliveFlag = false; // the most deranged line in the whole code ;).
	}
	
	if (win)
	{
		win->y = 240.0 - 0.5 * win->height;
		gw->add(win, true);
		gw->paused = true;
	}
}
