#include <allegro.h>
#include "animate.h"

/* used to create an empty animation */
void initialize_animation(ANIMATION * ap)
{
    int i;

    for(i = 0; i < ANIMATION_MAX_FRAMES; i++)
    {
        ap->image[i] = NULL;
    }
    ap->w = 0; ap->h = 0; ap->f = 0; ap->d = 0; ap->loaded = 0;
}

/* free an animation from memory */
void destroy_animation(ANIMATION * ap)
{
    int i;

    /* if animation is loaded, destroy it */
    if(ap->loaded)
    {
        for(i = 0; i < ap->f; i++)
        {
            if(ap->image[i] != NULL)
            {
                destroy_bitmap(ap->image[i]);
            }
        }
        ap->w = 0; ap->h = 0; ap->f = 0; ap->d = 0; ap->loaded = 0;
    }
}

/* load animation from a file pointer */
int load_animation_fp(ANIMATION * ap, PACKFILE * fp)
{
    char header[4];
    int i, j, k;

    /* check format */
    pack_fread(header, 4, fp);
    if(header[0] != 'A' || header[1] != 'N' || header[2] != 'I' || header[3] != 21)
    {
        return 0;
    }

    /* destroy old animation if necessary */
    if(ap->loaded)
    {
        destroy_animation(ap);
    }

    /* load header */
    ap->w = pack_igetl(fp);
    ap->h = pack_igetl(fp);
    ap->f = pack_igetl(fp);
    ap->d = pack_igetl(fp);

    /* load animation data */
    for(i = 0; i < ap->f; i++)
    {
        ap->image[i] = create_bitmap(ap->w, ap->h);
        for(j = 0; j < ap->h; j++)
        {
            for(k = 0; k < ap->w; k++)
            {
                ap->image[i]->line[j][k] = pack_getc(fp);
            }
        }
    }
    ap->loaded = 1;
    return 1;
}

/* load an animation from a filename */
int load_animation(ANIMATION * ap, char * fn)
{
    PACKFILE * fp;

    /* open the file */
    fp = pack_fopen(fn, "r");
    if(fp == NULL) return 0;

    /* read the animation data */
    if(load_animation_fp(ap, fp))
    {
        pack_fclose(fp);
        return 1;
    }
    pack_fclose(fp);

    return 0;
}

/* save animation to a file pointer */
int save_animation_fp(ANIMATION * ap, PACKFILE * fp)
{
    char header[4] = {'A', 'N', 'I', 21};
    int i, j, k;

    /* write format header */
    pack_fwrite(header, 4, fp);

    /* save header */
    pack_iputl(ap->w, fp);
    pack_iputl(ap->h, fp);
    pack_iputl(ap->f, fp);
    pack_iputl(ap->d, fp);

    /* save animation data */
    for(i = 0; i < ap->f; i++)
    {
        for(j = 0; j < ap->h; j++)
        {
            for(k = 0; k < ap->w; k++)
            {
                pack_putc(ap->image[i]->line[j][k], fp);
            }
        }
    }
    return 1;
}

/* save an animation */
int save_animation(ANIMATION * ap, char * fn)
{
    PACKFILE * fp;

    fp = pack_fopen(fn, "w");
    if(fp == NULL) return 0;

    if(save_animation_fp(ap, fp))
    {
        pack_fclose(fp);
        return 1;
    }
    pack_fclose(fp);

    return 0;
}

/* create an empty animation of the specified width and height */
int create_animation(ANIMATION * ap, int w, int h)
{
    if(ap->loaded) destroy_animation(ap);

    ap->w = w;
    ap->h = h;
    ap->f = 0;
    ap->d = 0;
    ap->loaded = 1;

    return 1;
}

/* add frame to end of animation */
int animation_add_frame(ANIMATION * ap, BITMAP * bp)
{
    if(ap->f < ANIMATION_MAX_FRAMES && ap->loaded)
    {
        ap->image[ap->f] = create_bitmap(ap->w, ap->h);
        if(ap->image[ap->f] == NULL) return 0;
        clear(ap->image[ap->f]);
        blit(bp, ap->image[ap->f], 0, 0, 0, 0, ap->w, ap->h);
        ap->f++;

        return 1;
    }
    return 0;
}

/* insert a frame of animation */
int animation_insert_frame(ANIMATION * ap, int f, BITMAP * bp)
{
    int i;

    /* make sure we are inserting image to appropriate place */
    if(f < ap->f && ap->f < ANIMATION_MAX_FRAMES && ap->loaded)
    {
        ap->image[ap->f] = create_bitmap(ap->w, ap->h);

        /* make room for the inserted frame */
        for(i = ap->f; i > f; i--)
        {
            blit(ap->image[i - 1], ap->image[i], 0, 0, 0, 0, ap->w, ap->h);
        }

        /* insert the new image */
        clear(ap->image[f]);
        blit(bp, ap->image[f], 0, 0, 0, 0, ap->w, ap->h);
        ap->f++;

        return 1;
    }
    return 0;
}

int animation_replace_frame(ANIMATION * ap, int f, BITMAP * bp)
{
    if(f < ap->f && ap->loaded)
    {
        clear(ap->image[f]);
        blit(bp, ap->image[f], 0, 0, 0, 0, ap->w, ap->h);
        return 1;
    }
    return 0;
}

int animation_delete_frame(ANIMATION * ap, int f)
{
    int i;

    if(f < ap->f && ap->loaded)
    {
        /* move other frames back one position */
        for(i = f; i < ap->f - 1; i++)
        {
            blit(ap->image[i + 1], ap->image[i], 0, 0, 0, 0, ap->w, ap->h);
        }
        destroy_bitmap(ap->image[ap->f - 1]);

        /* adjust frame counter */
        ap->f--;
        if(ap->f == 0)
        {
            ap->loaded = 0;
        }

        return 1;
    }
    return 0;
}

int animation_copy_frame(ANIMATION * ap, int f, BITMAP * bp)
{
    if(f < ap->f && ap->loaded)
    {
        /* create and copy */
        bp = create_bitmap(ap->w, ap->h);
        blit(ap->image[f], bp, 0, 0, 0, 0, ap->w, ap->h);

        return 1;
    }
    return 0;
}

BITMAP * get_animation(ANIMATION * ap, int f)
{
    int i = !ap->d ? f % ap->f : (f / ap->d) % ap->f;
    return ap->image[i];
}

BITMAP * get_animation_frame(ANIMATION * ap, int f)
{
    return ap->image[f];
}

/* draw the specified frame */
int draw_animation_frame(ANIMATION * ap, int f, BITMAP * bp, int x, int y)
{
    if(f < ap->f && ap->loaded)
    {
        draw_sprite(bp, ap->image[f], x, y);

        return 1;
    }
    return 0;
}

/* draw the specified frame (accounts for the delay variable) */
int blit_animation(ANIMATION * ap, unsigned long f, BITMAP * bp, int x, int y)
{
    int i;

    if(ap->f > 0)
    {
        i = !ap->d ? f % ap->f : (f / ap->d) % ap->f;
        if(i < ap->f && ap->loaded)
        {
            blit(ap->image[i], bp, 0, 0, x, y, bp->w, bp->h);
        
            return 1;
        }
    }
    return 0;
}

/* draw the specified frame (accounts for the delay variable) */
int draw_animation(ANIMATION * ap, unsigned long f, BITMAP * bp, int x, int y)
{
    int i;

    if(ap->f > 0)
    {
        i = !ap->d ? f % ap->f : (f / ap->d) % ap->f;
        if(i < ap->f && ap->loaded)
        {
            draw_sprite(bp, ap->image[i], x, y);
        
            return 1;
        }
    }
    return 0;
}

int draw_animation_trans(ANIMATION * ap, unsigned long f, BITMAP * bp, int x, int y)
{
    int i;

    if(ap->f > 0)
    {
        i = !ap->d ? f % ap->f : (f / ap->d) % ap->f;
        if(i < ap->f && ap->loaded)
        {
            draw_trans_sprite(bp, ap->image[i], x, y);
        
            return 1;
        }
    }
    return 0;
}

int draw_animation_frame_solid(ANIMATION * ap, unsigned long f, BITMAP * bp, int x, int y, int color)
{
    if(f < ap->f && ap->loaded)
    {
        draw_character(bp, ap->image[f], x, y, color);

        return 1;
    }
    return 0;
}

int draw_animation_solid(ANIMATION * ap, unsigned long f, BITMAP * bp, int x, int y, int color)
{
    int i = !ap->d ? f % ap->f : (f / ap->d) % ap->f;
    if(i < ap->f && ap->loaded)
    {
        draw_character(bp, ap->image[i], x, y, color);

        return 1;
    }
    return 0;
}
