#include <stdio.h>
#include "reverse.h"
#include "core.h"
#include "random.h"

/* count number of changes in the change map */
int reverse_ai_scan_changes(REVERSE_BOARD * bp)
{
    int i, j;
    int count = 0;

    for(i = 0; i < REVERSE_BOARD_RY; i++)
    {
        for(j = 0; j < REVERSE_BOARD_RX; j++)
        {
            if(bp->change[i][j])
            {
                count++;
            }
        }
    }
    return count;
}

/* determine if specified move is along the diagonals */
int reverse_ai_diagonal(int x, int y)
{
    /* upper left to lower right */
    if(x == y)
    {
        return 1;
    }

    /* upper right to lower left */
    else if((REVERSE_BOARD_RX - x - 1) == y)
    {
        return 1;
    }

    /* neither */
    return 0;
}

/* choose a move from move list for AI (beginner) */
int reverse_ai_choose(REVERSE_AI_MOVE_LIST * lp)
{
    int chosen;

    /* just choose one of the moves in the list */
    chosen = ncd_random() % lp->move_count;
    return chosen;
}

/* choose a move from move list for AI (intermediate) */
int reverse_ai_choose_good(REVERSE_AI_MOVE_LIST * lp)
{
    int i, j;
    REVERSE_AI_MOVE move;

    /* sort the list */
    for(i = 0; i < lp->move_count; i++)
    {
        for(j = 1; j < lp->move_count; j++)
        {
            if(lp->move[j].amount > lp->move[j - 1].amount)
            {
                move.amount = lp->move[j - 1].amount;
                move.x = lp->move[j - 1].x;
                move.y = lp->move[j - 1].y;
                lp->move[j - 1].amount = lp->move[j].amount;
                lp->move[j - 1].x = lp->move[j].x;
                lp->move[j - 1].y = lp->move[j].y;
                lp->move[j].amount = move.amount;
                lp->move[j].x = move.x;
                lp->move[j].y = move.y;
            }
        }
    }

    /* no moves? */
    if(lp->move_count <= 1)
    {
        return 0;
    }

    /* if there are moves */
    else
    {
        /* see if AI can choose from more than 1 position */
        if(lp->move[0].amount == lp->move[1].amount)
        {
            for(i = 1; i < lp->move_count && lp->move[i].amount == lp->move[0].amount; i++);
            return ncd_random() % i;
        }

        /* if it can't then choose the move at the top of the list */
        return ncd_random() % 2;
    }
}

/* choose a move from move list for AI (advanced) */
int reverse_ai_choose_better(REVERSE_AI_MOVE_LIST * lp)
{
    int i, j;
    REVERSE_AI_MOVE move;

    /* make corners appear as best moves to algorithm below */
    for(i = 0; i < lp->move_count; i++)
    {
        if((lp->move[i].x == 0) && (lp->move[i].y) == 0)
        {
            lp->move[i].amount += 1000;
        }
        else if((lp->move[i].x == (REVERSE_BOARD_RX - 1)) && (lp->move[i].y == 0))
        {
            lp->move[i].amount += 1000;
        }
        else if((lp->move[i].x == 0) && (lp->move[i].y == (REVERSE_BOARD_RY - 1)))
        {
            lp->move[i].amount += 1000;
        }
        else if((lp->move[i].x == (REVERSE_BOARD_RX - 1)) && (lp->move[i].y == (REVERSE_BOARD_RY - 1)))
        {
            lp->move[i].amount += 1000;
        }
    }

    /* sort the list (bubble sort) */
    for(i = 0; i < lp->move_count; i++)
    {
        for(j = 1; j < lp->move_count; j++)
        {
            if(lp->move[j].amount > lp->move[j - 1].amount)
            {
                move.amount = lp->move[j - 1].amount;
                move.x = lp->move[j - 1].x;
                move.y = lp->move[j - 1].y;
                lp->move[j - 1].amount = lp->move[j].amount;
                lp->move[j - 1].x = lp->move[j].x;
                lp->move[j - 1].y = lp->move[j].y;
                lp->move[j].amount = move.amount;
                lp->move[j].x = move.x;
                lp->move[j].y = move.y;
            }
        }
    }

    /* no moves? */
    if(lp->move_count <= 1)
    {
        return 0;
    }

    /* if there are moves */
    else
    {
        /* see if AI can choose from more than 1 position */
        if(lp->move[0].amount == lp->move[1].amount)
        {
            for(i = 1; i < lp->move_count && lp->move[i].amount == lp->move[0].amount; i++);
            return ncd_random() % i;
        }

        /* if it can't then choose the move at the top of the list */
        return 0;
    }
}

/* choose a move from move list for AI (expert) */
int reverse_ai_choose_best(REVERSE_AI_MOVE_LIST * lp)
{
    int i, j;
    REVERSE_AI_MOVE move;

    /* sort the list */
    for(i = 0; i < lp->move_count; i++)
    {
        for(j = 1; j < lp->move_count; j++)
        {
            if(lp->move[j].amount > lp->move[j - 1].amount)
            {
                move.amount = lp->move[j - 1].amount;
                move.x = lp->move[j - 1].x;
                move.y = lp->move[j - 1].y;
                lp->move[j - 1].amount = lp->move[j].amount;
                lp->move[j - 1].x = lp->move[j].x;
                lp->move[j - 1].y = lp->move[j].y;
                lp->move[j].amount = move.amount;
                lp->move[j].x = move.x;
                lp->move[j].y = move.y;
            }
        }
    }

    /* no moves? */
    if(lp->move_count <= 1)
    {
        return 0;
    }

    /* if there are moves */
    else
    {
        /* see if AI can choose from more than 1 position */
        if(lp->move[0].amount == lp->move[1].amount)
        {
            for(i = 1; i < lp->move_count && lp->move[i].amount == lp->move[0].amount; i++);
            return ncd_random() % i;
        }

        /* if it can't then choose the move at the top of the list */
        return 0;
    }
}

/* find opponent potential */
int reverse_ai_opponent_potential(REVERSE_BOARD * bp, int turn, int x, int y)
{
    REVERSE_BOARD        board;
    REVERSE_AI_MOVE_LIST move_list;
    REVERSE_AI_MOVE      move;
    int i, j;
    int hispotential = 0;

    /* make board to work with */
    for(i = 0; i < REVERSE_BOARD_RY; i++)
    {
        for(j = 0; j < REVERSE_BOARD_RX; j++)
        {
            board.data[i][j] = bp->data[i][j];
            board.change[i][j] = 0;
        }
    }

    /* build move list */
    move_list.move_count = 0;
    for(i = 0; i < REVERSE_BOARD_RY; i++)
    {
        for(j = 0; j < REVERSE_BOARD_RX; j++)
        {
            if(reverse_place(&board, (turn == 0) ? 1 : 0, j, i))
            {
                move_list.move[move_list.move_count].x = j;
                move_list.move[move_list.move_count].y = i;
                move_list.move[move_list.move_count].amount = reverse_ai_scan_changes(&board);
                move_list.move_count++;
                reverse_clear_changes(&board);
            }
        }
    }

    /* evaluate potential of each move */
    for(i = 0; i < move_list.move_count; i++)
    {
        /* if opponent gets a corner it's mucho potential points for him */
        if((move_list.move[i].x == 0) && (move_list.move[i].y) == 0)
        {
            move_list.move[i].amount += 1500;
        }
        else if((move_list.move[i].x == (REVERSE_BOARD_RX - 1)) && (move_list.move[i].y == 0))
        {
            move_list.move[i].amount += 1500;
        }
        else if((move_list.move[i].x == 0) && (move_list.move[i].y == (REVERSE_BOARD_RY - 1)))
        {
            move_list.move[i].amount += 1500;
        }
        else if((move_list.move[i].x == (REVERSE_BOARD_RX - 1)) && (move_list.move[i].y == (REVERSE_BOARD_RY - 1)))
        {
            move_list.move[i].amount += 1500;
        }

        /* if opponent picks these spots, that's good for you */
        else if(move_list.move[i].x == 1 && move_list.move[i].y == 0)
        {
            move_list.move[i].amount -= 750;
        }
        else if(move_list.move[i].x == 0 && move_list.move[i].y == 1)
        {
            move_list.move[i].amount -= 750;
        }
        else if(move_list.move[i].x == 1 && move_list.move[i].y == 1)
        {
            move_list.move[i].amount -= 750;
        }
        else if(move_list.move[i].x == REVERSE_BOARD_RX - 2 && move_list.move[i].y == 0)
        {
            move_list.move[i].amount -= 750;
        }
        else if(move_list.move[i].x == REVERSE_BOARD_RX - 1 && move_list.move[i].y == 1)
        {
            move_list.move[i].amount -= 750;
        }
        else if(move_list.move[i].x == REVERSE_BOARD_RX - 2 && move_list.move[i].y == 1)
        {
            move_list.move[i].amount -= 750;
        }
        else if(move_list.move[i].x == 1 && move_list.move[i].y == REVERSE_BOARD_RY - 1)
        {
            move_list.move[i].amount -= 750;
        }
        else if(move_list.move[i].x == 0 && move_list.move[i].y == REVERSE_BOARD_RY - 2)
        {
            move_list.move[i].amount -= 750;
        }
        else if(move_list.move[i].x == 1 && move_list.move[i].y == REVERSE_BOARD_RY - 2)
        {
            move_list.move[i].amount -= 750;
        }
        else if(move_list.move[i].x == REVERSE_BOARD_RX - 2 && move_list.move[i].y == REVERSE_BOARD_RY - 1)
        {
            move_list.move[i].amount -= 750;
        }
        else if(move_list.move[i].x == REVERSE_BOARD_RX - 1 && move_list.move[i].y == REVERSE_BOARD_RY - 2)
        {
            move_list.move[i].amount -= 750;
        }
        else if(move_list.move[i].x == REVERSE_BOARD_RX - 2 && move_list.move[i].y == REVERSE_BOARD_RY - 2)
        {
            move_list.move[i].amount -= 750;
        }

        /* if opponent can get edge he gains potential */
        else if(move_list.move[i].x == 0)
        {
            move_list.move[i].amount += 750;
        }
        else if(move_list.move[i].x == REVERSE_BOARD_RX - 1)
        {
            move_list.move[i].amount += 750;
        }
        else if(move_list.move[i].y == 0)
        {
            move_list.move[i].amount += 750;
        }
        else if(move_list.move[i].y == REVERSE_BOARD_RY - 1)
        {
            move_list.move[i].amount += 750;
        }

        /* if opponent can get diagonal he gains potential */
        else if(reverse_ai_diagonal(move_list.move[i].x, move_list.move[i].y))
        {
            move_list.move[i].amount += 750;
        }

        /* if opponent can get inner edge you gain potential */
        else if(move_list.move[i].x == 1)
        {
            move_list.move[i].amount -= 750;
        }
        else if(move_list.move[i].x == REVERSE_BOARD_RX - 2)
        {
            move_list.move[i].amount -= 750;
        }
        else if(move_list.move[i].y == 1)
        {
            move_list.move[i].amount -= 750;
        }
        else if(move_list.move[i].y == REVERSE_BOARD_RY - 2)
        {
            move_list.move[i].amount -= 750;
        }
    }

    /* sort move list to put highest potential on top */
    for(i = 0; i < move_list.move_count; i++)
    {
        for(j = 1; j < move_list.move_count; j++)
        {
            if(move_list.move[j].amount > move_list.move[j - 1].amount)
            {
                move.amount = move_list.move[j - 1].amount;
                move.x = move_list.move[j - 1].x;
                move.y = move_list.move[j - 1].y;
                move_list.move[j - 1].amount = move_list.move[j].amount;
                move_list.move[j - 1].x = move_list.move[j].x;
                move_list.move[j - 1].y = move_list.move[j].y;
                move_list.move[j].amount = move.amount;
                move_list.move[j].x = move.x;
                move_list.move[j].y = move.y;
            }
        }
    }

    /* retrieve highest potential */
    if(move_list.move_count > 0)
    {
        hispotential = move_list.move[0].amount;
    }

    return hispotential;
}

/* find a move's opposition potential */
int reverse_ai_potential(REVERSE_BOARD * bp, int turn, int x, int y)
{
    REVERSE_BOARD        board;
//    REVERSE_AI_MOVE_LIST move_list;
//    REVERSE_AI_MOVE      move;
    int i, j;
    int mypotential = 0;
    int hispotential = 0;

    /* make board to work with */
    for(i = 0; i < REVERSE_BOARD_RY; i++)
    {
        for(j = 0; j < REVERSE_BOARD_RX; j++)
        {
            board.data[i][j] = bp->data[i][j];
            board.change[i][j] = 0;
        }
    }

    /* find my potential */
    reverse_place(&board, turn, x, y);
    mypotential = reverse_ai_scan_changes(&board);
    if((x == 0) && (y == 0))
    {
        mypotential += 1000;
    }
    else if((x == (REVERSE_BOARD_RX - 1)) && (y == 0))
    {
        mypotential += 1000;
    }
    else if((x == 0) && (y == (REVERSE_BOARD_RY - 1)))
    {
        mypotential += 1000;
    }
    else if((x == (REVERSE_BOARD_RX - 1)) && (y == (REVERSE_BOARD_RY - 1)))
    {
        mypotential += 1000;
    }
    else if((x == 1) && (y > 0 && y < REVERSE_BOARD_RY - 1))
    {
        mypotential -= 750;
    }
    else if((x == REVERSE_BOARD_RX - 2) && (y > 0 && y < REVERSE_BOARD_RY - 1))
    {
        mypotential -= 750;
    }
    else if((y == 1) && (x > 0 && x < REVERSE_BOARD_RX - 1))
    {
        mypotential -= 750;
    }
    else if((y == REVERSE_BOARD_RY - 2) && (x > 0 && x < REVERSE_BOARD_RX - 1))
    {
        mypotential -= 750;
    }
    else if(x == 1 && y == 0)
    {
        mypotential -= 850;
    }
    else if(x == 0 && y == 1)
    {
        mypotential -= 850;
    }
    else if(x == 1 && y == 1)
    {
        mypotential -= 850;
    }
    else if(x == REVERSE_BOARD_RX - 2 && y == 0)
    {
        mypotential -= 850;
    }
    else if(x == REVERSE_BOARD_RX - 1 && y == 1)
    {
        mypotential -= 850;
    }
    else if(x == REVERSE_BOARD_RX - 2 && y == 1)
    {
        mypotential -= 850;
    }
    else if(x == 1 && y == REVERSE_BOARD_RY - 1)
    {
        mypotential -= 850;
    }
    else if(x == 0 && y == REVERSE_BOARD_RY - 2)
    {
        mypotential -= 850;
    }
    else if(x == 1 && y == REVERSE_BOARD_RY - 2)
    {
        mypotential -= 850;
    }
    else if(x == REVERSE_BOARD_RX - 2 && y == REVERSE_BOARD_RY - 1)
    {
        mypotential -= 850;
    }
    else if(x == REVERSE_BOARD_RX - 1 && y == REVERSE_BOARD_RY - 2)
    {
        mypotential -= 850;
    }
    else if(x == REVERSE_BOARD_RX - 2 && y == REVERSE_BOARD_RY - 2)
    {
        mypotential -= 850;
    }
    else if(reverse_ai_diagonal(x, y))
    {
        mypotential += 500;
    }
    else if(x == 0)
    {
        mypotential += 600;
    }
    else if(x == REVERSE_BOARD_RX - 1)
    {
        mypotential += 600;
    }
    else if(y == 0)
    {
        mypotential += 600;
    }
    else if(y == REVERSE_BOARD_RY - 1)
    {
        mypotential += 600;
    }
//    reverse_clear_changes(&board);
    reverse_handle_changes(&board, turn);

    /* find opponent's potential */
    hispotential = reverse_ai_opponent_potential(&board, turn, x, y);

    /* return the potential */
    return mypotential - hispotential;
}

/* the main AI function, picks best spot according to AI type */
int reverse_ai_place(REVERSE_BOARD * bp, int turn, int type)
{
    int i, j;
    REVERSE_AI_MOVE_LIST move_list;
    int chosen;

    /* handle beginner level AI */
    if(type == REVERSE_AI_TYPE_BEGINNER)
    {
        /* build list of moves for AI to choose from */
        move_list.move_count = 0;
        for(i = 0; i < REVERSE_BOARD_RY; i++)
        {
            for(j = 0; j < REVERSE_BOARD_RX; j++)
            {
                if(reverse_place(bp, turn, j, i))
                {
                    move_list.move[move_list.move_count].x = j;
                    move_list.move[move_list.move_count].y = i;
                    move_list.move_count++;
                }
            }
        }
        reverse_clear_changes(bp);

        /* if there's at least one place, choose place from list */
        if(move_list.move_count > 0)
        {
            chosen = reverse_ai_choose(&move_list);
            reverse_place(bp, turn, move_list.move[chosen].x, move_list.move[chosen].y);
            return move_list.move[chosen].y * REVERSE_BOARD_RX + move_list.move[chosen].x;
        }

        /* otherwise make it known that there's none */
        return -1;
    }

    /* intermediate level AI */
    else if(type == REVERSE_AI_TYPE_INTERMEDIATE)
    {
        /* build list of moves for AI to choose from */
        move_list.move_count = 0;
        for(i = 0; i < REVERSE_BOARD_RY; i++)
        {
            for(j = 0; j < REVERSE_BOARD_RX; j++)
            {
                if(reverse_place(bp, turn, j, i))
                {
                    move_list.move[move_list.move_count].x = j;
                    move_list.move[move_list.move_count].y = i;
                    move_list.move[move_list.move_count].amount = reverse_ai_scan_changes(bp);
                    move_list.move_count++;
                    reverse_clear_changes(bp);
                }
            }
        }

        /* if there's at least one place, choose place from list */
        if(move_list.move_count > 0)
        {
            chosen = reverse_ai_choose_good(&move_list);
            reverse_place(bp, turn, move_list.move[chosen].x, move_list.move[chosen].y);
            return move_list.move[chosen].y * REVERSE_BOARD_RX + move_list.move[chosen].x;
        }

        /* otherwise make it known that there's none */
        return -1;
    }

    /* advanced AI */
    else if(type == REVERSE_AI_TYPE_ADVANCED)
    {
        /* build list of moves for AI to choose from */
        move_list.move_count = 0;
        for(i = 0; i < REVERSE_BOARD_RY; i++)
        {
            for(j = 0; j < REVERSE_BOARD_RX; j++)
            {
                if(reverse_place(bp, turn, j, i))
                {
                    move_list.move[move_list.move_count].x = j;
                    move_list.move[move_list.move_count].y = i;
                    move_list.move[move_list.move_count].amount = reverse_ai_scan_changes(bp);
                    move_list.move_count++;
                    reverse_clear_changes(bp);
                }
            }
        }

        /* if there's at least one place, choose place from list */
        if(move_list.move_count > 0)
        {
            chosen = reverse_ai_choose_better(&move_list);
            reverse_place(bp, turn, move_list.move[chosen].x, move_list.move[chosen].y);
            return move_list.move[chosen].y * REVERSE_BOARD_RX + move_list.move[chosen].x;
        }

        /* otherwise make it known that there's none */
        return -1;
    }

    /* expert AI */
    else if(type == REVERSE_AI_TYPE_EXPERT)
    {
        /* build list of moves for AI to choose from */
        move_list.move_count = 0;
        for(i = 0; i < REVERSE_BOARD_RY; i++)
        {
            for(j = 0; j < REVERSE_BOARD_RX; j++)
            {
                if(reverse_place(bp, turn, j, i))
                {
                    move_list.move[move_list.move_count].x = j;
                    move_list.move[move_list.move_count].y = i;
                    move_list.move[move_list.move_count].amount = reverse_ai_potential(bp, turn, j, i);
                    move_list.move_count++;
                    reverse_clear_changes(bp);
                }
            }
        }

        /* if there's at least one place, choose place from list */
        if(move_list.move_count > 0)
        {
            chosen = reverse_ai_choose_best(&move_list);
            reverse_place(bp, turn, move_list.move[chosen].x, move_list.move[chosen].y);
            return move_list.move[chosen].y * REVERSE_BOARD_RX + move_list.move[chosen].x;
        }

        /* otherwise make it known that there's none */
        return -1;
    }
    return -1;
}
