/* Functions to load WAV files */


#include <allegro.h>
#include "t3ssal.h"

/* structure to store WAV header */
typedef struct
{
	char  rID[4];
    unsigned long rLen;
	char  wID[4];
	char  fID[4];
    unsigned long fLen;
    unsigned short wFormatTag;
    unsigned short nChannels;
    unsigned long nSamplesPerSec;
    unsigned long nAvgBytesPerSec;
    unsigned short nBlockAlign;
    unsigned short nFormatSpecific;
} WAVHEADER;


/* loads a wav file from a file pointer */
MSAMPLE * tc_load_wav_fp(PACKFILE * fp)
{
    MSAMPLE * sp;
    int TempLength;
    int slength;

    WAVHEADER wh;
    char dID[4];

    /* read the wav header */

    /* identifier */
    pack_fread(wh.rID, 4, fp);

    /* (length of file) - 8 */
    wh.rLen = pack_igetl(fp);

    /* wave identifier */
    pack_fread(wh.wID, 4, fp);

    /* format chunk identifier */
    pack_fread(wh.fID, 4, fp);

    /* length of format chunk */
    wh.fLen = pack_igetl(fp);
    TempLength = wh.fLen;

    /* variable (always 0x01) */
    wh.wFormatTag = pack_igetw(fp);
    TempLength -= 2;

    /* number of channels */
    wh.nChannels = pack_igetw(fp);
    TempLength -= 2;

    /* read sample rate */
    wh.nSamplesPerSec = pack_igetl(fp);
    TempLength -= 4;

    /* read bytes per second */
    wh.nAvgBytesPerSec = pack_igetl(fp);
    TempLength -= 4;

    /* read bytes per sample (should be 1) */
    wh.nBlockAlign = pack_igetw(fp);
    TempLength -= 2;

    /* read bits per sample (should be 8) */
    wh.nFormatSpecific = pack_igetw(fp);
    TempLength -= 2;
//        pack_fread(&wh.nFormatSpecific, 2, fp);

    /* skip rest of format chunk */
    if(TempLength > 0)
    {
        pack_fread(NULL, TempLength, fp);
    }

    /* check it */
    if(memcmp(wh.rID, "RIFF" ,4) || memcmp(wh.wID, "WAVE", 4) || memcmp(wh.fID, "fmt ", 4))
    {
		return NULL;
    }

    pack_fread(dID, 4, fp);

    /* make sure we're at a wave data chunk before reading the rest */
    if(memcmp(dID, "data", 4))
    {
		return NULL;
    }

    /* only support mono WAVs */
    if(wh.nChannels > 1)
    {
		return NULL;
    }

    /* get the length of the sample */
    slength = pack_igetl(fp);
    if(wh.nBlockAlign == 2)
    {
        slength >>= 1;
    }

    /* allocate memory for the sample structure */
    sp = create_sample(8, 0, wh.nSamplesPerSec, slength);
    pack_fread(sp->data, slength, fp);
    if(!sp)
    {
        return NULL;
    }
    return sp;
}


/* loads wave from the specified file and offset */
MSAMPLE * tc_load_wav(char * fn)
{
    PACKFILE * fp;
    MSAMPLE * sp;

    /* open the file */
    fp = pack_fopen(fn, "r");
    if(fp == NULL)
    {
        return NULL;
    }

    /* load the sample */
    sp = tc_load_wav_fp(fp);

    return sp;
}


/* saves a wave file to file pointer */
int tc_save_wav_fp(PACKFILE * fp, MSAMPLE * sp)
{
    /* create the header */
    char RiffHeader[4] = {'R', 'I', 'F', 'F'};
    unsigned long FileLength = 36 + sp->len;
    char WaveHeader[4] = {'W', 'A', 'V', 'E'};
    char FormatHeader[4] = {'f', 'm', 't', ' '};
    char DataHeader[4] = {'d', 'a', 't', 'a'};

    /* write header to file */
    pack_fwrite(RiffHeader, 4, fp);
    pack_iputl(FileLength, fp);
    pack_fwrite(WaveHeader, 4, fp);
    pack_fwrite(FormatHeader, 4, fp);
    pack_iputl(10, fp);
    pack_iputw(1, fp);
    pack_iputw(1, fp);
    pack_iputl(sp->freq, fp);
    pack_iputl(sp->freq, fp);
    pack_iputw(1, fp);
    pack_iputw(8, fp);
    pack_fwrite(DataHeader, 4, fp);
    pack_iputl(sp->len, fp);

    /* write sample data to file */
    pack_fwrite(sp->data, sp->len, fp);

    return 1;
}


/* saves wave to the specified file */
int tc_save_wav(char * fn, MSAMPLE * sp)
{
    PACKFILE * fp;

    /* open file */
    fp = pack_fopen(fn, "w");
    if(fp == NULL)
    {
        return 0;
    }

    /* save WAV to the file */
    if(!tc_save_wav_fp(fp, sp))
    {
        return 0;
    }

    /* close the file */
    pack_fclose(fp);

    return 1;
}
