#include <cstdlib>
#include <cmath>
#include "game.h"
#include "input.h"
#include "timer.h"
#include "ddisplay.h"
#include "graphics.h"
#include "text.h"
#include "sound.h"
#include "lup.h"
#include "gover.h"
#include "intro.h"

SPRITE enemy_sprites[2];

void RunGame(void)
{
  STREAM dungeon_music;
  GAME game;
  GOVER game_over;
  INTRO intro;
  int min_enemies = 5;
  int max_enemies = 7;
  int enemy_speed_min = 20;
  int enemy_speed_max = 30;
  int maps = 2;

  intro.DoScene();

  dungeon_music.Load("data/dungeon.mp3");
  dungeon_music.Play();

  game.ResetGame();

  game.SetVariables(min_enemies, max_enemies, enemy_speed_min, enemy_speed_max, maps);
  
  while(game.DoScene())
    {
      min_enemies = static_cast<int>(min_enemies * 1.5);
      max_enemies = static_cast<int>(max_enemies * 1.5);
      enemy_speed_min+=1;
      if(enemy_speed_min > 80)
	{
	  enemy_speed_min = 80;
	}
      enemy_speed_max+=2;
      if(enemy_speed_max > 80)
	{
	  enemy_speed_max = 80;
	}
      maps++;
      game.SetVariables(min_enemies, max_enemies, enemy_speed_min, enemy_speed_max, maps);
    }

  game_over.DoScene();

  dungeon_music.Destroy();
}

void GAME::SetVariables(int min_enemies, int max_enemies, int enemy_speed_min, int enemy_speed_max, int maps)
{
  
  if(max_enemies > MAX_ENEMIES)
    max_enemies = MAX_ENEMIES;

  if(min_enemies > MAX_ENEMIES)
    min_enemies = MAX_ENEMIES;

  min_enem = min_enemies;
  max_enem = max_enemies;
  enem_speed_min = enemy_speed_min;
  enem_speed_max = enemy_speed_max;

  if(num_maps > MAX_MAPS)
    num_maps = MAX_MAPS;

  num_maps = maps;
  
}

bool GAME::OnWinningQuad(void)
{
  if(dungeon_complete == true && current_room == num_maps - 1)
    {
      if(player.GetTileX() == 9 || player.GetTileX() == 10)
	{
	  if(player.GetTileY() == 7 || player.GetTileY() == 8)
	    {
	      beaten = true;
	      return true;
	    }
	}
    }
  return false;
}

void GAME::ResetGame(void)
{
  player.ResetPlayer();
} 

void GAME::DrawEnemies(int map_number)
{
  for(int i = 0; i < MAX_ENEMIES; i++)
    {
      if(maps[map_number].enemies[i].Alive())
	{
	  int tile_x = (maps[map_number].enemies[i].GetX() - 8) / 16;
	  int tile_y = (maps[map_number].enemies[i].GetY() - 8) / 16;
	  maps[map_number].enemies[i].Display(static_cast<int>((maps[map_number].tiles_alpha[tile_x][tile_y] - 50) * 1.25));
	  maps[map_number].enemies[i].UpdateAI();
	}
    }
}

void GAME::CheckAttackCollision(int map_number)
{
  int player_x = player.GetX() + 8;
  int player_y = player.GetY() + 8;
  int enemy_x;
  int enemy_y;
  int x_distance;
  int y_distance;

  if(player.Attacking())
    {
      for(int i = 0; i < MAX_ENEMIES; i++)
	{
	  if(maps[map_number].enemies[i].Alive())
	    {
	      enemy_x = maps[map_number].enemies[i].GetX();
	      enemy_y = maps[map_number].enemies[i].GetY();
	      x_distance = enemy_x - player_x;
	      y_distance = enemy_y - player_y;
	      if(player.GetFacing() == UP)
		{
		  if(x_distance < 12 && x_distance > -12)
		    {
		      if(y_distance < 0 && y_distance > -24)
			{
			  maps[map_number].enemies[i].Destroy();
			  player.AddHP();
			  hit.Play();
			  CheckCompleteness();
			}
		    }
		}
	      if(player.GetFacing() == DOWN)
		{
		  if(x_distance < 12 && x_distance > -12)
		    {
		      if(y_distance > 0 && y_distance < 24)
			{
			  maps[map_number].enemies[i].Destroy();
			  player.AddHP();
			  CheckCompleteness();
			  hit.Play();
			}
		    }
		}
	      if(player.GetFacing() == LEFT)
		{
		  if(x_distance < 0 && x_distance > -24)
		    {
		      if(y_distance > -12 && y_distance < 12)
			{
			  maps[map_number].enemies[i].Destroy();
			  player.AddHP();
			  CheckCompleteness();
			  hit.Play();
			}
		    }
		}
	      if(player.GetFacing() == RIGHT)
		{
		  if(x_distance > 0 && x_distance < 24)
		    {
		      if(y_distance > -12 && y_distance < 12)
			{
			  maps[map_number].enemies[i].Destroy();
			  player.AddHP();
			  hit.Play();
			  CheckCompleteness(); 
			}
		    }
		}
	    }
	}
    }
}

void GAME::CheckCollisionHit(int map_number)
{
  int enemy_x;
  int enemy_y;
  int x_distance;
  int y_distance;
  int player_x = player.GetX() + 8;
  int player_y = player.GetY() + 8;
  for(int i = 0; i < MAX_ENEMIES; i++)
    {
      if(maps[map_number].enemies[i].Alive())
	{
	  enemy_x = maps[map_number].enemies[i].GetX();
	  enemy_y = maps[map_number].enemies[i].GetY();
	  
	  x_distance = abs(enemy_x - player_x);
	  y_distance = abs(enemy_y - player_y);

	  //Calculate distance, the end.
	  if(sqrt(x_distance * x_distance + y_distance * y_distance) < 8)
	    {
	      if(player.GetInvincible() < 0)
		{
		  life_down.Play();
		}
	      player.ReduceHP();
	    }
	}
    }
}

void GAME::CheckExitCollisions(void)
{
  int playerY = player.GetTileY();
  int playerX = player.GetTileX();

  //We are on an exit tile.
  if(maps[current_room].tiles[playerX][playerY] > 0 && maps[current_room].tiles[playerX][playerY] < 5)
    {
      
      //Check to see if we go to the previous room or next room.
      switch(path[current_room])
	{
	case(UP):
	  if((playerX == 9 || playerX == 10) && playerY == 0)
	    {
	      GoNextRoom();
	    }
	  else
	    {
	      GoPrevRoom();
	    }
	  break;
	case(DOWN):
	  if((playerX == 9 || playerX == 10) && playerY == 14)
	    {
	      GoNextRoom();
	    }
	  else
	    {
	      GoPrevRoom();
	    }
	  break;
	case(LEFT):
	  if((playerY == 7 || playerY == 8) && playerX == 0)
	    {
	      GoNextRoom();
	    }
	  else
	    {
	      GoPrevRoom();
	    }
	  break;
	case(RIGHT):
	  if((playerY == 7 || playerY == 8) && playerX == 19)
	    {
	      GoNextRoom();
	    }
	  else
	    {
	      GoPrevRoom();
	    }
	  break;
	}
    }
}

void GAME::GoNextRoom(void)
{
  DUNGEON_DISPLAY display_next_room;

  current_room++;
  
  //Position Player.
  switch(path[current_room - 1])
    {
    case(UP):
      player.SetXY(160 - 8, 240 - 40);
      break;
    case(DOWN):
      player.SetXY(160 - 8, 40);
      break;
    case(LEFT):
      player.SetXY(320 - 40, 120 - 8);
      break;
    case(RIGHT):
      player.SetXY(40, 120 - 8);
      break;
    }

  display_next_room.SetLevel(current_room);
  display_next_room.SetFiends(EnemiesLeftInMap(current_room));
  display_next_room.DoScene();
}

void GAME::GoPrevRoom(void)
{
  DUNGEON_DISPLAY display_next_room;

  current_room--;

  switch(path[current_room])
    {
    case(UP):
      player.SetXY(160 - 8, 40);
      break;
    case(DOWN):
      player.SetXY(160 - 8, 240 - 40);
      break;
    case(LEFT):
      player.SetXY(40, 120 - 8);
      break;
    case(RIGHT):
      player.SetXY(320 - 40, 120 - 8);
      break;
    }

  display_next_room.SetLevel(current_room);
  display_next_room.SetFiends(EnemiesLeftInMap(current_room));
  display_next_room.DoScene();
}

void GAME::SetVars(int dif, int maps)
{
  difficulty = dif;
  num_maps = maps;
}

void GAME::DrawTextBox(char *text)
{
}

void GAME::GenerateMaps(void)
{
  int random;
  for(int i = 0; i < num_maps; i++)
    {
      //Create paths.
      random = rand() % 4;
	  
      if(random == 0)
	{
	  path[i] = UP;
	}
      
      if(random == 1)
	{
	  path[i] = DOWN;
	}
      
      if(random == 2)
	{
	  path[i] = LEFT;
	}
      
      if(random == 3)
	{
	  path[i] = RIGHT;
	}
	
      if(i > 0)
	{
	  while((path[i] == UP && path[i - 1] == DOWN) || (path[i] == DOWN && path[i - 1] == UP) || (path[i] == LEFT && path[i - 1] == RIGHT) || (path[i] == RIGHT && path[i - 1] == LEFT))
	    {
	      random = rand() % 4;
	      
	      if(random == 0)
		{
		  path[i] = UP;
		}

	      if(random == 1)
		{
		  path[i] = DOWN;
		}

	      if(random == 2)
		{
		  path[i] = LEFT;
		}

	      if(random == 3)
		{
		  path[i] = RIGHT;
		}
 
	    }
	}

      //Now set each tile.
      for(int j = 0; j < 15; j++)
	{
	  for(int k = 0; k < 20; k++)
	    {
	      //Generate map tilage.
	      maps[i].tiles[k][j] = 0;
	      maps[i].tiles_alpha[k][j] = 50;
	    }
	}
      
      //Create Next exits.
      if(i != num_maps - 1)
      {
	  switch(path[i])
	    {
	    case(UP):
	      maps[i].tiles[9][0] = 1;
	      maps[i].tiles[10][0] = 1;
	      break;
	    case(DOWN):
	      maps[i].tiles[9][14] = 2;
	      maps[i].tiles[10][14] = 2;
	      break;
	    case(LEFT):
	      maps[i].tiles[0][7] = 3;
	      maps[i].tiles[0][8] = 3;
	      break;
	    case(RIGHT):
	      maps[i].tiles[19][7] = 4;
	      maps[i].tiles[19][8] = 4;
	      break;
	    }
	  }

      //Last map past.  If it lead up, put an exit below
      if(i != 0)
	{
	  switch(path[i-1])
	    {
	    case(UP):
	      maps[i].tiles[9][14] = 2;
	      maps[i].tiles[10][14] = 2;
	      break;
	    case(DOWN):
	      maps[i].tiles[9][0] = 1;
	      maps[i].tiles[10][0] = 1;
	      break;
	    case(LEFT):
	      maps[i].tiles[19][7] = 4;
	       maps[i].tiles[19][8] = 4;
	      break;
	    case(RIGHT):
	      maps[i].tiles[0][7] = 3;
	      maps[i].tiles[0][8] = 3;
	      break;
	    }
	}
      
      int e;
      int rand_enemies = max_enem - min_enem;
      rand_enemies = rand() % rand_enemies;
      rand_enemies = rand_enemies + min_enem;

      int rand_enem_speed = enem_speed_max - enem_speed_min;
      rand_enem_speed = rand() % rand_enem_speed;
      rand_enem_speed = rand_enem_speed + enem_speed_min;
      

      //Now generate enemies...
      maps[i].num_enemies = rand_enemies;

      for(e = 0; e < maps[i].num_enemies; e++)
	{
	  maps[i].enemies[e].Init(rand_enem_speed);
	}
      while(e < MAX_ENEMIES)
	{
	  maps[i].enemies[e].Destroy();
	  e++;
	}
    }
}

void GAME::DrawMap(int map_num)
{
  int alpha;
  int distance_x;
  int distance_y;
  int distance;
  int tile_x;
  int tile_y;
  int player_x = player.GetX() + 8;
  int player_y = player.GetY() + 8;
  int bdis;
  double player_hp_percent = static_cast<double>(player.GetHP()) / static_cast<double>(player.GetMaxHP());

  for(int i = 0; i < 20; i++)
    {
      for(int k = 0; k < 15; k++)
	{
	  tile_x = i*16 + 8;
	  tile_y = k*16 + 8;
	  distance_x = abs(player_x - tile_x);
	  distance_y = abs(player_y - tile_y);

	  distance = static_cast<int>(sqrt(static_cast<double>((distance_x * distance_x) + (distance_y * distance_y))));

	  if(!player.Attacking())
	    {
	      alpha = 255 - (distance * 2);
	    }
	  else
	    {
	      bdis = static_cast<int>(abs(static_cast<int>(player.GetAttackAngle())));
	      if(bdis < 30)
		{
		  alpha = 255 - (distance * 2 - bdis  * 4);
		}
	      else
		{
		  bdis = 60 - bdis;
		  alpha = 255 - (distance * 2 - bdis  * 4);
		}
	    }
	  
	  if(alpha > 255)
	    {
	      alpha = 255;
	    }

	  if(alpha <= 50)
	    {
	      alpha = 50;
	    }

	  if(player_hp_percent < .50)
	    {
	      player_hp_percent = .50;
	    }

	  //alpha = static_cast<int>((static_cast<double>(alpha) * player_hp_percent));

	  if(dungeon_complete && map_num == (num_maps - 1))
	    {
	      int alpha_too;
	      int dis_x = abs(160 - (i * 16 + 8));
	      int dis_y = abs(120 - (k * 16 + 8));
	      int dis = static_cast<int>(sqrt(static_cast<double>((dis_x * dis_x) + (dis_y * dis_y))));
	      alpha_too = 255 - dis;
	      if(alpha_too > alpha)
		{
		  alpha = alpha_too;
		}
	      
	    }

	  tiles[maps[map_num].tiles[i][k]].DisplayTransparent(i*16, k*16, alpha);

	  maps[map_num].tiles_alpha[i][k] = alpha;
	  //tiles[0].Display(i*16, k*16);
	}
    }
}

void GAME::Pause(void)
{
}

void GAME::UnPause(void)
{
}

void GAME::Setup(void)
{
  enemy_sprites[0].LoadPNG("data/enemy1-0.png", true);
 
  enemy_sprites[1].LoadPNG("data/enemy1-1.png", true);

  tiles[0].LoadPNG("data/tile0.png", true);

  tiles[1].LoadPNG("data/tile1.png", true);

  tiles[2].LoadPNG("data/tile2.png", true);
  
  tiles[3].LoadPNG("data/tile3.png", true);

  tiles[4].LoadPNG("data/tile4.png", true);

  tiles[5].LoadPNG("data/tile5.png", true);

  tiles[6].LoadPNG("data/tile6.png", true);
  
  tiles[7].LoadPNG("data/tile7.png", true);
  
  tiles[8].LoadPNG("data/tile8.png", true);

  black.CreateBitmap(320, 240, 0);
  black.Clear(0, 0, 0);
  fade.Set(0, 100);

  swing.Load("data/light.mp3");
  hit.Load("data/hit.mp3");
  life_down.Load("data/ldown.wav");
  ding.Load("data/ding.mp3");

  player.LoadArt();

  GenerateMaps();

  beaten = false;
  dungeon_complete = false;

  current_room = 0;

  DUNGEON_DISPLAY display_next_room;

  display_next_room.SetLevel(current_room);
  display_next_room.SetFiends(EnemiesLeftInMap(current_room));
  display_next_room.DoScene();
}

void GAME::GameLoop(void)
{
  ClearScreen();
  if(ButtonAPress() && player.GetHP() > 0)
    {
      if(!player.Attacking())
	{
	  swing.Play();
	  player.Attack();
	}
    }
  if(ButtonUpDown() && player.GetHP() > 0 )
    {
      player.MoveUp();
    }

  if(ButtonDownDown() && player.GetHP() > 0)
    {
      player.MoveDown();
    }

  if(ButtonLeftDown() && player.GetHP() > 0)
    {
      player.MoveLeft();
    }

  if(ButtonRightDown() && player.GetHP() > 0 )
    {
      player.MoveRight();
    }

  if(ButtonEscPress())
    {
      player.SetHP(0);
    }

  if(fade.GetValue() >= 254)
    {
      EndScene(0);
    }

  CheckExitCollisions();

  if(OnWinningQuad())
    {
      player.AddEXP(current_room * 5);
      EndScene(1);
    }

  DrawMap(current_room);

  //Also updates the AI.
  DrawEnemies(current_room);

  player.Draw();

  player.DisplayHP();

  CheckAttackCollision(current_room);

  CheckCollisionHit(current_room);
  
  if(player.GetHP() == 0)
    {
      black.DisplayTransparent(0,0,fade.GetIntValue());
      fade.Update();
      if(fade.GetValue() > 255)
	{
	  fade.SetValue(255);
	}
    }
}

void GAME::Cleanup(void)
{
  player.ReleaseArt();
  
  for(int i = 0; i <= 4; i++)
    {
      tiles[i].Destroy();
    }  

  enemy_sprites[0].Destroy();
  enemy_sprites[1].Destroy();

  black.Destroy();

  swing.Destroy();
  hit.Destroy();
  life_down.Destroy();
  ding.Destroy();
}

int GAME::EnemiesLeftInMap(int map_number)
{
  int counter = 0;

  for(int i = 0; i < MAX_ENEMIES; i++)
    {
      if(maps[map_number].enemies[i].Alive())
	{
	  counter++;
	}
    }
  return counter;
}

int GAME::EnemiesLeft(void)
{
  int counter = 0;
  
  for(int i = 0; i < num_maps; i++)
    {
      counter+= EnemiesLeftInMap(i);
    }
  
  return counter;
}

void GAME::CheckCompleteness(void)
{
  if(dungeon_complete == false)
    {
      if(EnemiesLeft() == 0)
	{
	  dungeon_complete = true;
	  ding.Play();
	  CreateExitWarp();
	}
    }
}

void GAME::CreateExitWarp(void)
{
  maps[num_maps - 1].tiles[9][7] = 5;
  maps[num_maps - 1].tiles[10][7] = 6;
  maps[num_maps - 1].tiles[9][8] = 7;
  maps[num_maps - 1].tiles[10][8]  = 8;
}

void PLAYER::SetXY(int new_x, int new_y)
{
  x = static_cast<double>(new_x);
  y = static_cast<double>(new_y);
}

void PLAYER::Draw(void)
{
  int tempframe = static_cast<int>(aniframe);
  int frame;
  
  if(moving)
    {
      if(facing == UP || facing == DOWN)
	{
	  int tempframe = static_cast<int>(aniframe);
	  switch(tempframe)
	    {
	    case(0):
	    case(2):
	    case(4):
	      frame = 1;
	      break;
	    case(1):
	    case(3):
	    case(5):
	      frame = 2;
	      break;
	    default:
	      break;
	    }
	}
      else
	{
	  switch(tempframe)
	    {
	    case(0):
	    case(2):
	    case(4):
	      frame = 0;
	      break;
	    case(1):
	    case(3):
	    case(5):
	      frame = 1;
	      break; 
	    default:
	      break;
	    }
	}
    }
  else
    {
      frame = 0;
    }
 
  switch(facing)
    {
    case(DOWN):
      {
	frame+=3;
	break;
      }
    case(LEFT):
      {
	frame+=6;
	break;
      }
    case(RIGHT):
      {
	frame+=8;
	break;
      }
    default:
      break;
    }

  //Now we have a sprite frame.
  if(invincible.GetIntValue() >= 0 || hp == 0)
    {
      sprites[frame].DisplayTrans(static_cast<int>(x), static_cast<int>(y), 128);
      invincible.Update();
    }
  else
    {
      sprites[frame].Display(static_cast<int>(x), static_cast<int>(y));
    }

  SetMoving(false);

  //Draw the attack...
  if(attacking)
    {
      SPRITE temp_sprite;
      int angle_offset = 30;

      switch(facing)
	{
	case(UP):
	  angle_offset = -90 ;
	  break;
	case(RIGHT):
	  angle_offset = -30;
	  break;
	case(LEFT):
	  angle_offset = -140;
	  break;
	default:
	  break;
	}

      temp_sprite.CreateBitmap(30, 30, 32);
      temp_sprite.Clear32(0,0,0,1);
      attack_sprite.SetDestination(temp_sprite.GetBitmapPtr());
      attack_sprite.DisplayRotate(0,0,attack_angle + angle_offset);
      temp_sprite.DisplayAlpha(static_cast<int>(x) - 7, static_cast<int>(y) - 7);
      temp_sprite.Destroy();
      attack_angle-= 400*frame_time;
      if(attack_angle < -60)
	{
	  attacking = false;
	}
    }
}

void PLAYER::ResetPlayer(void)
{
  hp = 100;
  max_hp = 100;
  level = 1;
  exp = 0;
  level_up_exp = 20;
}

void PLAYER::LoadArt(void)
{
  SPRITE temp;

  aniframe = 0;
  x = 160 - 8;
  y = 120 - 8;
  attacking = false;
  facing = DOWN;

  invincible.Set(1, -1);

  temp.LoadPNG("data/herou0.png", true);
  sprites[0].Convert(temp);

  temp.LoadPNG("data/herou1.png", true);
  sprites[1].Convert(temp);

  temp.LoadPNG("data/herou2.png", true);
  sprites[2].Convert(temp);

  temp.LoadPNG("data/herod0.png", true);
  sprites[3].Convert(temp);

  temp.LoadPNG("data/herod1.png", true);
  sprites[4].Convert(temp);

  temp.LoadPNG("data/herod2.png", true);
  sprites[5].Convert(temp);

  temp.LoadPNG("data/herol0.png", true);
  sprites[6].Convert(temp);

  temp.LoadPNG("data/herol1.png", true);
  sprites[7].Convert(temp);

  temp.LoadPNG("data/heror0.png", true);
  sprites[8].Convert(temp);

  temp.LoadPNG("data/heror1.png", true); 
  sprites[9].Convert(temp);

  attack_sprite.LoadPNG("data/attack.png", false);
  
  temp.Destroy();
}

void PLAYER::ReleaseArt(void)
{
  for(int i = 0; i < 10; i++)
    {
      sprites[i].Destroy();
    }

  attack_sprite.Destroy();
}

void PLAYER::SetMoving(bool move)
{
  moving = move;
}


#define PLAYER_SPEED 64

void PLAYER::MoveUp(void)
{
  SetMoving(true);
  if(!attacking)
    {
      facing = UP;
      y-= PLAYER_SPEED * frame_time;
      if(y < - 10)
	{
	  y = - 10;
	}
      UpdateAnimationFrame();
    }
  else
    {
      SetMoving(false);
    }
}

void PLAYER::MoveDown(void)
{
  SetMoving(true);
  if(!attacking)
    {
      facing = DOWN;
      y+= PLAYER_SPEED * frame_time;
      if(y > 240 + 10)
	{
	  y = 240 + 10;
	}
      UpdateAnimationFrame();
    }
  else
    {
      SetMoving(false);
    }
}

void PLAYER::MoveLeft(void)
{
  SetMoving(true);
  if(!attacking)
    {
      facing = LEFT;
      x-= PLAYER_SPEED * frame_time;
      if(x < -8)
	{
	  x = -8;
	}
      UpdateAnimationFrame();
    }
  else
    {
      SetMoving(false);
    }
}

void PLAYER::MoveRight(void)
{
  SetMoving(true);
  if(!attacking)
    {
      facing = RIGHT;
      x+= PLAYER_SPEED * frame_time;
      if(x > 320 + 8)
	{
	  x = 320 + 8;
	}
      UpdateAnimationFrame();
    }
  else
    {
      SetMoving(false);
    }
  
}

void PLAYER::UpdateAnimationFrame(void)
{
  aniframe+= frame_time * 8;
  if(aniframe >= 6)
    {
      aniframe = aniframe - 6;
    }
}

void PLAYER::Attack(void)
{
  if(attacking == false)
    {
      attacking = true;
      attack_angle = 0;
    }
}

int PLAYER::GetTileX(void)
{
  return static_cast<int>(x + 8) / 16;
}

int PLAYER::GetTileY(void)
{
  return static_cast<int>(y + 10) / 16;
}

void PLAYER::ReduceHP(void)
{
  if(invincible.GetValue() <= 0)
    {
      hp -= 20;
      if(hp < 0)
	{
	  hp = 0;
	}
      SetInvincible();
    }
}

void PLAYER::AddHP(void)
{
  hp++;
  AddEXP(1); 
  if(hp > max_hp)
    {
      hp = max_hp;
    }
}

void PLAYER::AddEXP(int amount)
{
  exp+= amount;
  
  if(exp >= level_up_exp)
    {
      LEVEL_UP level_up;

      level_up.DoScene();
      max_hp+= 10;
      hp+=10;
      level_up_exp = level_up_exp * 2;
      level++;
    }
}

void ENEMY::Init(int new_speed)
{
  speed = new_speed;
  alive = true;
  x = (rand() % 300) + 8;
  y = (rand() % 220) + 8;
  aniframe = 0;
  goalx = static_cast<int>(x);
  goaly = static_cast<int>(y);
}

void ENEMY::Destroy(void)
{
  alive = false;
}

void ENEMY:: UpdateAI(void)
{
  if(alive)
    {
      if(goalx == x && goaly == y)
	{
	  goalx = (rand() % 300) + 8;
	  goaly = (rand() % 220) + 8;
	}
      else
	{
	  if(x < goalx)
	    {
	      x+= frame_time * speed;
	      if(x > goalx)
		{
		  x = goalx;
		}
	    }
	  else
	    {
	      x-= frame_time * speed;
	      if(x < goalx)
		{
		  x = goalx;
		}
	    }

	  if(y < goaly)
	    {
	      y+= frame_time * speed;
	      if(y > goaly)
		{
		  y = goaly;
		}
	    }
	  else
	    {
	      y-= frame_time * speed;
	      if(y < goaly)
		{
		  y = goaly;
		}
	    }
	}

      aniframe+= frame_time * 4;
      while(aniframe >= 2)
	{
	  aniframe-= 2;
	}
    }
}

void ENEMY::Display(int alpha)
{
  if(alive)
    {
      if(alpha < 0)
	{
	  alpha = 0;
	}
      if(alpha > 255)
	{
	  alpha = 255;
	}
      enemy_sprites[static_cast<int>(aniframe)].DisplayTransparent(static_cast<int>(x) - 8, static_cast<int>(y) - 8, alpha);
    }
}

void PLAYER::DisplayHP(void)
{
  alfont_textprintf_aa(back_buffer, fonts[10], 2, 0, makecol(255,255,255), "Level: %d", level);

  alfont_textprintf_aa(back_buffer, fonts[10], 2, 8, makecol(255,255,255), "HP: %d/%d", hp, max_hp);
}
