/*
 *    RTL.CPP - realtime lighting demo
 *
 *    By Shawn Hargreaves, 30 May 1998
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <allegro.h>

#include "rtl.h"



// how wide is the world?
#define W      512



// lots of nice global variables
int finished;
int dead;
int fade;

DATAFILE *data;

BITMAP *buffer;
BITMAP *buffer2;
BITMAP *lights;

COLOR_MAP *greyscale_add_map;
COLOR_MAP *color_add_map;
COLOR_MAP *lighting_map;

fix scroll;
fix scroll_vel;

int show_lights = TRUE;
int show_flares = TRUE;
int show_grey = FALSE;

int old_show_grey = FALSE;
int old_fade = 0;

int score;

int ambient_light = 32;

volatile int update_count;



// timer callback for controlling the game speed
void update_counter(void)
{
   update_count++;
}

END_OF_FUNCTION(update_counter);



// helper for drawing a lensflare component
void draw_flare(BITMAP *bmp, fix x, fix y, fix pos, int r, int g, int b, int size, fix a)
{
   int xx = x*pos + SCREEN_W/2*(1-pos);
   int yy = y*pos + SCREEN_H/2*(1-pos);

   circlefill(bmp, xx, yy, size, makecol(r*a, g*a, b*a));
}



// draws the lensflare effect
void draw_lens_flare(BITMAP *bmp, fix x, fix y, fix a)
{
   if ((x < 0) || (x >= SCREEN_W) || (y < 0) || (y > SCREEN_H))
      return;

   a *= ((SCREEN_W/2) - ABS(x-SCREEN_W/2)) / (SCREEN_W/2);
   a *= ((SCREEN_H/2) - ABS(y-SCREEN_H/2)) / (SCREEN_H/2);

   if (a > 1)
      a = 1;

   draw_flare(bmp, x, y, 0.7,  32, 8,  8,  11, a);
   draw_flare(bmp, x, y, 0.4,  0,  64, 32, 3,  a);
   draw_flare(bmp, x, y, 0.2,  32, 32, 0,  12, a);
   draw_flare(bmp, x, y, -0.4, 32, 64, 0,  8,  a);
   draw_flare(bmp, x, y, -0.9, 48, 24, 0,  5,  a);
}



// every ingame object is derived from this
class Thing
{
public:
   Thing() { next = NULL; kill = FALSE; }
   virtual ~Thing() { }

   static Thing *list;

   static void add(Thing *thing);
   static void do_update();
   static void do_collide(fix xpos, fix ypos);
   static void do_draw(BITMAP *bmp);
   static void do_draw_lights(BITMAP *bmp);
   static void do_draw_flares(BITMAP *bmp);
   static void kill_all();

protected:
   virtual int update() { return 0; }
   virtual int collide(fix xpos, fix ypos) { return 0; }
   virtual void draw(BITMAP *bmp) { }
   virtual void draw_light(BITMAP *bmp) { }
   virtual void draw_flare(BITMAP *bmp) { }

   Thing *next;

   int kill;

   static Thing *prev;
   static int in_update;
};



// global list of all the objects in the game
Thing *Thing::list = NULL;

Thing *Thing::prev = NULL;
int Thing::in_update = FALSE;



// adds a new object to the list
void Thing::add(Thing *thing)
{
   thing->next = list;
   list = thing;

   if (in_update)
      if (!prev)
	 prev = thing;
}



// updates all the objects in the list
void Thing::do_update()
{
   Thing *thing = list;
   Thing *next;

   in_update = TRUE;
   prev = NULL;

   while (thing) {
      next = thing->next;

      if ((thing->kill) || (thing->update() != 0)) {
	 delete thing;

	 if (prev)
	    prev->next = next;
	 else
	    list = next;
      }
      else
	 prev = thing;

      thing = next;
   }

   in_update = FALSE;
}



// checks a collision against all the objects in the list
void Thing::do_collide(fix xpos, fix ypos)
{
   if (finished)
      return;

   Thing *thing = list;

   while (thing) {
      if (thing->collide(xpos, ypos) != 0)
	 thing->kill = TRUE;

      thing = thing->next;
   }
}



// draws all the objects in the list
void Thing::do_draw(BITMAP *bmp)
{
   Thing *thing = list;

   while (thing) {
      if (!thing->kill)
	 thing->draw(bmp);
      thing = thing->next;
   }
}



// draws light maps for all the objects in the list
void Thing::do_draw_lights(BITMAP *bmp)
{
   Thing *thing = list;

   while (thing) {
      if (!thing->kill)
	 thing->draw_light(bmp);
      thing = thing->next;
   }
}



// draws flare overlays for all the objects in the list
void Thing::do_draw_flares(BITMAP *bmp)
{
   Thing *thing = list;

   while (thing) {
      if (!thing->kill)
	 thing->draw_flare(bmp);
      thing = thing->next;
   }
}



// destroys all the objects at the end of the program
void Thing::kill_all()
{
   Thing *thing = list;
   Thing *next;

   while (thing) {
      next = thing->next;
      delete thing;
      thing = next;
   }

   list = NULL;
}



#define EXPLODE_FRAMES        64
#define EXPLODE_SIZE          80



// the explosion object
class Explosion : public Thing
{
public:
   Explosion(fix xpos, fix ypos);

   static void generate();

protected:
   int update();
   void draw_light(BITMAP *bmp);
   void draw_flare(BITMAP *bmp);

   fix x;
   fix y;
   int t;

   static BITMAP *gfx[EXPLODE_FRAMES];
};



// pregenerated explosion animation
BITMAP *Explosion::gfx[EXPLODE_FRAMES];



// borrowed from the Allegro demo.c
void Explosion::generate()
{
   unsigned char *p;
   int i, j;
   int x, y;
   int xx, yy;
   int color;

   #define HOTSPOTS  48

   struct HOTSPOT
   {
      fix x, y;
      fix xc, yc;
   } hot[HOTSPOTS];

   for (i=0; i<HOTSPOTS; i++) {
      hot[i].x = hot[i].y = EXPLODE_SIZE/2;
      hot[i].xc = (float)(rand()&0xFFFF)/65536-0.5;
      hot[i].yc = (float)(rand()&0xFFFF)/65536-0.5;
   }

   for (i=0; i<EXPLODE_FRAMES; i++) {
      gfx[i] = create_bitmap(EXPLODE_SIZE, EXPLODE_SIZE);
      clear(gfx[i]);

      color = ((i<8) ? i*6 : (48-(i-8)*48/(EXPLODE_FRAMES-8)));

      for (j=0; j<HOTSPOTS; j++) {
	 for (x=-6; x<=6; x++) {
	    for (y=-6; y<=6; y++) {
	       xx = hot[j].x + x;
	       yy = hot[j].y + y;
	       if ((xx>=0) && (yy>=0) && (xx<EXPLODE_SIZE) && (yy<EXPLODE_SIZE)) {
		  p = gfx[i]->line[yy] + xx;
		  *p += (color >> ((ABS(x)+ABS(y))/3));
		  if (*p > 127)
		     *p = 127;
	       }
	    }
	 }

	 hot[j].x += hot[j].xc;
	 hot[j].y += hot[j].yc;

	 if (j&1)
	    hot[j].yc += 0.02;
      }

      for (x=0; x<EXPLODE_SIZE; x++) {
	 for (y=0; y<EXPLODE_SIZE; y++) {
	    j = gfx[i]->line[y][x];
	    gfx[i]->line[y][x] = makecol(MID(0, j*2, 255), 
					 MID(0, j*8-768, 255), 
					 MID(0, j*16-1792, 255));
	 }
      }
   }
}



// creates an explosion
Explosion::Explosion(fix xpos, fix ypos)
{
   do_collide(xpos, ypos);

   x = xpos;
   y = ypos;

   t = 0;
}



// animates an explosion
int Explosion::update()
{
   t++;

   if (t >= EXPLODE_FRAMES)
      return -1;

   if ((x-scroll >= 0) && (x-scroll < SCREEN_W)) {
      ambient_light += 64-t*64/EXPLODE_FRAMES;
      if (ambient_light > 255)
	 ambient_light = 255;
   }

   return 0;
}



// draws light around an explosion
void Explosion::draw_light(BITMAP *bmp)
{
   if ((x-scroll < -96) || (x-scroll > SCREEN_W+96))
      return;

   fix size = (fix)t*96/EXPLODE_FRAMES;
   size = 96-(96-size)*(96-size)/96;
   circlefill(bmp, x-scroll, y, size, 255-t*255/EXPLODE_FRAMES);
}



// draws the main part of the explosion
void Explosion::draw_flare(BITMAP *bmp)
{
   if ((x-scroll < -64) || (x-scroll > SCREEN_W+64))
      return;

   draw_trans_sprite(bmp, gfx[t], x-scroll-EXPLODE_SIZE/2, y-EXPLODE_SIZE/2);
}



// the missile object
class Missile : public Thing
{
public:
   Missile(fix pos);

protected:
   int update();
   void draw(BITMAP *bmp);
   void draw_light(BITMAP *bmp);
   void draw_flare(BITMAP *bmp);

   fix x;
   fix y;
   fix vel;
};



// creates a missile
Missile::Missile(fix pos)
{
   x = pos;
   y = SCREEN_H-16;
   vel = 0;
}



// animates a missile
int Missile::update()
{
   if (vel < 1)
      vel += 0.05;

   y -= vel*4;

   if (y < 24) {
      add(new Explosion(x, y));
      return -1;
   }

   return 0;
}



// draws a missile
void Missile::draw(BITMAP *bmp)
{
   BITMAP *missile = (BITMAP *)data[DATA_MISSILE].dat;
   draw_sprite(bmp, missile, x-scroll-missile->w/2, y-missile->h/2);
}



// draws light around a missile
void Missile::draw_light(BITMAP *bmp)
{
   BITMAP *mask = (BITMAP *)data[DATA_LIGHT_MASK].dat;
   draw_trans_sprite(bmp, mask, x-scroll-mask->w/2, y-8);
}



// draws a missile lens flare
void Missile::draw_flare(BITMAP *bmp)
{
   draw_lens_flare(bmp, x-scroll, y, vel);
}



// the player ship object
class Player : public Thing
{
public:
   Player();

protected:
   int update();
   int collide(fix xpos, fix ypos);
   void draw(BITMAP *bmp);
   void draw_light(BITMAP *bmp);

   fix pos;
   fix vel;

   int was_l;
   int was_f;
   int was_g;
   int was_fire;
};



// initialises the player
Player::Player()
{
   pos = W/2;
   vel = 0;

   was_l = FALSE;
   was_f = FALSE;
   was_g = FALSE;
   was_fire = FALSE;
}



// animates the player
int Player::update()
{
   if (!finished) {
      // esc quits
      if (key[KEY_ESC])
	 finished = TRUE;

      // 'l' toggles lights on or off
      if ((key[KEY_L]) && (!was_l))
	 show_lights = !show_lights;

      was_l = key[KEY_L];

      // 'f' toggles flares on or off
      if ((key[KEY_F]) && (!was_f))
	 show_flares = !show_flares;

      was_f = key[KEY_F];

      // 'g' toggles greyscale lightmap display
      if ((key[KEY_G]) && (!was_g))
	 show_grey = !show_grey;

      was_g = key[KEY_G];

      // arrow keys move the ship
      if (key[KEY_LEFT])
	 vel--;

      if (key[KEY_RIGHT])
	 vel++;
   }

   pos += vel;
   vel *= 0.8;

   // scroll the world to keep up with us
   if (pos < scroll+32)
      scroll_vel--;
   else if (pos > scroll+SCREEN_W-32)
      scroll_vel++;

   scroll += scroll_vel;
   scroll_vel *= 0.8;

   // space fires a missile
   if (!finished) {
      if ((key[KEY_SPACE]) && (!was_fire))
	 add(new Missile(pos));

      was_fire = key[KEY_SPACE];
   }

   return 0;
}



// checks if we have been hit by an explosion
int Player::collide(fix xpos, fix ypos)
{
   if ((xpos > pos-24) && (xpos < pos+24) && (ypos > SCREEN_H-32)) {
      finished = TRUE;
      dead = TRUE;
   }

   return 0;
}



// draws the player
void Player::draw(BITMAP *bmp)
{
   if (dead)
      return;

   BITMAP *ship = (BITMAP *)data[DATA_SHIP].dat;
   draw_sprite(bmp, ship, pos-scroll-ship->w/2, SCREEN_H-ship->h-6);
}



// draws lights around the player
void Player::draw_light(BITMAP *bmp)
{
   if (dead)
      return;

   BITMAP *mask = (BITMAP *)data[DATA_PLAYER_MASK].dat;
   draw_trans_sprite(bmp, mask, pos-scroll-mask->w/2, SCREEN_H-mask->h);
}



// the bomb object
class Bomb : public Thing
{
public:
   Bomb(fix pos);

protected:
   int update();
   void draw(BITMAP *bmp);
   void draw_light(BITMAP *bmp);
   void draw_flare(BITMAP *bmp);

   fix x;
   fix y;
};



// drops a new bomb
Bomb::Bomb(fix pos)
{
   x = pos;
   y = 28;
}



// animates the bomb
int Bomb::update()
{
   y += 1 + fix(score)/128;

   if (y > SCREEN_H-16) {
      add(new Explosion(x, y));
      return 1;
   }

   return 0;
}



// draws the bomb
void Bomb::draw(BITMAP *bmp)
{
   if ((x-scroll < -32) || (x-scroll > SCREEN_W+32))
      return;

   BITMAP *bomb = (BITMAP *)data[DATA_BOMB].dat;
   draw_sprite(bmp, bomb, x-scroll-bomb->w/2, y-bomb->h);
}



// draws lights around the bomb
void Bomb::draw_light(BITMAP *bmp)
{
   if ((x-scroll < -32) || (x-scroll > SCREEN_W+32))
      return;

   BITMAP *mask = (BITMAP *)data[DATA_BOMB_MASK].dat;
   draw_trans_sprite(bmp, mask, x-scroll-mask->w/2, y-mask->h);
}



// draws a bomb lens flare
void Bomb::draw_flare(BITMAP *bmp)
{
   if ((x-scroll < 0) || (x-scroll > SCREEN_W))
      return;

   draw_lens_flare(bmp, x-scroll, y, 0.8);
}



// the flying saucer object
class Saucer : public Thing
{
public:
   Saucer();
   ~Saucer();

   static int count;

protected:
   int update();
   int collide(fix xpos, fix ypos);
   void draw(BITMAP *bmp);
   void draw_light(BITMAP *bmp);

   fix pos;
   fix vel;
   int dir;
};



int Saucer::count = 0;



// creates a new flying saucer
Saucer::Saucer()
{
   if (rand()&1) {
      pos = scroll-32-(rand()&255);
      dir = 1;
   }
   else {
      pos = scroll+SCREEN_W+32+(rand()&255);
      dir = -1;
   }

   vel = 0;

   count++;
}



// handles the destruction of a flying saucer
Saucer::~Saucer()
{
   count--;
}



// animates the flying saucer
int Saucer::update()
{
   fix wanted;

   if (pos < scroll-64) {
      dir = 1;
      wanted = 3+scroll_vel;
   }
   else if (pos > scroll+SCREEN_W+64) {
      dir = -1;
      wanted = -3+scroll_vel;
   }
   else
      wanted = dir*2+scroll_vel/2;

   vel = vel*0.99 + wanted*0.01;
   pos += vel;

   if ((rand()&0xFF) < 2)
      add(new Bomb(pos));

   return 0;
}



// checks if we have been hit by an explosion
int Saucer::collide(fix xpos, fix ypos)
{
   if ((xpos > pos-24) && (xpos < pos+24) && (ypos < 32)) {
      score++;
      return 1;
   }

   return 0;
}



// draws the flying saucer
void Saucer::draw(BITMAP *bmp)
{
   if ((pos-scroll < -32) || (pos-scroll > SCREEN_W+32))
      return;

   BITMAP *saucer = (BITMAP *)data[DATA_SAUCER].dat;
   draw_sprite(bmp, saucer, pos-scroll-saucer->w/2, 16-saucer->h/2);
}



// draws lights around the flying saucer
void Saucer::draw_light(BITMAP *bmp)
{
   if ((pos-scroll < -32) || (pos-scroll > SCREEN_W+32))
      return;

   circlefill(bmp, pos-scroll, 16, 12, 96);
}



// streetlamp objects
class Streetlamp : public Thing
{
public:
   Streetlamp(int x);

protected:
   void draw_light(BITMAP *bmp);
   void draw_flare(BITMAP *bmp);

   fix pos;

   int xpos();

   static BITMAP *tmp;
};



// temporary bitmap for flare rotation
BITMAP *Streetlamp::tmp = NULL;



// initialises a streetlamp object
Streetlamp::Streetlamp(int x)
{
   pos = x;

   if (!tmp)
      tmp = create_bitmap(48, 48);
}



// wraps a streetlamp onto the visible screen
int Streetlamp::xpos()
{
   int x = pos-scroll;

   x %= W;
   if (x < -tmp->w/2)
      x += W;
   else if (x > SCREEN_W+tmp->w/2)
      x -= W;

   return x;
}



// draws a streetlamp illumination mask
void Streetlamp::draw_light(BITMAP *bmp)
{
   BITMAP *mask = (BITMAP *)data[DATA_LIGHT_MASK].dat;
   int x = xpos()-mask->w/2;
   int y = 190-mask->h;

   if ((x < -tmp->w/2) || (x > SCREEN_W+tmp->w/2))
      return;

   draw_trans_sprite(bmp, mask, x, y);
}



// draws streetlamp flares
void Streetlamp::draw_flare(BITMAP *bmp)
{
   BITMAP *mask = (BITMAP *)data[DATA_LIGHT_FLARE].dat;
   int x = xpos();
   int y = 164;

   if ((x < -tmp->w/2) || (x > SCREEN_W+tmp->w/2))
      return;

   clear(tmp);
   rotate_sprite(tmp, mask, tmp->w/2-mask->w/2, tmp->w/2-mask->h/2, itofix(64)+fatan2(itofix(SCREEN_H/2-y), itofix(SCREEN_W/2-x)));
   draw_trans_sprite(bmp, tmp, x-tmp->w/2, y-tmp->w/2);
}



// blender function for creating the additive color table
void add_blender(PALETTE pal, int x, int y, RGB *rgb)
{
   int r, g, b;

   r = (int)pal[x].r + (int)pal[y].r;
   g = (int)pal[x].g + (int)pal[y].g;
   b = (int)pal[x].b + (int)pal[y].b;

   rgb->r = MIN(r, 63);
   rgb->g = MIN(g, 63);
   rgb->b = MIN(b, 63);
}



// the opposite of shutdown :-)
void shutup()
{
   char argv0[256];
   char tmp[256];
   int x, y;

   srand(time(NULL));

   allegro_init();
   install_keyboard();
   install_timer();
   set_gfx_mode(GFX_AUTODETECT, 320, 200, 0, 0);

   get_executable_name(argv0, sizeof(argv0));
   replace_extension(tmp, argv0, "dat", sizeof(tmp));
   data = load_datafile(tmp);
   if (!data) {
      set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
      allegro_message("Error loading %s\n", tmp);
      exit(1);
   }

   ((RGB *)data[DATA_PAL].dat)[0] = black_palette[0];
   set_palette((RGB *)data[DATA_PAL].dat);

   rgb_map = new RGB_MAP;
   create_rgb_table(rgb_map, (RGB *)data[DATA_PAL].dat, NULL);

   greyscale_add_map = new COLOR_MAP;
   for (x=0; x<256; x++)
      for (y=0; y<256; y++)
	 greyscale_add_map->data[x][y] = MIN(x+y, 255);

   color_add_map = new COLOR_MAP;
   create_color_table(color_add_map, (RGB *)data[DATA_PAL].dat, add_blender, NULL);

   lighting_map = new COLOR_MAP;
   create_light_table(lighting_map, (RGB *)data[DATA_PAL].dat, 0, 0, 0, NULL);

   buffer = create_bitmap(SCREEN_W, SCREEN_H);
   buffer2 = create_bitmap(SCREEN_W, SCREEN_H);
   lights = create_bitmap(SCREEN_W, SCREEN_H);

   Explosion::generate();

   LOCK_VARIABLE(update_count);
   LOCK_FUNCTION(update_counter);

   install_int_ex(update_counter, BPS_TO_TIMER(70));
}



// displays the game title
int title_screen()
{
   set_palette(black_palette);
   blit((BITMAP *)data[DATA_TITLE].dat, screen, 0, 0, 0, 0, SCREEN_W, SCREEN_H);
   fade_in((RGB *)data[DATA_TITLE_PAL].dat, 3);

   do {
   } while (key[KEY_ESC] || key[KEY_SPACE] || key[KEY_ENTER]);

   for (;;) {
      if (key[KEY_ESC]) {
	 fade_out(5);
	 return FALSE;
      }

      if (key[KEY_SPACE] || key[KEY_ENTER]) {
	 fade_out(5);
	 return TRUE;
      }
   }
}



// activates an instance of the game
void start_game()
{
   Thing::kill_all();

   scroll = (W-SCREEN_W)/2;
   scroll_vel = 0;

   Thing::add(new Player);

   for (int i=0; i<10; i++)
      Thing::add(new Streetlamp(32+i*48));

   fade = 0;
   old_fade = -1;

   finished = FALSE;
   dead = FALSE;

   score = 0;

   update_count = 1;
}



// do an up-date :-)
void flying_fig()
{
   ambient_light = 32;

   Thing::do_update();

   if (Saucer::count < 1+(int)sqrt((fix)score)/2)
      Thing::add(new Saucer);

   if (finished) {
      if (fade > 0)
	 fade -= (dead) ? 2 : 8;
   }
   else {
      if (fade < 256)
	 fade += 4;
   }
}



// redraws the screen
void doodle()
{
   int fo, mo, bo, x, y, c;
   unsigned char *s1, *s2, *s3, *l, *d;

   BITMAP *bg = (BITMAP *)data[DATA_BG].dat;
   BITMAP *mg = (BITMAP *)data[DATA_MG].dat;
   BITMAP *fg = (BITMAP *)data[DATA_FG].dat;

   fo = (int)(scroll)%W;
   mo = (int)(scroll*2/3)%W;
   bo = (int)(scroll/3)%W;

   if (fo < 0)
      fo += W;

   if (mo < 0)
      mo += W;

   if (bo < 0)
      bo += W;

   if (show_grey) {
      // just show the greyscale lighting map
      clear_to_color(buffer, ambient_light);
      color_map = greyscale_add_map;
      drawing_mode(DRAW_MODE_TRANS, NULL, 0, 0);
      Thing::do_draw_lights(buffer);
      drawing_mode(DRAW_MODE_SOLID, NULL, 0, 0);
   }
   else if (show_lights) {
      // draw everything with proper lighting
      blit(fg, buffer2, 0, 0, W-fo, 0, W, SCREEN_H);
      blit(fg, buffer2, 0, 0, -fo, 0, W, SCREEN_H);

      Thing::do_draw(buffer2);

      clear_to_color(lights, ambient_light);
      color_map = greyscale_add_map;
      drawing_mode(DRAW_MODE_TRANS, NULL, 0, 0);
      Thing::do_draw_lights(lights);
      drawing_mode(DRAW_MODE_SOLID, NULL, 0, 0);

      for (y=0; y<SCREEN_H; y++) {
	 s1 = buffer2->line[y];
	 s2 = mg->line[y];
	 s3 = bg->line[y];
	 l = lights->line[y];
	 d = buffer->line[y];

	 for (x=0; x<SCREEN_W; x++) {
	    c = s1[x];
	    if (c)
	       d[x] = lighting_map->data[l[x]][c];
	    else {
	       c = s2[(x+mo)%W];
	       if (c)
		  d[x] = lighting_map->data[l[x]][c];
	       else
		  d[x] = s3[(x+bo)%W];
	    }
	 }
      }
   }
   else {
      // simple draw without lighting effects
      for (y=0; y<SCREEN_H; y++) {
	 s1 = fg->line[y];
	 s2 = mg->line[y];
	 s3 = bg->line[y];
	 d = buffer->line[y];

	 for (x=0; x<SCREEN_W; x++) {
	    c = s1[(x+fo)%W];
	    if (c)
	       d[x] = c;
	    else {
	       c = s2[(x+mo)%W];
	       if (c)
		  d[x] = c;
	       else
		  d[x] = s3[(x+bo)%W];
	    }
	 }
      }

      Thing::do_draw(buffer);
   }

   // overlay flare graphics
   if ((show_flares) && (!show_grey)) {
      color_map = color_add_map;
      drawing_mode(DRAW_MODE_TRANS, NULL, 0, 0);
      Thing::do_draw_flares(buffer);
      drawing_mode(DRAW_MODE_SOLID, NULL, 0, 0);
   }

   text_mode(-1);
   textprintf(buffer, font, 1, 1, makecol(0, 0, 0), "Score: %d", score);
   textprintf(buffer, font, 0, 0, makecol(96, 96, 96), "Score: %d", score);

   if ((show_grey != old_show_grey) || (fade != old_fade)) {
      old_show_grey = show_grey;
      old_fade = fade;

      PALETTE tmp;

      if (show_grey)
	 fade_interpolate(black_palette, (RGB *)data[DATA_GREY_PAL].dat, tmp, fade/4, 0, 255);
      else
	 fade_interpolate(black_palette, (RGB *)data[DATA_PAL].dat, tmp, fade/4, 0, 255);

      set_palette(tmp);
   }
   else
      vsync();

   blit(buffer, screen, 0, 0, 0, 0, SCREEN_W, SCREEN_H);
}



// display the score after a game
void results()
{
   RGB *pal = (RGB *)data[DATA_PAL].dat;

   clear(screen);
   textprintf_centre(screen, font, SCREEN_W/2, SCREEN_H/2, bestfit_color(pal, 63, 63, 63), "Score: %d", score);
   fade_in(pal, 3);

   do {
   } while (key[KEY_ESC] || key[KEY_SPACE] || key[KEY_ENTER]);

   do {
   } while (!key[KEY_ESC] && !key[KEY_SPACE] && !key[KEY_ENTER]);

   fade_out(5);
}



// exit cleanup routine
void shutdown()
{
   Thing::kill_all();

   unload_datafile(data);

   destroy_bitmap(buffer);
   destroy_bitmap(buffer2);
   destroy_bitmap(lights);

   delete rgb_map;
   delete greyscale_add_map;
   delete color_add_map;
   delete lighting_map;
}



// the main program body
int main()
{
   shutup();

   while (title_screen()) {
      start_game();

      while ((!finished) || (fade > 0)) {
	 while (update_count > 0) {
	    flying_fig();
	    update_count--;
	 }

	 doodle();
      }

      if (dead)
	 results();
   }

   shutdown();
   return 0;
}

END_OF_MAIN();

