#include <allegro.h>
#include <string.h>
#include "theme.h"
#include "tiles.h"
#include "tiletype.h"
#include "mentype.h"
#include "tileset.h"
#include "genrand.h"
#include "assert.h"
#include "global.h"
#include "str.h"
#include "gfx.h"

/* Checkerboard flags (for mini-tilebitmap mappings) */
/* The rendering of a tile depends on if it is `black' or `white' */
/*  on a checkerboard */
#define CHECKERBOARD_ODD   1
#define CHECKERBOARD_EVEN  2
#define CHECKERBOARD_ANY   3

/* Transformations needed to transform a bitpattern to the corresponding */
/*  lower left corner pattern */
/* Bit 1 - Rotate 90 degrees left */
/* Bit 2 - Rotate 180 degrees left */
/* Bit 3 - flip vertical (top <==> down) */
/* Bit 4 - flip horizontal (left <==> right) */
static const int flip_to_ll[3][3] = {
   { 4, 2, 2 },
   { 3, 0, 1 },
   { 0, 0, 8 }
};

/* Rotation matrices */
static const int rot90[2][2] = {    /* 90 degrees anti-clockwise */
   { 0, 1 },
   {-1, 0 }
};

/* Rotation matrices */
static const int rot180[2][2] = {   /* 180 degrees */
   {-1, 0 },
   { 0,-1 }
};

/* Rotation matrices */
static const int rot270[2][2] = {   /* 270 degrees anti-clockwise */
   { 0,-1 },
   { 1, 0 }
};

/* Rotation angles */
static const fixed quarter = 64<<16;
static const fixed half = 128<<16;
static const fixed threequarter = 192<<16;

static int num_mini_tiles = 0;
static MINITILE *mini_tile = NULL;

static MINIT_BMP *minitgfx = NULL;
static int minitgfx_count = 0;

TILE_LOCATION *open_tiles = NULL;
int *playboard = NULL;
int num_open_tiles = 0;
int tiles_in_play = 0;
int tiles_stackp = 0;

int num_tiles = 0;
TILE *tile = NULL;

int feature_id_count;
int *feature_open = NULL;
int *feature_type = NULL;
FEATURE_BONUS *feature_bonus = NULL;
MAP_MEN **feature_men;

BITMAP *emptytile = NULL;
BITMAP *grid = NULL;
static BITMAP *tilerenderbmp = NULL;


/* Count total number of bitmaps in a datafile (including nested dfs) */
static int count_all_bitmaps(const DATAFILE *df)
{
   int n = 0;
   const DATAFILE *dc;

   if (!df)
      return 0;

   for (dc = df; dc->type!=DAT_END; dc++) {
      if (dc->type==DAT_BITMAP)
         n++;
      if (dc->type==DAT_FILE)
         n+=count_all_bitmaps(dc->dat);
   }

   return n;
}

/* Count number of bitmaps in a datafile (don't nest) */
static int count_bitmaps(const DATAFILE *df)
{
   int n = 0;
   const DATAFILE *dc;

   if (!df)
      return 0;

   for (dc = df; dc->type!=DAT_END; dc++) {
      if (dc->type==DAT_BITMAP)
         n++;
   }

   return n;
}

/* Count number of sub datafiles */
static int count_files(const DATAFILE *df)
{
   int n = 0;
   const DATAFILE *dc;

   if (!df)
      return 0;

   for (dc = df; dc->type!=DAT_END; dc++) {
      if (dc->type==DAT_FILE)
         n++;
   }

   return n;
}

static int convert_bitmap_list(const DATAFILE *df, MINIT_BMP *mtb, const int flag)
{
   const DATAFILE *dc;
   int n;

   mtb->ngfx = count_bitmaps(df);
   mtb->bmp = malloc(mtb->ngfx * sizeof *mtb->bmp);
   mtb->conn_flags = flag;

   for (n=0; n<mtb->ngfx; n++) {
      mtb->bmp[n] = NULL;
   }

   n = 0;
   for (dc = df; dc->type!=DAT_END; dc++) {
      if (dc->type==DAT_BITMAP) {
         char s[128];
         snprintf(s, 128, "%s",get_datafile_property(dc, DAT_ID('C', 'K', 'B', 'D')));
         sscanf(s, "%d", &(mtb->checkerboard_pattern));

         mtb->bmp[n] = dc->dat;
         n++;
      }
   }

   return n;
}

static BITMAP* find_minitbitmap(const int flag, const int checkerboard)
{
   int n, c;

   for (n=0; n<minitgfx_count; n++) {
      if (minitgfx[n].conn_flags == flag && (minitgfx[n].checkerboard_pattern&checkerboard)) {
         c = genrandui_range(minitgfx[n].ngfx);
         return minitgfx[n].bmp[c];
      }
   }
   return NULL;
}

/* Flip a bit pattern               */
/* how&4 = Flip vertical            */
/* how&8 = Flip horizontal          */
int flip_pattern(const int pattern, const int how)
{
   int city_pattern = pattern & city_mask;
   int road_pattern = pattern & road_mask;
   int flipped_pattern = 0;

   /* Flip vertical (top <==> down) */
   if (how&4) {
      /* Swap bits 1 and 4 and 2 and 3 of city-like patterns */
      flipped_pattern |= ((city_pattern >> 3) & 0x10101010) | ((city_pattern << 3) & 0x80808080);
      flipped_pattern |= ((city_pattern >> 1) & 0x20202020) | ((city_pattern << 1) & 0x40404040);
      /* Swap bits 1 and 3 of road-like pattern */
      flipped_pattern |= ((road_pattern >> 2) & 0x01010101) | ((road_pattern << 2) & 0x04040404);
      flipped_pattern |= road_pattern & 0x0a0a0a0a;
   }

   /* Flip horizontal (left <==> right) */
   if (how&8) {
      /* Swap bits 1 and 2 and 3 and 4 of city-like pattern */
      flipped_pattern |= ((city_pattern >> 1) & 0x50505050) | ((city_pattern << 1) & 0xa0a0a0a0);
      /* Swap bits 2 and 4 of road-like pattern */
      flipped_pattern |= ((road_pattern >> 2) & 0x02020202) | ((road_pattern << 2) & 0x08080808);
      flipped_pattern |= road_pattern & 0x05050505;
   }

   return flipped_pattern;
}

/* Rotate a bit pattern             */
/* how>0 = left, clockwise          */
/* how<0 = right, anti-clockwise    */
/* Measured in quarters of degrees  */
static int rotate_pattern(const int pattern, const int how)
{
   int shift_left, shift_right;
   int rotated_pattern = 0;
   assert(how<=4);
   assert(how>=-4);

   /* How many bits to shift left */
   shift_left = how;
   if (shift_left<0)
      shift_left+=4;
   /* How many bits to shift to the right */
   shift_right = 4-shift_left;

   /* Rotate cities */
   rotated_pattern |= ((pattern & city_mask)>>shift_right | (pattern & city_mask)<<shift_left) & city_mask;

   /* Rotate roads */
   rotated_pattern |= ((pattern & road_mask)>>shift_right | (pattern & road_mask)<<shift_left) & road_mask;

   return rotated_pattern;
}

static int mt_flags_to_ll(const int pattern, const int how)
{
   int transformed_pattern = pattern;

   if (how&3)
      transformed_pattern = rotate_pattern(transformed_pattern, how);

   if (how&12)
      transformed_pattern = flip_pattern(transformed_pattern, how);

   return transformed_pattern;
}

/* Map minitile graphics onto a large tile */
void map_minitile_graphics(int id)
{
   BITMAP *bmp;
   int x, y;
   int transform;
   int checkerboard;

   /* Draw bitmap */
   for(y=0; y<3; y++) {
      for (x=0; x<3; x++) {
         if (x==1 && y==1) {
            for (transform=0; transform<4; transform++) {
               bmp = find_minitbitmap(mt_flags_to_ll(tile[id].minitile[y][x].connections, transform), CHECKERBOARD_ODD);
               if (bmp)
                  break;
            }
         } else {
            transform = flip_to_ll[y][x];
            checkerboard = (y^x)&1 ? CHECKERBOARD_EVEN : CHECKERBOARD_ODD;
            bmp = find_minitbitmap(mt_flags_to_ll(tile[id].minitile[y][x].connections, transform), checkerboard);
            if (!bmp) {
               for (transform=0; transform<4; transform++) {
                  bmp = find_minitbitmap(mt_flags_to_ll(tile[id].minitile[y][x].connections, transform), checkerboard);
                  if (bmp)
                     break;
               }
            }
         }
         tile[id].minitile[y][x].bmp = bmp;
         tile[id].minitile[y][x].transform = transform;
      }
   }
}

/* Fill a tile with the appropriate FARM id numbers */
static void floodfill_tile_farm(TILE *tile, int x, int y, int id)
{
   int conn;
   if (x<0 || y<0 || x>2 || y>2)
      return;

   conn = tile->minitile[y][x].connections;

   /* Ugly hack to detect odd-shaped tiles */   
   if ((x==2 && y==2 && conn == city_ul_mask) || (x==0 && y==0 && conn == city_lr_mask) ||
      (x==0 && y==2 && conn == city_ur_mask) || (x==2 && y==0 && conn == city_ll_mask))
      return;
      
   if (conn == 0xf0)
      return;
   
   if (conn & road_mask)
      return;
   
   if (tile->minitile[y][x].structnum == 0 && ((conn&city_mask) != 0xf0)) {
      if ((y!=1 || x!=1) || !tile->minitile[y][x].connections) {
         tile->minitile[y][x].structnum = id;

         if ((conn & 0x05050505) == 0) {
            /* Connect left */
            if ((tile->minitile[y][x].conn_dirs[1] & 0x60) != 0x60)
               floodfill_tile_farm(tile, x-1, y, id);
            /* Connect right */
            if ((tile->minitile[y][x].conn_dirs[2] & 0x90) != 0x90)
               floodfill_tile_farm(tile, x+1, y, id);
         }

         if ((conn & 0x0a0a0a0a) == 0) {
            /* Connect top */
            if ((tile->minitile[y][x].conn_dirs[0] & 0xc0) != 0xc0)
               floodfill_tile_farm(tile, x, y-1, id);
            /* Connect bottom */
            if ((tile->minitile[y][x].conn_dirs[3] & 0x30) != 0x30)
               floodfill_tile_farm(tile, x, y+1, id);
         }
      }
   }
}

/* Fill a tile with the appropriate road id numbers */
static void floodfill_tile_road(TILE *tile, int x, int y, int id)
{
   if (x<0 || y<0 || x>2 || y>2)
      return;
      
   if (x==1 && y==1 && (tile->flags & TILEF_MONASTERY)) {
      return;
   }

   if (tile->minitile[y][x].structnum == 0 && tile->minitile[y][x].connections&0x0f) {
      if ((tile->minitile[y][x].connections&road_mask) == 0x0a) {
         /* Horizontal connection */
         tile->minitile[y][x].structnum = id;
         floodfill_tile_road(tile, x-1, y, id);
         floodfill_tile_road(tile, x+1, y, id);
      } else if ((tile->minitile[y][x].connections&road_mask) == 0x05) {
         /* Vertical connection */
         tile->minitile[y][x].structnum = id;
         floodfill_tile_road(tile, x, y-1, id);
         floodfill_tile_road(tile, x, y+1, id);
      } else if ((tile->minitile[y][x].connections&road_mask) == 0x03 ||
          (tile->minitile[y][x].connections&road_mask) == 0x06 ||
          (tile->minitile[y][x].connections&road_mask) == 0x09 ||
          (tile->minitile[y][x].connections&road_mask) == 0x0c) {
         /* Bend */
         tile->minitile[y][x].structnum = id;
         floodfill_tile_road(tile, x, y-1, id);
         floodfill_tile_road(tile, x, y+1, id);
      }
      
      if ( (((x==0 || x==2) && (y==1)) || ((y==0 || y==2) && (x==1))) && !tile->minitile[y][x].structnum)
         tile->minitile[y][x].structnum = id;
   }
}

/* Fill a tile with the appropriate city id numbers */
static void floodfill_tile_city(TILE *tile, int x, int y, int id)
{
   int conn;
   if (x<0 || y<0 || x>2 || y>2)
      return;

   conn = tile->minitile[y][x].connections;

   if (tile->minitile[y][x].structnum == 0 && conn&0xf0) {
      if ( (x+y)&1 || (conn&0xf0) == 0xf0) {
         tile->minitile[y][x].structnum = id;

         if ((tile->minitile[y][x].conn_dirs[0] & 0xc0) == 0xc0)
            floodfill_tile_city(tile, x, y-1, id);
         if ((tile->minitile[y][x].conn_dirs[1] & 0x60) == 0x60)
            floodfill_tile_city(tile, x-1, y, id);
         if ((tile->minitile[y][x].conn_dirs[2] & 0x90) == 0x90)
            floodfill_tile_city(tile, x+1, y, id);
         if ((tile->minitile[y][x].conn_dirs[3] & 0x30) == 0x30)
            floodfill_tile_city(tile, x, y+1, id);
      }
   }
}

/* Construct a large tile from a (subset) of smaller ones */
void construct_tile(int id, int u, int l, int r, int d, int bonus)
{
   int x, y, n, conn;
   /* Create and draw a large tile */

   tile[id].flags = bonus;
   tile[id].angle = 0;

   /* Clear tile list */
   for(y=0; y<3; y++) {
      for (x=0; x<3; x++) {
         tile[id].minitile[y][x].connections = 0;
         tile[id].minitile[y][x].conn_dirs[0] = 0;
         tile[id].minitile[y][x].conn_dirs[1] = 0;
         tile[id].minitile[y][x].conn_dirs[2] = 0;
         tile[id].minitile[y][x].conn_dirs[3] = 0;
         tile[id].minitile[y][x].structnum = 0;
         tile[id].minitile[y][x].flags = 0;
         tile[id].minitile[y][x].man_team = -1;
         tile[id].minitile[y][x].man_type = 0;
      }
   }

   /* Preset mini-tiles */
   tile[id].minitile[0][1].connections = u;
   tile[id].minitile[1][0].connections = l;
   tile[id].minitile[1][2].connections = r;
   tile[id].minitile[2][1].connections = d;
   
   /* set proper partial directions */
   for (n=0; n<4; n++) {
      tile[id].minitile[0][1].conn_dirs[n] = tile[id].minitile[0][1].connections & mid_excity_flags[n];
      tile[id].minitile[1][0].conn_dirs[n] = tile[id].minitile[1][0].connections & mid_excity_flags[n];
      tile[id].minitile[1][2].conn_dirs[n] = tile[id].minitile[1][2].connections & mid_excity_flags[n];
      tile[id].minitile[2][1].conn_dirs[n] = tile[id].minitile[2][1].connections & mid_excity_flags[n];
   }

   /* Calculate other connections */

   /* Centre */
   /* Upper bit 1 (city) becomes bit 4 (city) of centre */
   /* Upper bit 2 (city) becomes bit 3 (city) of centre */
   /* Lower bit 3 (city) becomes bit 2 (city) of centre */
   /* Lower bit 4 (city) becomes bit 1 (city) of centre */
   /* Upper bit 1 (road) becomes bit 3 (road) of centre */
   /* Lower bit 3 (road) becomes bit 1 (road) of centre */
   tile[id].minitile[1][1].conn_dirs[0] |= ((u & 0x10101010)<<3)&city_mask;
   tile[id].minitile[1][1].conn_dirs[0] |= ((u & 0x20202020)<<1)&city_mask;
   tile[id].minitile[1][1].conn_dirs[3] |= ((d & 0x40404040)>>1)&city_mask;
   tile[id].minitile[1][1].conn_dirs[3] |= ((d & 0x80808080)>>3)&city_mask;
   tile[id].minitile[1][1].conn_dirs[0] |= ((u & 0x01010101)<<2)&road_mask;
   tile[id].minitile[1][1].conn_dirs[3] |= ((d & 0x04040404)>>2)&road_mask;
   /* Left bit 1 (city) becomes bit 2 (city) of centre */
   /* Left bit 4 (city) becomes bit 3 (city) of centre */
   /* Right bit 3 (city) becomes bit 4 (city) of centre */
   /* Right bit 2 (city) becomes bit 1 (city) of centre */
   /* Left bit 4 (road) becomes bit 2 (road) of centre */
   /* Right bit 2 (road) becomes bit 4 (road) of centre */
   tile[id].minitile[1][1].conn_dirs[1] |= ((l & 0x10101010)<<1)&city_mask;
   tile[id].minitile[1][1].conn_dirs[1] |= ((l & 0x80808080)>>1)&city_mask;
   tile[id].minitile[1][1].conn_dirs[2] |= ((r & 0x40404040)<<1)&city_mask;
   tile[id].minitile[1][1].conn_dirs[2] |= ((r & 0x20202020)>>1)&city_mask;
   tile[id].minitile[1][1].conn_dirs[1] |= ((l & 0x08080808)>>2)&road_mask;
   tile[id].minitile[1][1].conn_dirs[2] |= ((r & 0x02020202)<<2)&road_mask;

   tile[id].minitile[1][1].connections = 0;
   for(n=0; n<4; n++)
      tile[id].minitile[1][1].connections |= tile[id].minitile[1][1].conn_dirs[n];

   /* Cities through the centre should always have at least two adjacent corners */
   conn = tile[id].minitile[1][1].connections;
   if (is_power_of_two(conn&city_mask) ||
       is_power_of_two( ((conn&city_mask)^((conn&city_mask)<<2)) &city_mask)) {
      tile[id].minitile[1][1].connections&=~city_mask;
   }
   
   /* Corners, no need to consider roads! */
   for (x=0; x<3; x+=2) {
      /* Get upper */
      tile[id].minitile[2][x].conn_dirs[0] |= ((tile[id].minitile[1][x].connections & 0x10101010)<<3)&city_mask;
      tile[id].minitile[2][x].conn_dirs[0] |= ((tile[id].minitile[1][x].connections & 0x20202020)<<1)&city_mask;
      /* Get lower */
      tile[id].minitile[0][x].conn_dirs[3] |= ((tile[id].minitile[1][x].connections & 0x40404040)>>1)&city_mask;
      tile[id].minitile[0][x].conn_dirs[3] |= ((tile[id].minitile[1][x].connections & 0x80808080)>>3)&city_mask;

   }
   for(y=0; y<3; y+=2) {
      /* Get left */
      tile[id].minitile[y][2].conn_dirs[1] |= ((tile[id].minitile[y][1].connections & 0x10101010)<<1)&city_mask;
      tile[id].minitile[y][2].conn_dirs[1] |= ((tile[id].minitile[y][1].connections & 0x80808080)>>1)&city_mask;
      /* Get right */
      tile[id].minitile[y][0].conn_dirs[2] |= ((tile[id].minitile[y][1].connections & 0x40404040)<<1)&city_mask;
      tile[id].minitile[y][0].conn_dirs[2] |= ((tile[id].minitile[y][1].connections & 0x20202020)>>1)&city_mask;
   }

   for(x=0; x<3; x+=2) {
      for(n=0; n<4; n++) {
         tile[id].minitile[2][x].connections |= tile[id].minitile[2][x].conn_dirs[n];
         tile[id].minitile[0][x].connections |= tile[id].minitile[0][x].conn_dirs[n];
         tile[id].minitile[x][2].connections |= tile[id].minitile[x][2].conn_dirs[n];
         tile[id].minitile[x][0].connections |= tile[id].minitile[x][0].conn_dirs[n];
      }
   }

   /* Fill up border tiles */
   if ((tile[id].minitile[0][0].connections & 0x00000090) == 0x00000090)
      tile[id].minitile[0][0].connections |=  0x00000040;
   if ((tile[id].minitile[0][0].connections & 0x00000030) == 0x00000030)
      tile[id].minitile[0][0].connections |=  0x00000040;
   if ((tile[id].minitile[0][2].connections & 0x00000060) == 0x00000060)
      tile[id].minitile[0][2].connections |=  0x00000080;
   if ((tile[id].minitile[0][2].connections & 0x00000030) == 0x00000030)
      tile[id].minitile[0][2].connections |=  0x00000080;

   if ((tile[id].minitile[2][0].connections & 0x00000090) == 0x00000090)
      tile[id].minitile[2][0].connections |=  0x00000020;
   if ((tile[id].minitile[2][0].connections & 0x000000c0) == 0x000000c0)
      tile[id].minitile[2][0].connections |=  0x00000020;
   if ((tile[id].minitile[2][2].connections & 0x00000060) == 0x00000060)
      tile[id].minitile[2][2].connections |=  0x00000010;
   if ((tile[id].minitile[2][2].connections & 0x000000c0) == 0x000000c0)
      tile[id].minitile[2][2].connections |=  0x00000010;

   /* Weird tiles: centre tiles with 1 city bit set should not count it at all */
   for (n=0; n<4; n++) {
      if (is_power_of_two(tile[id].minitile[xypairs[n][1]][xypairs[n][0]].connections&city_mask)) {
         tile[id].minitile[xypairs[n][1]][xypairs[n][0]].connections&=~city_mask;
      }
   }
   
   /* Tiles with an inner `garden' */
   for (n=0; n<4; n+=2) {
      if ((tile[id].minitile[xypairs[n][1]][xypairs[n][0]].connections&city_mask)==0 && 
          (tile[id].minitile[1][1].connections&city_mask)==mid_excity_flags[n]) {
         tile[id].minitile[1][1].connections |= 0x000000f0;
         tile[id].minitile[1][1].conn_dirs[2] |= 0x00000090;
         tile[id].minitile[1][1].conn_dirs[1] |= 0x00000060;
         tile[id].minitile[1][0].connections |= 0x000000f0;
         tile[id].minitile[1][2].connections |= 0x000000f0;
         tile[id].minitile[1][0].conn_dirs[2] |= 0x00000090;
         tile[id].minitile[1][2].conn_dirs[1] |= 0x00000060;
      }
   }

   /* Calculate structure IDs */

   /* Fill farms */
   floodfill_tile_farm(tile+id, 0,1, feature_id_count++);
   floodfill_tile_farm(tile+id, 2,1, feature_id_count++);
   floodfill_tile_farm(tile+id, 1,0, feature_id_count++);
   floodfill_tile_farm(tile+id, 1,2, feature_id_count++);

   /* Fill cities */
   for (n=0; n<4; n++) {
      x = xypairs[n][0];
      y = xypairs[n][1];
      
      if ((tile[id].minitile[y][x].connections&mid_excity_flags[n]) == mid_excity_flags[n])
         tile[id].minitile[y][x].structnum = 0;
   }

   /* City Bridge */
   if (tile[id].flags & TILEF_CBRIDGE) {
      tile[id].minitile[2][1].structnum = tile[id].minitile[0][1].structnum = feature_id_count++;
      tile[id].minitile[1][0].structnum = tile[id].minitile[1][2].structnum = feature_id_count++;
   } else if (tile[id].flags & TILEF_CRBRIDGE) {
      tile[id].minitile[1][0].structnum = tile[id].minitile[1][2].structnum = feature_id_count++;
   } else {
      floodfill_tile_city(tile+id, 0,1, feature_id_count++);
      floodfill_tile_city(tile+id, 2,1, feature_id_count++);
      floodfill_tile_city(tile+id, 1,0, feature_id_count++);
      floodfill_tile_city(tile+id, 1,2, feature_id_count++);
   }

   /* Connect roads (edges) */
   if ((tile[id].minitile[0][1].connections&0x0f) == 0x04)
      tile[id].minitile[0][1].structnum = 0;
   if ((tile[id].minitile[2][1].connections&0x0f) == 0x01)
      tile[id].minitile[2][1].structnum = 0;
   if ((tile[id].minitile[1][0].connections&0x0f) == 0x02)
      tile[id].minitile[1][0].structnum = 0;
   if ((tile[id].minitile[1][2].connections&0x0f) == 0x08)
      tile[id].minitile[1][2].structnum = 0;

   floodfill_tile_road(tile+id, 0,1, feature_id_count++);
   floodfill_tile_road(tile+id, 2,1, feature_id_count++);
   floodfill_tile_road(tile+id, 1,0, feature_id_count++);
   floodfill_tile_road(tile+id, 1,2, feature_id_count++);

   /* Fix farm numbers on corner tiles */
   /* (we actually need a rewrite of this...) */
   if (!(tile[id].flags & TILEF_CBRIDGE)) {
      if (!tile[id].minitile[0][0].structnum && !tile[id].minitile[0][0].conn_dirs[0] && tile[id].minitile[0][0].connections != 0x10)
         tile[id].minitile[0][0].structnum = feature_id_count++;
      if (!tile[id].minitile[2][0].structnum && !tile[id].minitile[2][0].conn_dirs[3] && tile[id].minitile[2][0].connections != 0x20)
         tile[id].minitile[2][0].structnum = feature_id_count++;
      if (!tile[id].minitile[0][2].structnum && !tile[id].minitile[0][2].conn_dirs[0] && tile[id].minitile[0][2].connections != 0x80)
         tile[id].minitile[0][2].structnum = feature_id_count++;
      if (!tile[id].minitile[2][2].structnum && !tile[id].minitile[2][2].conn_dirs[3] && tile[id].minitile[2][2].connections != 0x40)
         tile[id].minitile[2][2].structnum = feature_id_count++;
   
      if (!tile[id].minitile[0][0].structnum && !tile[id].minitile[0][0].conn_dirs[1] && tile[id].minitile[0][0].connections != 0x10)
         tile[id].minitile[0][0].structnum = feature_id_count++;
      if (!tile[id].minitile[2][0].structnum && !tile[id].minitile[2][0].conn_dirs[1] && tile[id].minitile[2][0].connections != 0x20)
         tile[id].minitile[2][0].structnum = feature_id_count++;
      if (!tile[id].minitile[0][2].structnum && !tile[id].minitile[0][2].conn_dirs[2] && tile[id].minitile[0][2].connections != 0x80)
         tile[id].minitile[0][2].structnum = feature_id_count++;
      if (!tile[id].minitile[2][2].structnum && !tile[id].minitile[2][2].conn_dirs[2] && tile[id].minitile[2][2].connections != 0x40)
         tile[id].minitile[2][2].structnum = feature_id_count++;
   }

   /* Connect farms around a bent in the road */
   if ((tile[id].minitile[1][1].connections&road_mask) == 0x03 ||
       (tile[id].minitile[1][1].connections&road_mask) == 0x0c)
       if ((tile[id].minitile[0][0].connections&city_mask) != 0xf0)
         tile[id].minitile[0][0].structnum = tile[id].minitile[2][2].structnum;
   if ((tile[id].minitile[1][1].connections&road_mask) == 0x06 ||
       (tile[id].minitile[1][1].connections&road_mask) == 0x09)
       if ((tile[id].minitile[2][0].connections&city_mask) != 0xf0)
         tile[id].minitile[2][0].structnum = tile[id].minitile[0][2].structnum;

   /* Connect farms around a bent in the river */
   if ((tile[id].minitile[1][1].connections&road_mask) == 0x300 ||
       (tile[id].minitile[1][1].connections&road_mask) == 0xc00)
      tile[id].minitile[0][0].structnum = tile[id].minitile[2][2].structnum;
   if ((tile[id].minitile[1][1].connections&road_mask) == 0x600 ||
       (tile[id].minitile[1][1].connections&road_mask) == 0x900)
      tile[id].minitile[2][0].structnum = tile[id].minitile[0][2].structnum;
      
   /* Bridge */
   if (tile[id].flags & TILEF_BRIDGE) {
      tile[id].minitile[2][1].structnum = tile[id].minitile[0][1].structnum;
      tile[id].minitile[1][0].structnum = tile[id].minitile[1][2].structnum;
   } else if (tile[id].flags & TILEF_CRBRIDGE) {
      tile[id].minitile[2][1].structnum = tile[id].minitile[0][1].structnum;
      tile[id].minitile[1][0].structnum = tile[id].minitile[1][2].structnum;
   }

   /* Ugly hack */
   /* Double bend */
   if (tile[id].flags & TILEF_DBLBEND) {
      if ((tile[id].minitile[2][0].connections & city_mask) ==  (tile[id].minitile[0][2].connections & city_mask))
         tile[id].minitile[2][0].structnum = tile[id].minitile[0][2].structnum;
      
      if ((tile[id].minitile[0][1].connections & city_mask) ==  (tile[id].minitile[1][0].connections & city_mask))
         tile[id].minitile[0][1].structnum = tile[id].minitile[1][0].structnum;
      if ((tile[id].minitile[2][1].connections & city_mask) ==  (tile[id].minitile[1][2].connections & city_mask))
         tile[id].minitile[2][1].structnum = tile[id].minitile[1][2].structnum;
   }

   /* Monastery */
   if (bonus & TILEF_MONASTERY)
      tile[id].minitile[1][1].structnum = feature_id_count++;
}

/* Render a big tile */
void render_tile(int id)
{
   char s[128];
   BITMAP *bmp;
   int x, y, n;
   int transform, conn;
   /* Create and draw a large tile */

   /* Clear tile list */
   for(y=0; y<3; y++) {
      for (x=0; x<3; x++) {
         //bmp = find_minitbitmap(0);
         //blit(bmp, tile[id].gfx, 0,0, x*16,y*16, 16, 16);
         
         bmp = tile[id].minitile[y][x].bmp;
         transform = tile[id].minitile[y][x].transform;

         if (bmp) {
            if (transform&3) {
               rotate_sprite(tile[id].gfx, bmp, x*16,y*16, quarter*(4-(transform&3)));
            } else if (transform&8) {
               draw_sprite_h_flip(tile[id].gfx, bmp, x*16,y*16);
            } else if (transform&4) {
               draw_sprite_v_flip(tile[id].gfx, bmp, x*16,y*16);
            } else {
               draw_sprite(tile[id].gfx, bmp, x*16,y*16);
            }
         }
      }
   }

   if (tile[id].flags & TILEF_CRBRIDGE) {
      snprintf(s, 128, "%08X", tile[id].flags & TILEF_CRBRIDGE);
      bmp = get_theme_gfx(THEME_BONUS_GFX, s);
      if (bmp) {
         draw_sprite(tile[id].gfx, bmp, (TILE_SIZE-bmp->w)/2, (TILE_SIZE-bmp->h)/2);
      }
   }

   if (tile[id].flags & TILEF_CBRIDGE) {
      snprintf(s, 128, "%08X", tile[id].flags & TILEF_CBRIDGE);
      bmp = get_theme_gfx(THEME_BONUS_GFX, s);
      if (bmp) {
         draw_sprite(tile[id].gfx, bmp, (TILE_SIZE-bmp->w)/2, (TILE_SIZE-bmp->h)/2);
      }
   }

   if (tile[id].flags & TILEF_INN) {
      snprintf(s, 128, "%08X", tile[id].flags & TILEF_INN);
      bmp = get_theme_gfx(THEME_BONUS_GFX, s);
      if (bmp) {
         for(y=0; y<3; y++) {
            for (x=0; x<2; x++) {
               if ((tile[id].minitile[y][x+1].connections & 0x0f)) {
                  tile[id].minitile[y][x].flags |= tile[id].flags;
                  draw_sprite(tile[id].gfx, bmp, x*16,y*16);
                  goto break_double_loop1;
               }
            }
         }
      }
   }
break_double_loop1:

   if (tile[id].flags & TILEF_SHIELD) {
      snprintf(s, 128, "%08X", tile[id].flags & TILEF_SHIELD);
      bmp = get_theme_gfx(THEME_BONUS_GFX, s);
      if (bmp) {
         for(y=0; y<3; y++) {
            for (x=0; x<3; x++) {
               if ((tile[id].minitile[y][x].connections & 0xf0) == 0xf0) {
                  draw_sprite(tile[id].gfx, bmp, x*16,y*16);
                  goto break_double_loop;
               }
            }
         }
      }
   }
break_double_loop:

   if (tile[id].flags & TILEF_TRADEANY) {
      bmp = get_goods_bitmap((tile[id].flags & TILEF_TRADEANY)>>8);
      for(y=0; y<3; y++) {
         for (x=0; x<3; x++) {
            if ((tile[id].minitile[y][x].connections & 0xf0) == 0xf0) {
               draw_sprite(tile[id].gfx, get_goods_bitmap(0), x*16,y*16);
               draw_sprite(tile[id].gfx, bmp, x*16,y*16);
               goto break_double_loop2;
            }
         }
      }
   }
break_double_loop2:

   if (tile[id].flags & (TILEF_VOLCANOE|TILEF_PORTAL)) {
      snprintf(s, 128, "%08X", tile[id].flags & (TILEF_VOLCANOE|TILEF_PORTAL));
      bmp = get_theme_gfx(THEME_BONUS_GFX, s);
      if (bmp) {
         conn = tile[id].minitile[1][1].connections;
         if (conn==0 || is_power_of_two(conn)) {
            draw_sprite(tile[id].gfx, bmp, 16,16);
         } else {
            /* First try to find an empty tile */
            for(y=0; y<3; y++) {
               for (x=0; x<3; x++) {
                  if (tile[id].minitile[y][x].connections == 0x00) {
                     draw_sprite(tile[id].gfx, bmp, x*16,y*16);
                     goto break_double_loop3;
                  }
               }
            }
            /* Not found, try one with no city on it at least */
            for(y=0; y<3; y++) {
               for (x=0; x<3; x++) {
                  if ((tile[id].minitile[y][x].connections & 0xf0) == 0x00) {
                     draw_sprite(tile[id].gfx, bmp, x*16,y*16);
                     goto break_double_loop3;
                  }
               }
            }
         }
      }
   }
break_double_loop3:

   if (tile[id].flags & TILEF_DRAGON) {
      snprintf(s, 128, "%08X", tile[id].flags & TILEF_DRAGON);
      bmp = get_theme_gfx(THEME_BONUS_GFX, s);
      if (bmp) {
         for(y=0; y<3; y++) {
            for (x=2; x>=0; x--) {
               if ((tile[id].minitile[y][x].connections & 0xf0) != 0xf0) {
                  draw_sprite(tile[id].gfx, bmp, x*16,y*16);
                  goto break_double_loop4;
               }
            }
         }
      }
   }
break_double_loop4:

   /* Dont draw princess on top of shield */
   if (tile[id].flags & TILEF_PRINCESS) {
      if (tile[id].flags & TILEF_SHIELD)
         n = 1;
      else
         n = 0;

      snprintf(s, 128, "%08X", tile[id].flags & TILEF_PRINCESS);
      bmp = get_theme_gfx(THEME_BONUS_GFX, s);
      if (bmp) {
         for(y=0; y<3; y++) {
            for (x=0; x<3; x++) {
               if ((tile[id].minitile[y][x].connections & 0xf0) == 0xf0) {
                  if (n) {
                     n--;
                  } else {
                     draw_sprite(tile[id].gfx, bmp, x*16,y*16);
                     goto break_double_loop5;
                  }
               }
            }
         }
      }
   }
break_double_loop5:

   if (tile[id].flags & TILEF_MONASTERY || tile[id].flags & TILEF_CATHEDRAL) {
      snprintf(s, 128, "%08X", tile[id].flags & (TILEF_MONASTERY|TILEF_CATHEDRAL));
      bmp = get_theme_gfx(THEME_BONUS_GFX, s);
      if (bmp) {
         draw_sprite(tile[id].gfx, bmp, (TILE_SIZE-bmp->w)/2, (TILE_SIZE-bmp->h)/2);
      }
   }

   if (tile[id].flags & TILEF_BRIDGE && (tile[id].minitile[1][1].connections&road_mask) == 0x0f) {
      snprintf(s, 128, "%08X", tile[id].flags & TILEF_BRIDGE);
      bmp = get_theme_gfx(THEME_BONUS_GFX, s);
      if (bmp) {
         draw_sprite(tile[id].gfx, bmp, (TILE_SIZE-bmp->w)/2, (TILE_SIZE-bmp->h)/2);
      }
   }

   if (tile[id].flags & TILEF_DBLBEND && (tile[id].minitile[1][1].connections&road_mask) == 0x0f) {
      snprintf(s, 128, "%08X", tile[id].flags & TILEF_DBLBEND);
      bmp = get_theme_gfx(THEME_BONUS_GFX, s);
      if (bmp) {
         draw_sprite(tile[id].gfx, bmp, (TILE_SIZE-bmp->w)/2, (TILE_SIZE-bmp->h)/2);
      }
   }

   if (tile[id].flags & TILEF_SIEGE) {
      snprintf(s, 128, "%08X", tile[id].flags & TILEF_SIEGE);
      bmp = get_theme_gfx(THEME_BONUS_GFX, s);
      if (bmp) {
         draw_sprite(tile[id].gfx, bmp, (TILE_SIZE-bmp->w)/2, (TILE_SIZE-bmp->h)/2);
      }
   }
}

/* Rotate a big tile 270 deg anti-clockwise */
void rotate_big_tile_270(int id)
{
   MINITILE minitile[3][3];
   BITMAP *bmp;
   int x, y;
   int new_x, new_y;

   for (y=0; y<3; y++) {
      for (x=0; x<3; x++) {
         minitile[y][x] = tile[id].minitile[y][x];
      }
   }

   for (y=0; y<3; y++) {
      for (x=0; x<3; x++) {
         new_x = 1+(x-1)*rot270[0][0]+(y-1)*rot270[0][1];
         new_y = 1+(x-1)*rot270[1][0]+(y-1)*rot270[1][1];

         tile[id].minitile[new_y][new_x].connections =
            rotate_pattern(minitile[y][x].connections, 1);

         tile[id].minitile[new_y][new_x].structnum = minitile[y][x].structnum;
      }
   }

   bmp = tilerenderbmp;
   rotate_sprite(bmp, tile[id].gfx, 0,0, quarter);
   tilerenderbmp = tile[id].gfx;
   tile[id].gfx = bmp;

   /* Re-render tile */
   tile[id].angle = (tile[id].angle+270)%360;
}

/* Rotate a big tile 90 deg anti-clockwise */
void rotate_big_tile_90(int id)
{
   rotate_big_tile_270(id);
   rotate_big_tile_270(id);
   rotate_big_tile_270(id);
}

/* Rotate a big tile 180 deg (anti-clockwise, but doesn't matter) */
void rotate_big_tile_180(int id)
{
   rotate_big_tile_270(id);
   rotate_big_tile_270(id);
}

/* Initialize a loaded tileset */
void initialize_tileset(void)
{
   int c, n;
   int tiles_constructed;

   /* Get number of tiletypes */

   tiles_in_play = 0;
   tiles_stackp = 0;
   num_tiles = 0;
   feature_id_count = 1;

   /* Get actual number of tiles in play */
   for (c=0; c<num_tile_types && game_tileset[c].amount; c++) {
      num_tiles += game_tileset[c].amount;
   }

   for (c=0; c<num_start_tile_types && start_tiles[c].amount; c++) {
      num_tiles += start_tiles[c].amount;
   }
   
   /* Allocate memory for tileset */
   tile = realloc(tile, num_tiles * sizeof *tile);
   playboard = malloc(num_tiles * sizeof *playboard);
   for (n=0; n<num_tiles; n++) {
      tile[n].gfx = create_bitmap(48, 48);
      playboard[n] = -1;
   }

   open_tiles = malloc((2*num_tiles+2) * sizeof *open_tiles);
   num_open_tiles = 0;

   tilerenderbmp = create_bitmap(48, 48);

   /* Construct tiles */
   tiles_constructed = 0;
   
   /* Start tiles */
   for (c=0; c<num_start_tiles && start_tiles[c].amount; c++) {
      for (n=0; n<start_tiles[c].amount; n++) {

         construct_tile(tiles_constructed, start_tiles[c].up,
                           start_tiles[c].left, start_tiles[c].right,
                              start_tiles[c].down, start_tiles[c].flags);
         map_minitile_graphics(tiles_constructed);
         render_tile(tiles_constructed);

         tiles_constructed++;
      }
   }

   /* Regular tiles */
   for (c=0; c<num_tile_types && game_tileset[c].amount; c++) {
      for (n=0; n<game_tileset[c].amount; n++) {

         construct_tile(tiles_constructed, game_tileset[c].up,
                           game_tileset[c].left, game_tileset[c].right,
                              game_tileset[c].down, game_tileset[c].flags);
         map_minitile_graphics(tiles_constructed);
         render_tile(tiles_constructed);

         tiles_constructed++;
      }
   }
}

void initialize_tiles(void)
{
   int n, c;
   DATAFILE *dc;
   char s[128];

   if (!theme_loaded()) {
      set_gfx_mode(GFX_TEXT, 0,0, 0,0);
      allegro_message("No theme loaded!\n");
      abort();
   }

   /* Initialize mini-tiles */
   /* Convert list */
   /* Allocate memory and initialize data structure */
   minitgfx_count = count_files(odd_tilegfx_dat) + count_files(even_tilegfx_dat) + count_files(common_tilegfx_dat);
   minitgfx = malloc(minitgfx_count * sizeof *minitgfx);
   for (n=0; n<minitgfx_count; n++) {
      minitgfx[n].ngfx = 0;
      minitgfx[n].conn_flags = 0;
      minitgfx[n].checkerboard_pattern = 0;
      minitgfx[n].bmp = NULL;
   }
   
   /* Convert bitmaps */
   n = 0;
   
   /* Load odd-checkerboard minitiles */
   for (dc = odd_tilegfx_dat; dc->type!=DAT_END; dc++) {
      if (dc->type==DAT_FILE) {
         snprintf(s, 128, "%s",get_datafile_property(dc, DAT_ID('N', 'A', 'M', 'E')));
         sscanf(s, "%x_DAT", &c);
         
         minitgfx[n].checkerboard_pattern = CHECKERBOARD_ODD;
         
         convert_bitmap_list(dc->dat, minitgfx+n, c);
         n++;
      }
   }

   /* Load even-checkerboard minitiles */
   for (dc = even_tilegfx_dat; dc->type!=DAT_END; dc++) {
      if (dc->type==DAT_FILE) {
         snprintf(s, 128, "%s",get_datafile_property(dc, DAT_ID('N', 'A', 'M', 'E')));
         sscanf(s, "%x_DAT", &c);
         //printf("%s: %d, %d[%d]\n", s, c, n, minitgfx_count);
         
         minitgfx[n].checkerboard_pattern = CHECKERBOARD_EVEN;
         
         convert_bitmap_list(dc->dat, minitgfx+n, c);
         n++;
      }
   }

   /* Load common (both odd and even) checkerboard minitiles */
   for (dc = common_tilegfx_dat; dc->type!=DAT_END; dc++) {
      if (dc->type==DAT_FILE) {
         snprintf(s, 128, "%s",get_datafile_property(dc, DAT_ID('N', 'A', 'M', 'E')));
         sscanf(s, "%x_DAT", &c);
         //printf("%s: %d, %d[%d]\n", s, c, n, minitgfx_count);
         
         minitgfx[n].checkerboard_pattern = CHECKERBOARD_ANY;
         
         convert_bitmap_list(dc->dat, minitgfx+n, c);
         n++;
      }
   }


   emptytile = get_theme_gfx(THEME_UI_GFX, "TILE");
   grid = get_theme_gfx(THEME_UI_GFX, "GRID");
}

/* Set bonus count */
static void initialize_feature(int id)
{
   TILE *t;
   MINITILE *mt;
   int n, x, y;
   int inn_flag, shield_flag, trade_flag, cathedral_flag;

   feature_open[id] = 0;
   feature_type[id] = FEATURE_NONE;

   feature_bonus[id].cathedrals = 0;
   feature_bonus[id].shields = 0;
   feature_bonus[id].inns = 0;
   feature_bonus[id].siege = 0;
   for (n=0; n<3; n++)
      feature_bonus[id].goods[n] = 0;
   
   for (n=0; n<num_tiles; n++) {
      inn_flag = 0;
      shield_flag = 0;
      trade_flag = 0;
      cathedral_flag = 0;
      t = &(tile[n]);
            
      for(y=0; y<3; y++) {
         for(x=0; x<3; x++) {
            mt = &(t->minitile[y][x]);
            
            /* Don't count bonuses that are already taken */
            /* Inn already taken... */
            if (t->flags&TILEF_INN && feature_bonus[mt->structnum].inns) {
               inn_flag = 1;
            }
            
            /* Shield already taken... */
            if (t->flags&TILEF_SHIELD && feature_bonus[mt->structnum].shields) {
               shield_flag = 1;
            }
            
            /* Tradegoods already taken... */
            if (t->flags&TILEF_TRADEANY && feature_bonus[mt->structnum].goods[((t->flags & TILEF_TRADEANY)>>8) - 1]) {
               trade_flag = 1;
            }
            
            if (mt->structnum==id) {
               /* Check shields */
               /* Make sure we don't accidentally count the shield twice */
               if (t->flags & TILEF_SHIELD && !shield_flag && (mt->connections&city_mask) == 0xf0) {
                  feature_bonus[id].shields++;
                  shield_flag = 1;
               }
               
               /* Town under siege */
               if (t->flags & TILEF_SIEGE && (mt->connections&city_mask)) {
                  feature_bonus[id].siege = 1;
               }
               
               /* Check tradegoods */
               /* Make sure we don't accidentally count the goods twice */
               if (t->flags & TILEF_TRADEANY && !trade_flag && (mt->connections&0xf0) == 0xf0) {
                  feature_bonus[id].goods[((t->flags & TILEF_TRADEANY)>>8) - 1]++;
                  trade_flag = 1;
               }

               /* Check cathedrals */
               /* Make sure we don't accidentally count it twice */
               if (t->flags & TILEF_CATHEDRAL && !cathedral_flag) {
                  feature_bonus[id].cathedrals++;
                  cathedral_flag = 1;
               }
               
               /* Check inns */
               /* Make sure we don't accidentally count it twice */
               if (t->flags&TILEF_INN && !inn_flag && (mt->connections&0x0f)) {
                  feature_bonus[id].inns++;
                  inn_flag = 1;
               }
            }
         }
      }
   }
}

/* Shuffle tiles and place starting tile */
void initialize_gameboard(void)
{
   int n, c;
   int x, y;
   TILE t;
   
   /* Shuffle starting tiles */
   /* That is, all but the first and last starting tile */
   for (n=1; n<num_start_tiles-1; n++) {
      c = genrandui_range(num_start_tiles-2)+1;

      /* Swap tiles */
      t = tile[c];
      tile[c] = tile[n];
      tile[n] = t;
   }
   
   /* Shuffle pile */
   for (n=num_start_tiles; n<num_tiles; n++) {
      c = num_start_tiles+genrandui_range(num_tiles-num_start_tiles);

      /* Swap tiles */
      t = tile[c];
      tile[c] = tile[n];
      tile[n] = t;
   }

   /* Initialize all feature closure code */
   feature_open = calloc(1, feature_id_count * sizeof *feature_open);
   feature_type = calloc(1, feature_id_count * sizeof *feature_type);
   feature_bonus = calloc(1, feature_id_count * sizeof *feature_bonus);
   for(n=0; n<feature_id_count; n++) {
      initialize_feature(n);
   }

   /* Place starting tile at the front of the list and count feature openings */
   for (n=0; n<num_tiles; n++) {
      if (tile[n].minitile[0][1].connections == start_tiles[0].up &&
          tile[n].minitile[2][1].connections == start_tiles[0].down &&
          tile[n].minitile[1][0].connections == start_tiles[0].left &&
          tile[n].minitile[1][2].connections == start_tiles[0].right) {
         t = tile[0];
         tile[0] = tile[n];
         tile[n] = t;
         break;
      }
   }

   /* Count remaining openings for each feature */
   for (n=0; n<num_tiles; n++) {
      for (c=0; c<4; c++) {
         x = xypairs[c][0];
         y = xypairs[c][1];
         if (tile[n].minitile[y][x].connections && y!=x)
            feature_open[tile[n].minitile[y][x].structnum]++;
      }
   }

   /* Initialize datastructure to keep track of peasants in each structure */
   feature_men = malloc(feature_id_count * sizeof *feature_men);
   for(c=0; c<feature_id_count; c++) {
      feature_men[c] = malloc(num_teams * sizeof **feature_men);
      for (n=0; n<num_teams; n++) {
         feature_men[c][n].men[PEASANTS] = 0;
         feature_men[c][n].men[BIGMEN] = 0;
         feature_men[c][n].men[MASTERBUILDERS] = 0;
         feature_men[c][n].men[PIGS] = 0;
      }
   }

   place_tile(pop_tile(),  MAP_CX, MAP_CY);
}

void destroy_tiles(void)
{
   int n;

   /* Destroy playingfield datastructure */
   free(playboard);
   playboard = NULL;
   free(open_tiles);
   open_tiles = NULL;

   num_open_tiles = 0;
   tiles_in_play = 0;
   tiles_stackp = 0;

   free(feature_open);
   free(feature_type);
   free(feature_bonus);
   
   feature_open = NULL;
   feature_type = NULL;
   feature_bonus = NULL;

   /* Unload minitile graphics */
   for (n=0; n<minitgfx_count; n++) {
      free(minitgfx[n].bmp);
   }
   free(minitgfx);
   minitgfx_count = 0;

   for (n=0; n<num_tiles; n++) {
      destroy_bitmap(tile[n].gfx);
   }

   free (tile);
   tile = NULL;

   destroy_bitmap(tilerenderbmp);
   tilerenderbmp = NULL;

   num_tiles = 0;

   free (mini_tile);
   mini_tile = NULL;

   num_mini_tiles = 0;

   for(n=0; n<feature_id_count; n++)
      free(feature_men[n]);
   free(feature_men);
   feature_men = NULL;

}

BITMAP *get_tile_bitmap(int id)
{
   assert(id<num_tiles);
   return tile[id].gfx;
}

/* Pop a tile off the pile. Returns NOTILE if there are no more tiles waiting */
int pop_tile(void)
{
   if (tiles_stackp==num_tiles)
      return NOTILE;
   else
      return tiles_stackp++;
}

int tiles_remaining(void)
{
   return num_tiles - tiles_stackp;
}

/* Does a binary search in the list of tiles placed on the map. */
/*  The code could use an actual hashing algorithm */
int find_tile(int x, int y)
{
   int n;

   for (n=0; n<tiles_in_play && tile[playboard[n]].y <= y; n++)
      if (tile[playboard[n]].x == x && tile[playboard[n]].y == y) {
         return playboard[n];
      }

   return NOTILE;
#if 0
   int u, l, n;

   u = tiles_in_play-1;
   l = 0;
   n = (u+l)/2;

   while (u>=l) {
      n = (u+l)/2;
      if (tile[playboard[n]].y < y) {
         /* Lower half */
         l = n+1;
      } else if (tile[playboard[n]].y > y) {
         /* Lower half */
         u = n-1;
      } else {
         if (tile[playboard[n]].x < x) {
            /* Lower half */
            l = n+1;
         } else if (tile[playboard[n]].x > x) {
            /* Lower half */
            u = n-1;
         } else {
            return playboard[n];
         }
      }
   }

   if (tile[playboard[n]].x == x && tile[playboard[n]].y == y) {
      return playboard[n];
   }

   if (tile[playboard[u]].x == x && tile[playboard[u]].y == y) {
      return playboard[u];
   }

   if (tile[playboard[l]].x == x && tile[playboard[l]].y == y) {
      return playboard[l];
   }


   return NOTILE;
#endif
}

/********************************************/
/*  Dealing with the set of open tiles      */
/********************************************/

/* Does a binary search in the list of open board locations */
static inline int find_open_tile(int x, int y)
{
   int u, l, n;

   u = num_open_tiles-1;
   l = 0;
   n = (u-l)/2;

   while (u>=l) {
      n = (u+l)/2;
      if (open_tiles[n].y < y) {
         /* Lower half */
         l = n+1;
      } else if (open_tiles[n].y > y) {
         /* Lower half */
         u = n-1;
      } else {
         if (open_tiles[n].x < x) {
            /* Lower half */
            l = n+1;
         } else if (open_tiles[n].x > x) {
            /* Lower half */
            u = n-1;
         } else {
            return n;
         }
      }
   }

   if (open_tiles[n].x == x && open_tiles[n].y == y) {
      return n;
   }

   return -1;
}

/* Sort the list of open tiles */
static void sort_opentile_list(void)
{
   /* Tile sorter callback. Sort first by y, then by x */
   static int tile_sort_callback(const void *t1, const void *t2)
   {
      const TILE_LOCATION *tn1 = (const TILE_LOCATION *)(t1);
      const TILE_LOCATION *tn2 = (const TILE_LOCATION *)(t2);
      if (tn1->y > tn2->y)
         return 1;
      if (tn1->y < tn2->y)
         return -1;

      return tn1->x > tn2->x;
   }
   qsort(open_tiles, num_open_tiles, sizeof *open_tiles, tile_sort_callback);
}

void mark_open_tile(int x, int y)
{
   /* Don't insert tiles that are already in the list */
   if (find_open_tile(x, y)==-1 && find_tile(x, y)==NOTILE) {
      open_tiles[num_open_tiles].x = x;
      open_tiles[num_open_tiles].y = y;
      num_open_tiles++;
      sort_opentile_list();
   }
}

void remove_open_tile(int x, int y)
{
   int c;
   c = find_open_tile(x, y);
   if (c!=-1) {
      num_open_tiles--;
      open_tiles[c].x = open_tiles[num_open_tiles].x;
      open_tiles[c].y = open_tiles[num_open_tiles].y;
      sort_opentile_list();
   }
}

/* Connect features */
static void replace_feature_id(int old_id, int new_id)
{
   int n, c;
   int x, y;

   if (!old_id || !new_id)
      return;

   /* Calculate new number of open sides                                    */
   /* This is the sum of the previous two, minus the sides that now connect */
   /* This must be calculated even if old_id==new_id, because two sides of  */
   /*  the placed tilemay beconnected `the long way round' as well as on    */
   /*  the tile itself */
   if (old_id == new_id) {
      feature_open[new_id]-=2;

      /* Don't bother with the rest, as it wouldn't change anything and only */
      /*  slows things down. */
      return;
   }
   
   feature_open[new_id]+=feature_open[old_id]-2;
   feature_type[old_id] = FEATURE_NONE;
   
   /* Join bonuses */
   for (n=0; n<3; n++)
      feature_bonus[new_id].goods[n] += feature_bonus[old_id].goods[n];

   feature_bonus[new_id].shields += feature_bonus[old_id].shields;
   feature_bonus[new_id].cathedrals += feature_bonus[old_id].cathedrals;
   feature_bonus[new_id].inns += feature_bonus[old_id].inns;
   feature_bonus[new_id].siege |= feature_bonus[old_id].siege;

   for (n=0; n<3; n++)
      feature_bonus[old_id].goods[n] = 0;

   feature_bonus[old_id].shields = 0;
   feature_bonus[old_id].cathedrals = 0;
   feature_bonus[old_id].inns = 0;
   feature_bonus[old_id].siege = 0;

   /* Join men */
   for (n=0; n<num_teams; n++)
      for (c=0; c<MENTYPES; c++)
         feature_men[new_id][n].men[c] += feature_men[old_id][n].men[c];

   /* Paint tiles */
   for (n=0; n<tiles_in_play; n++) {
      for(y=0; y<3; y++) {
         for(x=0; x<3; x++) {
            if (tile[playboard[n]].minitile[y][x].structnum==old_id)
               tile[playboard[n]].minitile[y][x].structnum = new_id;
         }
      }
   }
}

void connect_tile(int x, int y)
{
   int id, upper, lower, left, right;

   id = find_tile(x, y);
   upper = find_tile(x, y-1);
   lower = find_tile(x, y+1);
   left = find_tile(x-1, y);
   right = find_tile(x+1, y);

   if (id == NOTILE)
      return;

   //printf ("%d\t%d\t%d\t%d\t%d\n", id, upper, lower, left, right);

   if (right != NOTILE) {
      replace_feature_id(tile[right].minitile[1][0].structnum, tile[id].minitile[1][2].structnum);
      if ((tile[right].minitile[1][0].connections&city_mask) == 0) {
         replace_feature_id(tile[right].minitile[0][0].structnum, tile[id].minitile[0][2].structnum);
         replace_feature_id(tile[right].minitile[2][0].structnum, tile[id].minitile[2][2].structnum);
      }
   }

   if (left != NOTILE) {
      replace_feature_id(tile[left].minitile[1][2].structnum, tile[id].minitile[1][0].structnum);
      if ((tile[left].minitile[1][2].connections&city_mask) == 0) {
         replace_feature_id(tile[left].minitile[0][2].structnum, tile[id].minitile[0][0].structnum);
         replace_feature_id(tile[left].minitile[2][2].structnum, tile[id].minitile[2][0].structnum);
      }
   }

   if (upper != NOTILE) {
      replace_feature_id(tile[upper].minitile[2][1].structnum, tile[id].minitile[0][1].structnum);
      if ((tile[upper].minitile[2][1].connections&city_mask) == 0) {
         replace_feature_id(tile[upper].minitile[2][0].structnum, tile[id].minitile[0][0].structnum);
         replace_feature_id(tile[upper].minitile[2][2].structnum, tile[id].minitile[0][2].structnum);
      }
   }

   if (lower != NOTILE) {
      replace_feature_id(tile[lower].minitile[0][1].structnum, tile[id].minitile[2][1].structnum);
      if ((tile[lower].minitile[0][1].connections&city_mask) == 0) {
         replace_feature_id(tile[lower].minitile[0][0].structnum, tile[id].minitile[2][0].structnum);
         replace_feature_id(tile[lower].minitile[0][2].structnum, tile[id].minitile[2][2].structnum);
      }
   }
}

/* Place a tile on the game board */
void place_tile(int id, int map_x, int map_y)
{
   int x, y;
   
   /* Tile sorter callback. Sort first by y, then by x */
   static int tile_sort_callback(const void *t1, const void *t2)
   {
      const int tn1 = *(const int *)(t1);
      const int tn2 = *(const int *)(t2);
      if (tile[tn1].y > tile[tn2].y)
         return 1;
      if (tile[tn1].y < tile[tn2].y)
         return -1;

      return tile[tn1].x > tile[tn2].x;
   }
   int c, num, conn;
   
   /* This can be done more efficient... */
   playboard[tiles_in_play] = id;
   tile[playboard[tiles_in_play]].x = map_x;
   tile[playboard[tiles_in_play]].y = map_y;
   tiles_in_play++;

   /* Update the list of open tiles */
   remove_open_tile(map_x, map_y);
   mark_open_tile(map_x+1, map_y);
   mark_open_tile(map_x-1, map_y);
   mark_open_tile(map_x, map_y+1);
   mark_open_tile(map_x, map_y-1);

   /* Sort the in-game tiles */
   qsort(playboard, tiles_in_play, sizeof *playboard, tile_sort_callback);

   /* Record the type of features on this tile */
   for (c=0; c<4; c++) {
      y = xypairs[c][1];
      x = xypairs[c][0];
      num = tile[id].minitile[y][x].structnum;
      conn = tile[id].minitile[y][x].connections;
      if (conn==0)
         feature_type[num] = FEATURE_FARM;
      else if ((conn&mid_excity_flags[c]) == mid_excity_flags[c])
         feature_type[num] = FEATURE_CITY;
      else if (conn&road_mask)
         feature_type[num] = FEATURE_ROAD;
   }
   if (tile[id].flags & TILEF_MONASTERY) {
      num = tile[id].minitile[1][1].structnum;
      feature_type[num] = FEATURE_MONASTERY;
   } else if (tile[id].minitile[1][1].connections==0) {
      num = tile[id].minitile[1][1].structnum;
      feature_type[num] = FEATURE_FARM;
   }

   /* Properly assign FARM numbers to corner tiles */
   for (c=0; c<4; c++) {
      y = xycorners[c][1];
      x = xycorners[c][0];

      num = tile[id].minitile[y][x].structnum;

      if (feature_type[num] != FEATURE_CITY && feature_type[num] != FEATURE_ROAD)
         feature_type[num] = FEATURE_FARM;

      if ((tile[id].minitile[y][x].connections&city_mask) == 0xf0)
         feature_type[num] = FEATURE_CITY;         
   }
}

/* Helper function: Returns TRUE if a rivertile will fit */
static int rivertile_fits(const int id, const int upper, const int lower, const int left, const int right)
{
   int pattern;
   int flag;

   /* River must connect, but a river tile may not be adjacent to more than */
   /*  one horizontal or vertical river tile when placed */
   
   if (right!=NOTILE) {
      pattern = tile[right].minitile[1][0].connections & 0x62626262;
      flag = flip_pattern(tile[id].minitile[1][2].connections, 8) & 0x62626262;

      if (pattern!=flag)
         return 0;
         
      pattern = tile[right].minitile[1][0].connections & 0x00000200;
      
      if (!pattern)
         return 0;
         
      /* Check if placing the tile here will force the next river tile to be */
      /*  adjacent to more than one tile */
      /* (Check only needed if current tile is a bend) */
      if ((tile[id].minitile[1][1].connections & 0x00000500) && (tile[id].minitile[1][1].connections & 0x00000a00)) {
         if ((tile[right].minitile[2][1].connections & 0x00000500) && (tile[id].minitile[2][1].connections & 0x00000500))
            return 0;
         if ((tile[right].minitile[0][1].connections & 0x00000500) && (tile[id].minitile[0][1].connections & 0x00000500))
            return 0;
      }
         
      return 1;
   }

   if (left!=NOTILE) {
      pattern = tile[left].minitile[1][2].connections & 0x98989898;
      flag = flip_pattern(tile[id].minitile[1][0].connections, 8) & 0x98989898;

      if (pattern!=flag)
         return 0;

      pattern = tile[left].minitile[1][2].connections & 0x00000800;

      if (!pattern)
         return 0;

      /* Check if placing the tile here will force the next river tile to be */
      /*  adjacent to more than one tile */
      /* (Check only needed if current tile is a bend) */
      if ((tile[id].minitile[1][1].connections & 0x00000500) && (tile[id].minitile[1][1].connections & 0x00000a00)) {
         if ((tile[left].minitile[2][1].connections & 0x00000500) && (tile[id].minitile[2][1].connections & 0x00000500))
            return 0;
         if ((tile[left].minitile[0][1].connections & 0x00000500) && (tile[id].minitile[0][1].connections & 0x00000500))
            return 0;
      }

      return 1;
   }

   if (upper!=NOTILE) {
      pattern = tile[upper].minitile[2][1].connections & 0x31313131;
      flag = flip_pattern(tile[id].minitile[0][1].connections, 4) & 0x31313131;

      if (pattern!=flag)
         return 0;

      pattern = tile[upper].minitile[2][1].connections & 0x00000100;

      if (!pattern)
         return 0;

      /* Check if placing the tile here will force the next river tile to be */
      /*  adjacent to more than one tile */
      /* (Check only needed if current tile is a bend) */
      if ((tile[id].minitile[1][1].connections & 0x00000500) && (tile[id].minitile[1][1].connections & 0x00000a00)) {
         if ((tile[upper].minitile[1][2].connections & 0x00000a00) && (tile[id].minitile[1][2].connections & 0x00000a00))
            return 0;
         if ((tile[upper].minitile[1][0].connections & 0x00000a00) && (tile[id].minitile[1][0].connections & 0x00000a00))
            return 0;
      }

      return 1;
   }

   if (lower!=NOTILE) {
      pattern = tile[lower].minitile[0][1].connections & 0xc4c4c4c4;
      flag = flip_pattern(tile[id].minitile[2][1].connections, 4) & 0xc4c4c4c4;

      if (pattern!=flag)
         return 0;

      pattern = tile[lower].minitile[0][1].connections & 0x00000400;

      if (!pattern)
         return 0;

      /* Check if placing the tile here will force the next river tile to be */
      /*  adjacent to more than one tile */
      /* (Check only needed if current tile is a bend) */
      if ((tile[id].minitile[1][1].connections & 0x00000500) && (tile[id].minitile[1][1].connections & 0x00000a00)) {
         if ((tile[lower].minitile[1][0].connections & 0x00000a00) && (tile[id].minitile[1][0].connections & 0x00000a00))
            return 0;
         if ((tile[lower].minitile[1][2].connections & 0x00000a00) && (tile[id].minitile[1][2].connections & 0x00000a00))
            return 0;
      }

      return 1;
   }
   
   return 0;
}

/* Returns TRUE if a tile will fit *in its current orientation* */
int tile_fits(const int id, const int x, const int y)
{
   int upper, lower, left, right;
   int pattern;
   int flag;

   if (id == NOTILE)
      return 0;

   if (find_tile(x, y) != NOTILE)
      return 0;

   upper = find_tile(x, y-1);
   lower = find_tile(x, y+1);
   left = find_tile(x-1, y);
   right = find_tile(x+1, y);

   if (upper==lower && left==right) {
      /* Only possible if all are -1, as tile IDs are unique */
      /* Tiles must be placed adjacent to each other */
      return 0;
   }

   /* The river must be placed continuously */
   if (tile[id].minitile[1][1].connections & river_mask)
      return rivertile_fits(id, upper, lower, left, right);

   /* See if it will fit on the right */
   if (right!=NOTILE) {
      pattern = tile[right].minitile[1][0].connections & 0x62626262;
      flag = flip_pattern(tile[id].minitile[1][2].connections, 8) & 0x62626262;

      /*
      printf("%02x, %02x (%02x)\n", tile[right].minitile[1][0].connections,
                                    flip_pattern(tile[id].minitile[1][2].connections, 8),
                                    tile[id].minitile[1][2].connections);
      */

      if (pattern!=flag)
         return 0;
   }

   if (left!=NOTILE) {
      pattern = tile[left].minitile[1][2].connections & 0x98989898;
      flag = flip_pattern(tile[id].minitile[1][0].connections, 8) & 0x98989898;

      if (pattern!=flag)
         return 0;
   }

   if (upper!=NOTILE) {
      pattern = tile[upper].minitile[2][1].connections & 0x31313131;
      flag = flip_pattern(tile[id].minitile[0][1].connections, 4) & 0x31313131;

      if (pattern!=flag)
         return 0;
   }

   if (lower!=NOTILE) {
      pattern = tile[lower].minitile[0][1].connections & 0xc4c4c4c4;
      flag = flip_pattern(tile[id].minitile[2][1].connections, 4) & 0xc4c4c4c4;

      if (pattern!=flag)
         return 0;
   }
   
   return 1;
}

/* Returns TRUE if a tile will fit anywhere *in its current orientation* */
int tile_fits_anywhere(const int id)
{
   int n;

   if (id == NOTILE)
      return 0;

   for (n=0; n<num_open_tiles; n++)
      if (tile_fits(id, open_tiles[n].x, open_tiles[n].y))
         return 1;

   return 0;
}

/* Returns TRUE if a tile will fit anywhere *in any orientation* */
int tile_fits_anywhere_anyway(const int id)
{
   int n;

   if (id == NOTILE)
      return 0;

   for (n=0; n<4; n++) {
      if (tile_fits_anywhere(id))
         return 1;
      rotate_big_tile_270(id);
   }

   return 0;
}
