#include <allegro.h>
#include <string.h>
#include "tileset.h"
#include "tiles.h"
#include "maptiles.h"
#include "team.h"
#include "genrand.h"
#include "assert.h"
#include "global.h"
#include "rules.h"
#include "theme.h"
#include "str.h"
#include "gfx.h"

/* Rule modifiers */
int rule_flags = 0;
int city_shield_bonus = 1;
int base_farm_multi = 3;
int bonus_farm_multi = 4;
int game_cathedral_multi = 3;
int final_cathedral_multi = 0;
int game_inn_multi = 2;
int final_inn_multi = 0;
int tradegood_bonus = 10;

void place_man(const int tx, const int ty, const int mtx, const int mty, const int t, const int type)
{
   int n, id;

   n = find_tile(tx, ty);

   if (n == NOTILE)
      return;

   id = tile[n].minitile[mty][mtx].structnum;
   feature_men[id][t].men[type]++;
   team[t].men[type]--;

   tile[n].minitile[mty][mtx].man_type = type;
   tile[n].minitile[mty][mtx].man_team = t;
}

/* Add an integer to a list of integers if it isn't already there */
static inline int store_integer(const int value, int *list, int list_size)
{
   int c;

   for (c=0; c<list_size; c++) {
      if (list[c]==value)
         return list_size;
   }
   list[list_size] = value;
   list_size++;

   return list_size;
}

/* Find a list of all farms adjacent to a city. *argv should be large */
/*  enough to hold all of them */
static int find_farm_list(const int city_id, int *argv)
{
   int x, y;
   int count;
   int num;
   int n;

   count = 0;

   for (n=0; n<tiles_in_play; n++) {
      if ((tile[playboard[n]].flags&TILEF_MONASTERY)==0)
         for(y=0; y<3; y++) {
            for(x=0; x<3; x++) {
               /* Repeat for beleaguered cities */
               /* (fudge) */
               if (tile[playboard[n]].minitile[y][x].structnum==city_id) {
                  if (x>0) {
                     num = tile[playboard[n]].minitile[y][x-1].structnum;
                     if (feature_type[num] == FEATURE_FARM)
                        count = store_integer(num, argv, count);
                  }
                  if (x<2) {
                     num = tile[playboard[n]].minitile[y][x+1].structnum;
                     if (feature_type[num] == FEATURE_FARM)
                        count = store_integer(num, argv, count);
                  }
                  if (y>0) {
                     num = tile[playboard[n]].minitile[y-1][x].structnum;
                     if (feature_type[num] == FEATURE_FARM)
                        count = store_integer(num, argv, count);
                  }
                  if (y<2) {
                     num = tile[playboard[n]].minitile[y+1][x].structnum;
                     if (feature_type[num] == FEATURE_FARM)
                        count = store_integer(num, argv, count);
                  }
               }
            }
         }
   }

   return count;
}

/* Award points for the player with the most farmers in a set of farms */
/* Base score is the score without a pig, bonus score is the score with a pig */
/* Scores per city rather than per meadow */
static void score_farms(int *list, int count, int base_score, int bonus_score)
{
   int *team_men;
   int *team_pigs;
   int num_men;
   int c;
   int id;

   team_pigs = malloc(num_teams * sizeof *team_pigs);
   team_men = malloc(num_teams * sizeof *team_men);

   /* Find the total number of men each player has in the farms */
   for(c=0; c<num_teams; c++) {
      team_pigs[c] = 0;
      team_men[c] = 0;
   }

   for(id=0; id<count; id++) {
      for(c=0; c<num_teams; c++) {
         team_pigs[c] += feature_men[list[id]][c].men[PIGS];
         team_men[c] += get_feature_team_men_count(list[id], c);
      }
   }

   num_men = 0;
   for(c=0; c<num_teams; c++)
      if (team_men[c] > num_men)
         num_men=team_men[c];

   /* No men here at all */
   if (!num_men)
      return;

   /* Find teams that deserve points for this city */
   for (c=0; c<num_teams; c++) {
      if (team_men[c]==num_men) {
         if (team_pigs[c])
            team[c].score[FEATURE_FARM]+=bonus_score;
         else
            team[c].score[FEATURE_FARM]+=base_score;
      }
   }
}

/* Count monastery */
static int score_monastery(const int x, const int y)
{
   int score=0;
   int tx, ty;

   for(tx=-1; tx<2; tx++) {
      for(ty=-1; ty<2; ty++) {
         if (find_tile(x+tx, y+ty)!=NOTILE)
            score++;
      }
   }

   return score;
}

/* Award tradegoods from the completed city id to the trader */
static void score_tradegoods(int id, int trader)
{
   int n;
   
   for (n=0; n<3; n++) {
      team[trader].goods[n] += feature_bonus[id].goods[n];
      feature_bonus[id].goods[n] = 0;
   }
}

/* Count city points. */
/* The base and bonus multipliers affect Cathedrals and such */
static void score_city(const int id, const int base_multi, const int bonus_multi)
{
   int n, c, num_men;
   int x, y;
   int score = 0;
   int multi;

   /* Don't count unfinished cities with cathedrals */
   if (bonus_multi==0 && feature_bonus[id].cathedrals)
      return;

   /* Count number of tiles */
   for (n=0; n<tiles_in_play; n++) {
      for(y=0; y<3; y++) {
         for(x=0; x<3; x++) {
            if (tile[playboard[n]].minitile[y][x].structnum==id) {
               score++;
               goto break_double_loop;
            }
         }
      }
break_double_loop:;
   }
   
   /* Bonus Shields */
   score += feature_bonus[id].shields*city_shield_bonus;

   /* Find the largest force */
   num_men = get_feature_largest_force(id);

   /* No men here at all */
   if (!num_men)
      return;

   if (score==2)  /* City of two tiles only counts for 2 points */
      multi = 1;
   else if (feature_bonus[id].cathedrals)
      multi = bonus_multi;
   else
      multi = base_multi;
      
   /* Cities under siege are worth one point less per tile */
   multi -= feature_bonus[id].siege;

   /* Find teams that deserve points for this city */
   for (c=0; c<num_teams; c++) {
      if (get_feature_team_men_count(id, c)==num_men) {
         team[c].score[FEATURE_CITY]+=multi*score;
      }
   }
}

/* Count road points. */
/* The base and bonus multipliers affect inns and such */
static void score_road(const int id, const int base_multi, const int bonus_multi)
{
   int n, c, num_men;
   int x, y;
   int score = 0;
   int multi;
   int terminal_count;

   /* Don't count unfinished roads with inns */
   if ((bonus_multi==0 && feature_bonus[id].inns))
      return;

   /* Count number of tiles */
   terminal_count = 0;
   for (n=0; n<tiles_in_play; n++) {
      for(y=0; y<3; y++) {
         for(x=0; x<3; x++) {
            if (tile[playboard[n]].minitile[y][x].structnum==id) {
               if (tile[playboard[n]].minitile[1][1].structnum!=id && !(tile[playboard[n]].flags&TILEF_MOD))
                  terminal_count++;
               score++;
               goto break_double_loop;
            }
         }
      }
break_double_loop:;
   }

//   if (terminal_count==0)
//      return;

   /* Find the largest force */
   num_men = get_feature_largest_force(id);

   /* No men here at all */
   if (!num_men)
      return;

   if (feature_bonus[id].inns)
      multi = bonus_multi;
   else
      multi = base_multi;

   /* Find teams that deserve points for this city */
   for (c=0; c<num_teams; c++) {
      if (get_feature_team_men_count(id, c)==num_men) {
         team[c].score[FEATURE_ROAD]+=multi*score;
      }
   }
}

/* Calculate score for a tile */
/* Should only be called once for each tile */
/* trader is the player the tradegoods should be awarded to. */
void score_tile(int x, int y, int trader)
{
   int id;
   int tx, ty;
   int n, c;

   n = find_tile(x, y);

   if (n==NOTILE)
      return;

   /* Check roads and cities */
   for (c=0; c<4; c++) {
      //conn = tile[n].minitile[xypairs[c][1]][xypairs[c][0]].connections;
      id = tile[n].minitile[xypairs[c][1]][xypairs[c][0]].structnum;

      if (feature_open[id]==0) {
         if (feature_type[id] == FEATURE_ROAD) {
            /* Score road */
            score_road(id, 1, game_inn_multi);
            remove_men(id);
         } else if (feature_type[id] == FEATURE_CITY) {
            /* Score city */
            score_city(id, 2, game_cathedral_multi);
            score_tradegoods(id, trader);
            remove_men(id);
         }
      }
   }

   /* Score monasteries */
   for(tx=-1; tx<2; tx++) {
      for(ty=-1; ty<2; ty++) {
         n = find_tile(x+tx, y+ty);
         if (n!=NOTILE && tile[n].flags&TILEF_MONASTERY && tile[n].minitile[1][1].man_team!=-1) {
            /* Check if it is completed */
            if (score_monastery(x+tx, y+ty)==9) {
               /* Add score */
               team[tile[n].minitile[1][1].man_team].score[FEATURE_MONASTERY]+=9;
               remove_men(tile[n].minitile[1][1].structnum);
            }
         }
      }
   }
}

/* End score for partially build structures and farms */
void end_score(void)
{
   int n, c, id;
   int *farm_list;
   int m_count;
   int siege_bonus;

   for (id = 0; id<feature_id_count; id++) {
      /* Only score unfinished items */
      if (feature_open[id]>0) {
         switch (feature_type[id]) {
            case FEATURE_ROAD:
               score_road(id, 1, final_inn_multi);
               remove_men(id);
               break;
            case FEATURE_CITY:
               score_city(id, 1, final_cathedral_multi);
               remove_men(id);
               break;
         }
      }
   }

   /* Monasteries */
   for (n = 0; n<tiles_in_play; n++) {
      if (tile[n].flags&TILEF_MONASTERY && tile[n].minitile[1][1].man_team!=-1) {
         team[tile[n].minitile[1][1].man_team].score[FEATURE_MONASTERY] += score_monastery(tile[n].x, tile[n].y);
         remove_men(tile[n].minitile[1][1].structnum);
      }
   }

   /* Farms */
   farm_list = malloc(feature_id_count * sizeof *farm_list);

   for (id = 0; id<feature_id_count; id++) {
      /* Only score finished items */
      if (feature_open[id]==0 && feature_type[id]==FEATURE_CITY) {
         /* Find farms adjacent to this city */
         m_count = find_farm_list(id, farm_list);
         /* Award score for peasants - double if the city is under siege */
         siege_bonus = feature_bonus[id].siege+1;
         score_farms(farm_list, m_count, 
                     base_farm_multi*siege_bonus, bonus_farm_multi*siege_bonus);
      }
   }
   free(farm_list);
   
   /* Tradegoods */
   for (n=0; n<GOODS_TYPES; n++) {
      /* Find majority */
      m_count = 0;
      for (c=0; c<num_teams; c++) {
         if (team[c].goods[n]>m_count)
            m_count = team[c].goods[n];
      }
      
      /* Award points */
      if (m_count) {
         for (c=0; c<num_teams; c++) {
            if (team[c].goods[n]==m_count)
               team[c].score_goods[n] += tradegood_bonus;
         }
      }
   }
   

   for (id = 0; id<feature_id_count; id++)
      if (feature_type[id]!=FEATURE_NONE)
         remove_men(id);
}

int can_place_flag(int tile_x, int tile_y, int mx, int my, int team, int type)
{
   int n;
   int structnum;

   n = find_tile(tile_x, tile_y);
   if (n==NOTILE)
      return FALSE;
      
   structnum = tile[n].minitile[my][mx].structnum;
   switch (type) {
      case PEASANTS:
      case BIGMEN:
         return get_feature_men_count(structnum)==0;
      case MASTERBUILDERS:
         if (feature_type[structnum] == FEATURE_CITY || feature_type[structnum] == FEATURE_ROAD)
            return get_feature_team_men_count(tile[n].minitile[my][mx].structnum, team)!=0;
      case PIGS:
         if (feature_type[structnum] != FEATURE_CITY && feature_type[structnum] != FEATURE_ROAD)
            return get_feature_team_men_count(tile[n].minitile[my][mx].structnum, team)!=0;
   }
   
   return FALSE;
}

void draw_flags(BITMAP *dest, int dest_x, int dest_y, int tx, int ty, int team, int type)
{
   int mtx, mty;
   int structnum;
   int n;

   n = find_tile(tx, ty);
   if (n>NOTILE) {
      for(mtx=0; mtx<3; mtx++) {
         for (mty=0; mty<3; mty++) {
         
            structnum = tile[n].minitile[mty][mtx].structnum;
            
            switch (type) {
               case PEASANTS:
               case BIGMEN:
                  if (!get_feature_men_count(structnum))
                     draw_sprite(dest, get_flag_bitmap(0), dest_x+mtx*TILE_SIZE/3, 
                                    dest_y+mty*TILE_SIZE/3-21+8);
                  break;
               case MASTERBUILDERS:
                  if ((feature_type[structnum] == FEATURE_CITY || feature_type[structnum] == FEATURE_ROAD) && get_feature_team_men_count(tile[n].minitile[mty][mtx].structnum, team)!=0)
                     draw_sprite(dest, get_flag_bitmap(0),
                                 dest_x+mtx*TILE_SIZE/3,
                                 dest_y+mty*TILE_SIZE/3-21+8);
                  break;
               case PIGS:
                  if (feature_type[structnum] != FEATURE_CITY && feature_type[structnum] != FEATURE_ROAD && get_feature_team_men_count(tile[n].minitile[mty][mtx].structnum, team)!=0)
                     draw_sprite(dest, get_flag_bitmap(0),
                                 dest_x+mtx*TILE_SIZE/3,
                                 dest_y+mty*TILE_SIZE/3-21+8);
                  break;
            }
         }
      }
   }
}

/* Return the number of men that might be placed on a tile */
int count_number_flags(int tx, int ty)
{
   int mtx, mty;
   int n;
   int num = 0;

   n = find_tile(tx, ty);
   if (n>NOTILE) {
      for(mtx=0; mtx<3; mtx++) {
         for (mty=0; mty<3; mty++) {
            if (!get_feature_men_count(tile[n].minitile[mty][mtx].structnum))
               num++;
         }
      }
   }
   
   return num;
}

static inline void draw_men(BITMAP *dest, int tile_num, int dest_x, int dest_y)
{
   int mtx, mty;
   int n;

   n = tile_num;

   for(mtx=0; mtx<3; mtx++) {
      for (mty=0; mty<3; mty++) {
         if (tile[n].minitile[mty][mtx].man_team!=-1) {
            const RLE_SPRITE *sprite = get_team_man_sprite(tile[n].minitile[mty][mtx].man_team, tile[n].minitile[mty][mtx].man_type);
            draw_rle_sprite(dest, sprite, dest_x+mtx*TILE_SIZE/3,
                                          dest_y+(mty+1)*TILE_SIZE/3 - sprite->h);
            /* Shadow outline */
            sprite = get_man_shadow_sprite(tile[n].minitile[mty][mtx].man_type);
            draw_rle_sprite(dest, sprite, dest_x+mtx*TILE_SIZE/3,
                                          dest_y+(mty+1)*TILE_SIZE/3 - sprite->h);

            /* Modifiers */
            if (tile[n].minitile[mty][mtx].man_type==0) {
               if ((tile[n].flags & TILEF_MONASTERY) && mty==1 && mtx==1)
                  sprite = get_man_attrib_sprite(MAN_PRIEST);
               else if (feature_type[tile[n].minitile[mty][mtx].structnum] == FEATURE_CITY)
                  sprite = get_man_attrib_sprite(MAN_KNIGHT);
               else if (feature_type[tile[n].minitile[mty][mtx].structnum] == FEATURE_ROAD)
                  sprite = get_man_attrib_sprite(MAN_ROBBER);
               else //if (feature_type[tile[n].minitile[mty][mtx].structnum] == FEATURE_FARM)
                  sprite = get_man_attrib_sprite(MAN_FARMER);
               if (sprite) {
                  draw_rle_sprite(dest, sprite, 
                                  dest_x+mtx*TILE_SIZE/3,
                                  dest_y+(mty+1)*TILE_SIZE/3 - sprite->h);
               }
            }
         }
      }
   }
}

void draw_tilemap(BITMAP *dest, int float_tile, int dest_x, int dest_y, int tx, int ty, int w, int h)
{
   int x, y;
   int n;

   for (y=0; y<h; y++) {
      for (x=0; x<w; x++) {
         blit(emptytile, dest, 0,0, dest_x+x*TILE_SIZE, dest_y+y*TILE_SIZE, TILE_SIZE, TILE_SIZE);
      }
   }
   
   for (n=0; n<tiles_in_play; n++) {
      x = tile[playboard[n]].x - tx;
      y = tile[playboard[n]].y - ty;
      if (in_rect(x,y, 0,0, w,h)) {
         blit(tile[playboard[n]].gfx, dest, 0,0, dest_x+x*TILE_SIZE, dest_y+y*TILE_SIZE, TILE_SIZE, TILE_SIZE);
         draw_trans_sprite(dest, grid, dest_x+x*TILE_SIZE, dest_y+y*TILE_SIZE);
         draw_men(dest, playboard[n], dest_x+x*TILE_SIZE, dest_y+y*TILE_SIZE);

/*            
      {
         int tx, ty;

         for(tx=0; tx<3; tx++) {
            for (ty=0; ty<3; ty++) {
               textprintf_centre_ex(dest, get_menu_font(), dest_x+x*TILE_SIZE+8+tx*TILE_SIZE/3, dest_y+y*TILE_SIZE+ty*TILE_SIZE/3, blue, -1, "%x", tile[playboard[n]].minitile[ty][tx].structnum);
               //textprintf_centre_ex(dest, get_menu_bold_font(), dest_x+x*TILE_SIZE+8+tx*TILE_SIZE/3, dest_y+y*TILE_SIZE+ty*TILE_SIZE/3, black, -1, "%x", tile[playboard[n]].minitile[ty][tx].connections);
               //textprintf_centre_ex(dest, get_menu_bold_font(), dest_x+x*TILE_SIZE+8+tx*TILE_SIZE/3, dest_y+y*TILE_SIZE+ty*TILE_SIZE/3, white-63, -1, "%x", (tile[playboard[n]].minitile[ty][tx].connections&0xf0)>>4);
               //textprintf_centre_ex(dest, get_menu_bold_font(), dest_x+x*TILE_SIZE+8+tx*TILE_SIZE/3, dest_y+y*TILE_SIZE+ty*TILE_SIZE/3, black, -1, "%d", feature_bonus[tile[playboard[n]].minitile[ty][tx].structnum].goods[1]);
               //textprintf_centre_ex(dest, get_menu_bold_font(), dest_x+x*TILE_SIZE+8+tx*TILE_SIZE/3, dest_y+y*TILE_SIZE+ty*TILE_SIZE/3, black, -1, "%d", feature_type[tile[playboard[n]].minitile[ty][tx].structnum]);
            }
         }
      }
//            */
      }
   }

   for (n=0; n<num_open_tiles; n++) {
      x = open_tiles[n].x - tx;
      y = open_tiles[n].y - ty;
      if (in_rect(x,y, 0,0, w,h)) {
         if (tile_fits(float_tile, tx+x, ty+y))
            draw_lit_sprite(dest, emptytile, dest_x+x*TILE_SIZE, dest_y+y*TILE_SIZE, 128);
      }
   }
}
