/*----------------------------------------------------------------
 *   utimer.c -- UNIX default millisecond timer function
 *----------------------------------------------------------------
 *  libnet is (c) Copyright Chad Catlett and George Foot 1997-1998
 *
 *  Please look in `docs' for details, documentation and
 *  distribution conditions.
 */

/* Why is this here?  Why don't I just use `clock'?  Well, I
 * did, but then found that under Linux in particular `clock'
 * does not indicate real-life time, it indicates the process's
 * time only.  When `rdmtest' sat waiting for input, the value
 * returned by `clock' wasn't increasing.  I thought this was a
 * bug in the RDM code, when in fact it's just a
 * misunderstanding.  Also, this way lets platforms use whatever
 * their most accurate timing method is, to get better
 * granularity than the 1 second provided by `time'.  Plus the
 * user can override this function.  Everybody wins.
 *
 * It should return a time in milliseconds.
 */

#include <libnet.h>
#include "libnet/timer.h"

#if !defined TARGET_LINUX && !defined TARGET_UNIX
   #error Something is wrong with the makefile...
#endif

#include <sys/time.h>
#include <unistd.h>
#include "libnet/threads.h"

unsigned int _net_default_timer_func (void)
{
   static struct timeval start_time;
   static int virgin = 1;
   MUTEX_DEFINE_STATIC(timer);

   struct timeval current_time;
   unsigned int retval;
   
   if (virgin) {
      virgin = 0;
      gettimeofday (&start_time, NULL);
      MUTEX_CREATE(timer);
   }
   
   MUTEX_LOCK(timer);
   
   gettimeofday (&current_time, NULL);
   current_time.tv_sec -= start_time.tv_sec;
   current_time.tv_usec -= start_time.tv_usec;
   
   retval = current_time.tv_sec * 1000 + current_time.tv_usec / 1000;

   MUTEX_UNLOCK(timer);

   return retval;
}
