//
// The Lord of the Stars - Copyright (c) 2005 Rodrigo Braz Monteiro
// Rodrigo Braz Monteiro's entry for TINS'05
// This source file is subject to GNU General Public License
// See included license.txt for more information
//
// You can reach the author at:
// http://zeratul.cellosoft.com
// zeratul@cellosoft.com
//

// Headers
#include <allegro.h>
#include <math.h>
#include "enemy.h"
#include "playframe.h"
#include "data.h"
#include "player.h"
#include "particle.h"
#include "kernel.h"


////            ////
//// BASE ENEMY ////
////            ////

// Constructor
Enemy::Enemy() {
	Hue = rand() % 350 + 5;
	//Hue = 350;
	DieStep = 0;
	BlinkStep = 0;
}


// Destructor
Enemy::~Enemy() {
	destroy_bitmap(bmp);
}


// Update
void Enemy::Update() {
	// Die
	if (Life <= 0 && DieStep == 0) {
		parent->ScoreToAdd += ScoreValue;
		DieStep = 1;
		Particle *debris;
		for (int i=0;i<20;i++) {
			debris = CreateParticle();
			debris->Friendly = true;
			debris->DrawMethod = 0;
			debris->bmp = Data.DebrisBmp;
			debris->vx = (rand()%9-4)*2-1;
			debris->vy = (rand()%9-4)*2+1;
			debris->px = 3;
			debris->py = 3;
		}
		play_sample(Data.Sample[3],255,MIN(MAX(0,int(x)*255/screen->w),255),1000,0);
	}

	// Dying
	if (DieStep > 0) {
		Particle *explosion;
		for (int i=0;i<5;i++) {
			explosion = CreateParticle();
			explosion->Friendly = true;
			explosion->DrawMethod = 1;
			explosion->ttl = 10;
			explosion->bmp = Data.ExplosionBmp;
			explosion->vx = rand()%9-4;
			explosion->vy = rand()%9-4;
			explosion->px = 16;
			explosion->py = 16;
			explosion->x += rand()%13-6;
			explosion->y += rand()%13-6;
		}

		DieStep++;
		if (DieStep == 10) {
			Destroy();
			return;
		}
	}

	// Blink
	if (BlinkStep > 0) BlinkStep--;

	// Check for collision with player
	if (Collides(parent->Player)) {
		if (!parent->Player->TakingDamage && parent->Player->HasControl) {
			parent->Player->Damage(60);
			play_sample(Data.Sample[4],255,MAX(0,MIN(int(x)*255/screen->w,255)),1000,0);
		}
	}

	// Update as sprite
	Sprite::Update();
}


// Draw
void Enemy::Draw() {
	// Draw as sprite
	if (BlinkStep && (BlinkStep/4) % 2 == 0 || DieStep != 0) {
		int dx = int(x)-px;
		int dy = int(y)-py;
		set_color_blender(255,255,255,255);
		draw_lit_sprite(Kernel.BackBuffer,bmp,dx,dy,255);
		Kernel.SetDirty(dx,dy,bmp->w,bmp->h);
	}
	else Sprite::Draw();
}


// Damage
void Enemy::Damage (int value,int attackhue) {
	BlinkStep = 36;
	int ahue = attackhue;
	if (ahue < 0) ahue += 360;
	if (ahue > 360) ahue -= 360;

	int dist = ahue - Hue;
	if (dist < 0) dist += 360;
	if (dist > 360) dist -= 360;
	if (dist > 180) dist = 360 - dist;

	double efficiency = pow(1-(double(dist)/180.0),3.2);
	int damage = int(double(value) * efficiency);
	if (Life > 0) ScoreValue += int(efficiency * damage);
	Life -= damage;
}


// Make colored bmp
void Enemy::MakeBmp(BITMAP* src,int colorToReplace) {
	bmp = create_bitmap(src->w,src->h);
	int col;
	int pink = makecol(255,0,255);
	int huecolor = Data.HSVtoRGB(Hue,1,1);
	if (huecolor == pink) huecolor = makecol(255,0,254);
	for (int y=bmp->h;--y>=0;) {
		for (int x=bmp->w;--x>=0;) {
			col = getpixel(src,x,y);
			if (col == colorToReplace) col = huecolor;
			putpixel(bmp,x,y,col);
		}
	}
}


////     ////
//// ORC ////
////     ////

// Constructor
EnemyOrc::EnemyOrc() {
	// Init variables
	ScoreValue = 50;
	Life = 300;
	x = 680;
	y = float(rand()%340+60);
	px = 24;
	py = 24;
	radius = 14;
	ShootTimer = rand()%80;
	shootx = -26;
	shooty = 17;
	Type = SPRITE_TYPE_ORC;

	// Set bitmap
	MakeBmp(Data.EnemyBmp[0],makecol(255,255,255));
}


// Draw
void EnemyOrc::Draw() {
	// Draw as enemy
	Enemy::Draw();
}


// Update
void EnemyOrc::Update() {
	// Set speed;
	vx = -3;

	// Die after it leaves screen
	if (x < -40) {
		Destroy();
		return;
	}

	// Dying
	if (DieStep > 0) {
		Enemy::Update();
		return;
	}

	// Shoot
	ShootTimer++;
	if (ShootTimer == 75 || ShootTimer == 80) {
		play_sample(Data.Sample[6],255,MAX(0,MIN(255,int(x)*255/screen->w)),1000,0);
		Particle *shoot = CreateParticle();
		shoot->Damage = 30;
		shoot->DrawMethod = 1;
		shoot->bmp = Data.EnemyShotBmp[0];
		shoot->vx = -8;
		float dx = parent->Player->x-x;
		if (dx == 0) dx = 1;
		shoot->vy = (parent->Player->y-y)/dx*-8;
		if (shoot->vy > 3) shoot->vy = 3;
		if (shoot->vy < -3) shoot->vy = -3;
		shoot->px = 4;
		shoot->py = 4;
		shoot->radius = 2;
		if (ShootTimer == 80) ShootTimer = 0;
	}

	// Update as enemy
	Enemy::Update();
}


////      ////
//// URUK ////
////      ////

// Constructor
EnemyUruk::EnemyUruk() {
	// Init variables
	ScoreValue = 250;
	Life = 700;
	x = 690;
	y = float(rand()%340+60);
	px = 37;
	py = 23;
	radius = 16;
	ShootTimer = rand()%80;
	shootx = -32;
	shooty = 2;
	Type = SPRITE_TYPE_URUK;
	vx = -2 -float(rand()%5/4);

	// Set bitmap
	MakeBmp(Data.EnemyBmp[5],makecol(255,255,255));
}

// Draw
void EnemyUruk::Draw() {
	// Draw as enemy
	Enemy::Draw();
}


// Update
void EnemyUruk::Update() {

	// Die after it leaves screen
	if (x < -50) {
		Destroy();
		return;
	}

	// Dying
	if (DieStep > 0) {
		Enemy::Update();
		return;
	}

	// Track player
	else {
		// Move towards player in Y axis
		int ty = (int)parent->Player->y;
		if (abs(targety-ty) > 10) targety = ty + rand()%9 - 4;
		if (targety > y+5) vy += 0.1;
		if (targety < y-5) vy -= 0.1;
	}
	vy *= 0.9;
	if (vy > 3) vy = 3;
	if (vy < -3) vy = -3;

	// Shoot
	ShootTimer++;
	if (ShootTimer == 100 || ShootTimer == 105 || ShootTimer == 110) {
		play_sample(Data.Sample[6],255,MAX(0,MIN(255,int(x)*255/screen->w)),1000,0);
		Particle *shoot = CreateParticle();
		shoot->Damage = 40;
		shoot->DrawMethod = 1;
		shoot->bmp = Data.EnemyShotBmp[0];
		shoot->vx = -8;
		float dx = parent->Player->x-x;
		if (dx == 0) dx = 1;
		shoot->vy = (parent->Player->y-y)/dx*-8;
		if (shoot->vy > 5) shoot->vy = 5;
		if (shoot->vy < -5) shoot->vy = -5;
		shoot->px = 4;
		shoot->py = 4;
		shoot->radius = 2;
		if (ShootTimer == 110) ShootTimer = 0;
	}

	// Update as enemy
	Enemy::Update();
}


////       ////
//// TROLL ////
////       ////

// Constructor
EnemyTroll::EnemyTroll() {
	// Init variables
	ScoreValue = 300;
	Life = 1600;
	x = 690;
	y = float(rand()%340+60);
	px = 32;
	py = 24;
	radius = 20;
	ShootTimer = rand()%220;
	shootx = -29;
	shooty = 12;
	mode = 0;
	Type = SPRITE_TYPE_TROLL;

	// Set bitmap
	MakeBmp(Data.EnemyBmp[1],makecol(255,255,255));
}


// Draw
void EnemyTroll::Draw() {
	// Draw as enemy
	Enemy::Draw();
}


// Update
void EnemyTroll::Update() {
	// Die after it leaves screen
	if (x < -50) {
		Destroy();
		return;
	}

	// Dying
	if (DieStep > 0) {
		Enemy::Update();
		return;
	}

	// Wave is off, get out
	if (!parent->WaveActive) {
		vx = -2;
	}

	// Track player
	else {
		// Move towards player in Y axis
		int ty = (int)parent->Player->y;
		if (abs(targety-ty) > 10) targety = ty + rand()%9 - 4;
		if (targety > y+5) vy += 0.1;
		if (targety < y-5) vy -= 0.1;

		// Set horizontal speed
		int tvx;
		if (mode == 0) {
			if (x < 50) mode = 1;
			tvx = -2;
		}
		else {
			if (x > 500) mode = 0;
			tvx = 1;
		}
		if (vx < tvx) vx += 0.05;
		else if (vx > tvx) vx -= 0.1;
	}
	vy *= 0.9;
	vx *= 0.95;
	if (vy > 3) vy = 3;
	if (vy < -3) vy = -3;

	// Shoot
	ShootTimer++;
	if (ShootTimer == 220) {
		play_sample(Data.Sample[7],255,MAX(0,MIN(255,int(x)*255/screen->w)),1000,0);
		Particle *shoot = CreateParticle();
		shoot->DrawMethod = 1;
		shoot->Damage = 90;
		shoot->bmp = Data.EnemyShotBmp[1];
		shoot->vx = -6;
		shoot->px = 8;
		shoot->py = 12;
		shoot->radius = 9;
		ShootTimer = 0;
	}

	// Update as enemy
	Enemy::Update();
}

////      ////
//// MINE ////
////      ////
// (Also includes asteroids)

// Constructor
EnemyMine::EnemyMine() {
	// Init variables
	x = 680;
	y = float(rand()%340+60);
	Type = SPRITE_TYPE_MINE;
	bmp = NULL;
}

EnemyMine::EnemyMine(int i) {
	bmp = NULL;
	MineType = i;
	x = 680;
	y = float(rand()%340+60);
	Type = SPRITE_TYPE_MINE;

	switch (MineType) {
		case 0:
			MakeBmp(Data.EnemyBmp[2],makecol(255,255,255));
			radius = 12;
			px = 12;
			py = 12;
			BlowRadius = 50;
			BlowDamage = 150;
			ScoreValue = 150;
			Life = 250;
			break;

		case 1:
			MakeBmp(Data.EnemyBmp[3],makecol(255,255,255));
			radius = 16;
			px = 14;
			py = 16;
			BlowRadius = 16;
			BlowDamage = 80;
			ScoreValue = 50;
			Life = 250;
			break;

		case 2:
			MakeBmp(Data.EnemyBmp[4],makecol(255,255,255));
			radius = 12;
			px = 15;
			py = 16;
			BlowRadius = 12;
			BlowDamage = 60;
			ScoreValue = 30;
			Life = 150;
			break;
	}
}


// Draw
void EnemyMine::Draw() {
	// Draw as enemy
	Enemy::Draw();
}


// Update
void EnemyMine::Update() {
	// Set speed;
	vx = -3;

	// Die after it leaves screen
	if (x < -40) {
		Destroy();
		return;
	}

	// Dying
	if (DieStep > 0) {
		Enemy::Update();
		return;
	}

	// Check for collision with player
	float prevRad = radius;
	radius = BlowRadius;
	if (parent->Player->Alive && Collides(parent->Player)) {
		Life = 0;
		if (!parent->Player->TakingDamage && parent->Player->HasControl) {
			parent->Player->Damage(BlowDamage);
		}
	}
	radius = prevRad;

	// Update as enemy
	Enemy::Update();
}
