//
// The Lord of the Stars - Copyright (c) 2005 Rodrigo Braz Monteiro
// Rodrigo Braz Monteiro's entry for TINS'05
// This source file is subject to GNU General Public License
// See included license.txt for more information
//
// You can reach the author at:
// http://zeratul.cellosoft.com
// zeratul@cellosoft.com
//

// Headers
#include <allegro.h>
#define ALFONT_DLL
#include <alfont.h>
#include <fblend.h>
#include "playframe.h"
#include "data.h"
#include "kernel.h"
#include "input.h"
#include "audio.h"
#include "player.h"
#include "enemy.h"
#include "people.h"
#include "particle.h"
#include "wave.h"
#include "highscores.h"
#include "dialogue.h"


// Constructor
PlayFrame::PlayFrame () {
	Data.Log("Starting playframe constructor");
	// Generate color slider
	ColorSlider = create_bitmap(screen->w,24);
	int curColor;
	for (int j=0;j<4;j++) {
		curColor = makecol(100+j*20,100+j*20,100+j*20);
		for (int i=screen->w;--i>=0;) {
			putpixel(ColorSlider,i,j,curColor);
		}
	}
	for (int i=screen->w;--i>=0;) {
		curColor = GetColorFromSlider((float)i/float(screen->w));
		for (int j=4;j<24;j++) {
			putpixel(ColorSlider,i,j,curColor);
		}
	}

	Data.Log("Loading bmps");
	// Load bitmaps
	char path[512];
	Data.MakePath(path,"img/cursor.bmp");
	SliderCursor = load_bitmap(path,NULL);
	Data.MakePath(path,"img/statusbar.bmp");
	StatusBar = load_bitmap(path,NULL);

	// Init vars
	SliderPos = 0;
	SliderSpeed = 0;
	JustShot = false;
	PlayerScore = 0;
	PlayerLives = 2;
	ScoreToAdd = 0;
	CurrentDialogue = 0;
	DialoguePerson[0] = 0;
	DialoguePerson[1] = 1;
	DialogueSpeaker = 0;
	Dialogue = false;
	CurrentWave = -1;
	WaveActive = true;
	GameOver = false;
	GameWin = false;
	NamePos = 0;
	TimeToRessPlayer = 0;
	NextWaveTimer = 0;
	HasWave = 0;
	pause = false;
	for (int i=0;i<32;i++) PlayerName[i] = 0;

	// Create player
	Data.Log("Creating player");
	Player = new PlayerClass;
	AddSprite(Player);

	// Force update
	Kernel.RedrawAll = true;
	Data.Log("Done initing playframe");

	// Midi
	Audio.SetMidi(1);

	// Waves
	Wave[0] = new EnemyWave("The Space Shire");
	Wave[0]->MidiNumber = 1;
	Wave[0]->SpawnMinTime = 50;
	Wave[0]->SpawnMaxTime = 70;
	Wave[0]->EnemyChance[ENEMY_ORC] = 20;
	Wave[0]->EnemyChance[ENEMY_TROLL] = 0;
	Wave[0]->EnemyChance[ENEMY_MINE] = 0;
	Wave[0]->EnemyChance[ENEMY_ASTEROID1] = 50;
	Wave[0]->EnemyChance[ENEMY_ASTEROID2] = 70;
	Wave[0]->EnemyChance[ENEMY_URUK] = 0;
	Wave[0]->SpawnCount = 50;
	Wave[1] = new EnemyWave("Vacuumtop");
	Wave[1]->MidiNumber = 1;
	Wave[1]->SpawnMinTime = 30;
	Wave[1]->SpawnMaxTime = 40;
	Wave[1]->EnemyChance[ENEMY_ORC] = 80;
	Wave[1]->EnemyChance[ENEMY_TROLL] = 0;
	Wave[1]->EnemyChance[ENEMY_MINE] = 0;
	Wave[1]->EnemyChance[ENEMY_ASTEROID1] = 120;
	Wave[1]->EnemyChance[ENEMY_ASTEROID2] = 120;
	Wave[1]->EnemyChance[ENEMY_URUK] = 0;
	Wave[1]->SpawnCount = 80;
	Wave[2] = new EnemyWave("Rivenmoon");
	Wave[2]->MidiNumber = 2;
	Wave[2]->SpawnMinTime = 25;
	Wave[2]->SpawnMaxTime = 40;
	Wave[2]->EnemyChance[ENEMY_ORC] = 80;
	Wave[2]->EnemyChance[ENEMY_TROLL] = 0;
	Wave[2]->EnemyChance[ENEMY_MINE] = 0;
	Wave[2]->EnemyChance[ENEMY_ASTEROID1] = 140;
	Wave[2]->EnemyChance[ENEMY_ASTEROID2] = 100;
	Wave[2]->EnemyChance[ENEMY_URUK] = 20;
	Wave[2]->SpawnCount = 100;
	Wave[3] = new EnemyWave("The Asteroid Fields of Morr");
	Wave[3]->MidiNumber = 2;
	Wave[3]->SpawnMinTime = 10;
	Wave[3]->SpawnMaxTime = 20;
	Wave[3]->EnemyChance[ENEMY_ORC] = 80;
	Wave[3]->EnemyChance[ENEMY_TROLL] = 20;
	Wave[3]->EnemyChance[ENEMY_MINE] = 0;
	Wave[3]->EnemyChance[ENEMY_ASTEROID1] = 200;
	Wave[3]->EnemyChance[ENEMY_ASTEROID2] = 260;
	Wave[3]->EnemyChance[ENEMY_URUK] = 40;
	Wave[3]->SpawnCount = 250;
	Wave[4] = new EnemyWave("Loathelorien");
	Wave[4]->MidiNumber = 3;
	Wave[4]->SpawnMinTime = 10;
	Wave[4]->SpawnMaxTime = 40;
	Wave[4]->EnemyChance[ENEMY_ORC] = 100;
	Wave[4]->EnemyChance[ENEMY_TROLL] = 10;
	Wave[4]->EnemyChance[ENEMY_MINE] = 0;
	Wave[4]->EnemyChance[ENEMY_ASTEROID1] = 200;
	Wave[4]->EnemyChance[ENEMY_ASTEROID2] = 160;
	Wave[4]->EnemyChance[ENEMY_URUK] = 80;
	Wave[4]->SpawnCount = 170;
	Wave[5] = new EnemyWave("The Dead Marches");
	Wave[5]->MidiNumber = 3;
	Wave[5]->SpawnMinTime = 20;
	Wave[5]->SpawnMaxTime = 30;
	Wave[5]->EnemyChance[ENEMY_ORC] = 20;
	Wave[5]->EnemyChance[ENEMY_TROLL] = 20;
	Wave[5]->EnemyChance[ENEMY_MINE] = 30;
	Wave[5]->EnemyChance[ENEMY_ASTEROID1] = 80;
	Wave[5]->EnemyChance[ENEMY_ASTEROID2] = 40;
	Wave[5]->EnemyChance[ENEMY_URUK] = 60;
	Wave[5]->SpawnCount = 150;
	Wave[6] = new EnemyWave("Lunas Morgul");
	Wave[6]->MidiNumber = 4;
	Wave[6]->SpawnMinTime = 25;
	Wave[6]->SpawnMaxTime = 40;
	Wave[6]->EnemyChance[ENEMY_ORC] = 10;
	Wave[6]->EnemyChance[ENEMY_TROLL] = 40;
	Wave[6]->EnemyChance[ENEMY_MINE] = 40;
	Wave[6]->EnemyChance[ENEMY_ASTEROID1] = 20;
	Wave[6]->EnemyChance[ENEMY_ASTEROID2] = 20;
	Wave[6]->EnemyChance[ENEMY_URUK] = 60;
	Wave[6]->SpawnCount = 200;
	Wave[7] = new EnemyWave("Mortor");
	Wave[7]->MidiNumber = 4;
	Wave[7]->SpawnMinTime = 15;
	Wave[7]->SpawnMaxTime = 20;
	Wave[7]->EnemyChance[ENEMY_ORC] = 10;
	Wave[7]->EnemyChance[ENEMY_TROLL] = 40;
	Wave[7]->EnemyChance[ENEMY_MINE] = 40;
	Wave[7]->EnemyChance[ENEMY_ASTEROID1] = 20;
	Wave[7]->EnemyChance[ENEMY_ASTEROID2] = 20;
	Wave[7]->EnemyChance[ENEMY_URUK] = 30;
	Wave[7]->SpawnCount = 250;
	Wave[8] = new EnemyWave("Star Quake");
	Wave[8]->MidiNumber = 4;
	Wave[8]->SpawnMinTime = 10;
	Wave[8]->SpawnMaxTime = 15;
	Wave[8]->EnemyChance[ENEMY_ORC] = 10;
	Wave[8]->EnemyChance[ENEMY_TROLL] = 10;
	Wave[8]->EnemyChance[ENEMY_MINE] = 10;
	Wave[8]->EnemyChance[ENEMY_ASTEROID1] = 20;
	Wave[8]->EnemyChance[ENEMY_ASTEROID2] = 20;
	Wave[8]->EnemyChance[ENEMY_URUK] = 10;
	Wave[8]->SpawnCount = 500;
	Wave[9] = new EnemyWave("Ending");
	Wave[9]->MidiNumber = 0;
	Wave[9]->SpawnMinTime = 100;
	Wave[9]->SpawnMaxTime = 100;
	Wave[9]->EnemyChance[ENEMY_ORC] = 0;
	Wave[9]->EnemyChance[ENEMY_TROLL] = 0;
	Wave[9]->EnemyChance[ENEMY_MINE] = 0;
	Wave[9]->EnemyChance[ENEMY_ASTEROID1] = 0;
	Wave[9]->EnemyChance[ENEMY_ASTEROID2] = 0;
	Wave[9]->EnemyChance[ENEMY_URUK] = 0;
	Wave[9]->SpawnCount = 0;
}


// Destructor
PlayFrame::~PlayFrame () {
	// Kill bitmaps
	if (ColorSlider) destroy_bitmap(ColorSlider);
	ColorSlider = NULL;
	if (SliderCursor) destroy_bitmap(SliderCursor);
	SliderCursor = NULL;
	if (StatusBar) destroy_bitmap(StatusBar);
	StatusBar = NULL;

	// Kill sprites
	list<Sprite*>::iterator cur;
	for (cur=Sprites.begin();cur!=Sprites.end();cur++) {
		delete *cur;
	}
	Sprites.erase(Sprites.begin(),Sprites.end());
}


// Update play frame logic
bool PlayFrame::Update () {
	// Leave
	if (Input.KeyDown[KEY_ESC]) {
		if (pause) {
			pause = false;
			Fade = 0;
		}
		else if (Dialogue) {
			Dialogue = false;
			DialogueLine = 0;
			WaveActive = true;
			CurrentDialogue++;
		}
		else {
			GameOver = true;
			Player->HasControl = false;
			WaveActive = false;
		}
	}

	// Pause
	if (!GameOver && Input.KeyDown[Data.GameKeys[7]]) {
		if (pause) {
			pause = false;
			Fade = 0;
		}
		else {
			pause = true;
			Fade = 130;
			FadeMode = 0;
		}
	}
	if (pause) return BaseFrame::Update();

	// Update score
	if (ScoreToAdd > 50) {
		ScoreToAdd -= 10;
		PlayerScore += 10;
	}
	else if (ScoreToAdd > 10) {
		ScoreToAdd -= 2;
		PlayerScore += 2;
	}
	else if (ScoreToAdd > 0) {
		ScoreToAdd -= 1;
		PlayerScore += 1;
	}

	// Player ressurection
	if (!Player->Alive && !GameOver) {
		TimeToRessPlayer++;
		if (TimeToRessPlayer >= 120) {
			TimeToRessPlayer = 0;
			if (PlayerLives > 0) {
				PlayerLives--;
				Player->Enable();
			}
			else {
				WaveActive = false;
				GameOver = true;
			}
		}
	}

	// Game over
	if (GameOver) {
		PlayerScore += ScoreToAdd;
		ScoreToAdd = 0;

		if (Fade < 128) {
			Fade++;
			clear_keybuf();
		}

		else {
			while (keypressed()) {
				int scorepos = HighScores.ScorePos(PlayerScore);
				if (scorepos == -1) {
					Audio.SetMidi(-1);
					Switch(2);
					break;
				}
				int playerkey = readkey();
				if ((playerkey >> 8) == KEY_ENTER && strlen(PlayerName) > 0) {
					HighScores.Insert(PlayerScore,PlayerName);
					Switch(2);
				}
				else if ((playerkey >> 8) == KEY_BACKSPACE) {
					if (NamePos > 0) NamePos--;
					if (NamePos == 16) PlayerName[NamePos+1] = 0;
					else PlayerName[NamePos] = 0;
				}
				else if ((playerkey & 0xFF) >= 0x30 && (playerkey & 0xFF) <= 0x7a) {
					PlayerName[NamePos] = playerkey & 0xFF;
					if (NamePos < 16) NamePos++;
				}
			}
		}
	}

	// Wave management
	if (HasWave && WaveActive) {
		if (Wave[CurrentWave]->SpawnCount <= 0) {
			HasWave = false;
		}
	}

	if (!HasWave && !GameOver) {
		NextWaveTimer++;
		WaveActive = false;

		// End of game, win! woo!
		if (CurrentWave == 9) {
			HasWave = false;
			GameWin = true;
			WaveActive = false;
			GameOver = true;
			Player->HasControl = false;
			Data.UnlockNext();
		}

		// Start next wave
		else if (NextWaveTimer == 300) {
			NextWaveTimer = 0;
			CurrentWave++;
			DialogueLine = 0;
			GetNextDialogueLine();
			HasWave = true;
		}
	}

	// Spawn enemies
	if (WaveActive) {
		Enemy *toAdd = Wave[CurrentWave]->SpawnEnemy();
		if (toAdd) AddSprite(toAdd);
	}

	// Spawn stars
	if (true) {
		int depth = rand() % 4;
		Particle *star = new Particle;
		star->Friendly = true;
		star->DrawMethod = 2;
		star->x = screen->w-1;
		star->y = (rand() % 408) + 48;
		star->vx = -3 - (1<<(depth-1));
		star->Color = Data.HSVtoRGB(rand()%360,0.3,float(2+depth)/5);
		AddSprite(star);
	}

	// Update sprites
	list<Sprite*>::iterator cur;
	list<Sprite*>::iterator next;
	for (cur=Sprites.begin();cur!=Sprites.end();cur=next) {
		next = cur;
		next++;
		(*cur)->Update();
	}

	// Dialogue
	if (Dialogue) {
		DialoguePos += 0.3;
		if (DialoguePos > alfont_text_height(Data.Font[1])*DialogueLines
			|| Input.KeyDown[Data.GameKeys[2]] || Input.MouseDown[0]) GetNextDialogueLine();
	}

	// Vars
	int w,h;
	w = screen->w;
	h = screen->h;

	// Shoot
	if (JustShot) {
		JustShot = false;
		play_sample(Data.Sample[1],255,128,1000,0);
	}

	// Slider mouse movement
	get_mouse_mickeys(&MouseMoveX,&MouseMoveY);
	position_mouse(w/2,h/2);
	int PrevSlider = int(SliderPos);
	if (Player->HasControl) {
		if (key[Data.GameKeys[3]] && SliderSpeed > -10) SliderSpeed -= 1;
		if (key[Data.GameKeys[4]] && SliderSpeed < 10) SliderSpeed += 1;
		SliderPos += MouseMoveX;
	}

	// Slider speed move
	SliderPos += SliderSpeed;
	SliderSpeed *= 0.9;
	while (SliderPos <= -w/2) SliderPos += w;
	while (SliderPos > w/2) SliderPos -= w;
	if (SliderPos != PrevSlider) Kernel.SetDirty(0,screen->h-24,screen->w,24);

	// Update as base
	return BaseFrame::Update();
}


// Render playframe screen
void PlayFrame::Render () {
	// Clear bg
	clear_bitmap(Kernel.BackBuffer);

	// Set vars
	int w,h;
	w = screen->w;
	h = screen->h;

	// Render sprites
	list<Sprite*>::reverse_iterator cur;
	list<Sprite*>::reverse_iterator next;
	for (cur=Sprites.rbegin();cur!=Sprites.rend();cur=next) {
		next = cur;
		next++;
		(*cur)->Draw();
	}

	// Draw dialogue
	if (Dialogue) {
		fblend_trans(Data.DialogueBmp,Kernel.BackBuffer,120,280,128);
		if (DialogueSpeaker == 0) rectfill(Kernel.BackBuffer,128,348,211,431,Data.Person[DialoguePerson[0]]->color);
		else rectfill(Kernel.BackBuffer,428,288,511,371,Data.Person[DialoguePerson[1]]->color);
		draw_sprite(Kernel.BackBuffer,Data.Person[DialoguePerson[0]]->bmp,130,350);
		draw_sprite(Kernel.BackBuffer,Data.Person[DialoguePerson[1]]->bmp,430,290);
		Kernel.SetDirty(120,280,400,160);

		int drawPos = (int)MAX(DialoguePos,0);
		set_clip_rect(Kernel.BackBuffer,222,322,416,396);
		DialogueLines = Data.BreakText(DialogueText,222,322-drawPos,195,Data.Font[1],Data.Person[DialoguePerson[DialogueSpeaker]]->color);
		set_clip_rect(Kernel.BackBuffer,0,0,w-1,h-1);
	}

	// Draw slider bar
	blit(ColorSlider,Kernel.BackBuffer,0,0,w/2-(int)SliderPos,h-24,w/2+(int)SliderPos,ColorSlider->h);
	blit(ColorSlider,Kernel.BackBuffer,0,0,-w/2-(int)SliderPos,h-24,w,ColorSlider->h);
	
	// Draw slider cursor
	draw_sprite(Kernel.BackBuffer,SliderCursor,screen->w/2-8,screen->h-24);

	// Draw status bar
	draw_sprite(Kernel.BackBuffer,StatusBar,0,0);
	Kernel.SetDirty(0,0,StatusBar->w,StatusBar->h);
	
	// Draw color
	rectfill(Kernel.BackBuffer,275,0,365,33,GetCurrentSliderColor(1,1));
	
	// Draw score
	alfont_textprintf_right_aa_ex(Kernel.BackBuffer, Data.Font[0], 535, 8, makecol(101,128,171), -1, "%i", PlayerScore);

	// Draw shield and lives
	rectfill(Kernel.BackBuffer,66,8,66+143*Player->Life/Player->Ship->MaxLife,24+8,makecol(101,128,171));
	alfont_textprintf_aa_ex(Kernel.BackBuffer, Data.Font[0], 242, 8, makecol(101,128,171), -1, "%i", PlayerLives);

	// Wave name
	if (WaveActive) {
		if (Wave[CurrentWave]->BirthTime > 0) {
			alfont_textprintf_centre_aa_ex(Kernel.BackBuffer, Data.Font[2], 320, 100, makecol(251,244,200), -1, "~ Wave %i ~", CurrentWave+1);
			alfont_textprintf_centre_aa_ex(Kernel.BackBuffer, Data.Font[3], 320, 145, makecol(251,244,200), -1, "\"%s\"", Wave[CurrentWave]->name);
			Kernel.SetDirty(50,100,590,180);
		}
	}

	// Draw as base
	BaseFrame::Render();

	// Draw gameover/highscore request
	if (GameOver) {
		int white = makecol(255,255,255);
		alfont_textout_centre_aa_ex(Kernel.BackBuffer, Data.Font[0], "Game over!", 320, 210, white, -1);
		if (Fade == 128) {
			int scorepos = HighScores.ScorePos(PlayerScore);
			if (scorepos != -1) {
				alfont_textprintf_centre_aa_ex(Kernel.BackBuffer, Data.Font[1], 320, 245, white, -1, "Your score of %i is highscore #%i! Enter your name:", PlayerScore, scorepos+1);
				alfont_textprintf_centre_aa_ex(Kernel.BackBuffer, Data.Font[1], 320, 265, white, -1, "<%s>", PlayerName);
			}
			else {
				alfont_textout_centre_aa_ex(Kernel.BackBuffer, Data.Font[0], "Press any key to continue.", 320, 255, white, -1);
			}
			if (GameWin) {
				if (Data.NewUnlock == -1) alfont_textout_centre_aa_ex(Kernel.BackBuffer, Data.Font[1], "No new characters to unlock.", 320, 285, white, -1);
				else alfont_textprintf_centre_aa_ex(Kernel.BackBuffer, Data.Font[1], 320, 285, white, -1, "Unlocked character %s!", Data.Person[Data.NewUnlock+5]->name);
			}
		}
		Kernel.SetDirty(240,200,160,110);
	}

	// Draw "pause" caption
	if (pause) {
		alfont_textout_centre_aa_ex(Kernel.BackBuffer, Data.Font[0], "~ Pause ~", 320, 230, makecol(255,255,255), -1);
		Kernel.SetDirty(240,230,160,40);
	}
}


// Get color in slider position
int PlayFrame::GetColorFromSlider (float pos) {
	return Data.HSVtoRGB(pos*360,1,1);
}


// Gets current slider color
int PlayFrame::GetCurrentSliderColor (float s,float v) {
	int pos = (int)SliderPos;
	while (pos < 0) pos += screen->w;
	pos = pos % screen->w;
	return Data.HSVtoRGB(float(pos)/float(screen->w)*360,s,v);
}


// Add sprite
void PlayFrame::AddSprite(Sprite* target) {
	target->parent = this;
	Sprites.push_back(target);
}


// Damages enemies within a certain area
void PlayFrame::DamageArea(int hue,int damage,int x,int y,int radius) {
	list<Sprite*>::iterator cur;
	for (cur=Sprites.begin();cur!=Sprites.end();cur++) {
		Enemy *current = dynamic_cast<Enemy*> (*cur);
		if (!current) continue;
		if (current->Collides(x,y,radius)) current->Damage(damage,hue);
	}
}


// Set current dialogue text
void PlayFrame::SetDialogueText(const char *text,int speaker) {
	if (Dialogue == false) play_sample(Data.Sample[0],255,128,1000,0);
	strcpy(DialogueText,text);
	DialogueSpeaker = speaker;
	DialoguePos = -35;
	Dialogue = true;
}


// Get next line
void PlayFrame::GetNextDialogueLine () {
	if (DialogueLine >= Data.Dialogue[CurrentDialogue]->n_entry) {
		Dialogue = false;
		DialogueLine = 0;
		CurrentDialogue++;
		WaveActive = true;
		return;
	}

	DialogEntry *cur = Data.Dialogue[CurrentDialogue]->entry[DialogueLine];
	DialoguePerson[0] = cur->left;
	DialoguePerson[1] = cur->right;
	SetDialogueText(cur->msg,cur->speaker);
	DialogueLine++;
}


// Counts sprites of a specific type
int PlayFrame::CountSprites(SpriteType type) {
	int count = 0;
	list<Sprite*>::iterator cur;
	for (cur=Sprites.begin();cur!=Sprites.end();cur++) {
		if ((*cur)->Type == type) count++;
	}
	return count;
}
