//
// The Lord of the Stars - Copyright (c) 2005 Rodrigo Braz Monteiro
// Rodrigo Braz Monteiro's entry for TINS'05
// This source file is subject to GNU General Public License
// See included license.txt for more information
//
// You can reach the author at:
// http://zeratul.cellosoft.com
// zeratul@cellosoft.com
//

#include <allegro.h>
#include "kernel.h"
#include "baseframe.h"
#include "audio.h"
#include "playframe.h"
#include "selectframe.h"
#include "titleframe.h"
#include "data.h"
#include "input.h"
#include "introframe.h"


// Constructor
KernelClass::KernelClass () {
	// Init vars
	Run = true;
	CurrentFrame = NULL;
	BackBuffer = NULL;
	SetGraphicsMode = false;

	// Init dirty rects
	for (int i=0;i<80;i++) {
		for (int j=0;j<60;j++) {
			DirtyRects[i][j] = 0;
		}
	}
}


// Destructor
KernelClass::~KernelClass() {
	// Destroy bitmaps
	if (BackBuffer)	destroy_bitmap(BackBuffer);
	BackBuffer = NULL;
}


// Blit screen
void KernelClass::BlitScreen () {
	if (!BackBuffer) return;

	int dx,dy;
	acquire_screen();

	// Draw all
	if (RedrawAll) {
		blit (BackBuffer,screen,0,0,0,0,screen->w,screen->h);
		RedrawAll = false;
	}

	// Dirty rects
	else {
		for (int i=0;i<80;i++) {
			for (int j=0;j<60;j++) {
				if (DirtyRects[i][j] > 0) {
					DirtyRects[i][j]--;
					dx = i<<3;
					dy = j<<3;
					blit (BackBuffer,screen,dx,dy,dx,dy,8,8);
				}
			}
		}
	}

	release_screen();
}


// UPDATE CURRENT FRAME
bool KernelClass::UpdateCurrentFrame() {
	if (CurrentFrame) return CurrentFrame->Update();
	else return false;
}


// RENDER CURRENT FRAME
void KernelClass::RenderCurrentFrame() {
	if (CurrentFrame) CurrentFrame->Render();
}


// TIMER FUNCTION
volatile int Tick = 0;
void DoTick () { 
	Tick++;
}
END_OF_FUNCTION(DoTick)


// MAIN LOOP
void KernelClass::Loop () {
	// Install Allegro timer to have constant fps
	LOCK_VARIABLE(Tick);
    LOCK_FUNCTION(DoTick);
	install_int_ex(DoTick, BPS_TO_TIMER(60));

	// Set Run = false to quit
	bool refresh = true;
	while (Run) {
		while (Tick > 0) {
			poll_mouse();
			Audio.Update();
			Input.Update();
			refresh = UpdateCurrentFrame();
			if (!refresh) NextFrame();
			Tick--;
		}
		if (refresh) {
			if (SetGraphicsMode) {
				//set_color_depth(Data.ColorDepth);
				int winW = 640;
				int winH = 480;
				int mode;
				if (Data.FullScreen == 1) mode = GFX_AUTODETECT_FULLSCREEN;
				else mode = GFX_AUTODETECT_WINDOWED;
				set_gfx_mode(mode, winW, winH, 0, 0);
				SetGraphicsMode = false;
			}
			RenderCurrentFrame();
			BlitScreen();
			refresh = false;
		}
		else rest(0);
	}

	// Remove Allegro timer
	remove_int(DoTick);
}


// Sets dirty rectangles
void KernelClass::SetDirty(int x,int y,int w,int h) {
	int rx1,ry1,rx2,ry2;
	rx1 = MIN(MAX(0,x/8),79);
	ry1 = MIN(MAX(0,y/8),59);
	rx2 = MIN(MAX(0,(x+w)/8),79);
	ry2 = MIN(MAX(0,(y+h)/8),59);

	for (int i=rx1;i<=rx2;i++) {
		for (int j=ry1;j<=ry2;j++) {
			DirtyRects[i][j] = 2;
		}
	}
}


// Set next frame
void KernelClass::NextFrame() {
	if (CurrentFrame) delete CurrentFrame;
	CurrentFrame = NULL;

	switch (SwitchNext) {
		case -1:
			Run = false;
			break;
		case 0:
			CurrentFrame = new SelectFrame;
			break;
		case 1:
			CurrentFrame = new PlayFrame;
			break;
		case 2:
			CurrentFrame = new TitleFrame;
			break;
		case 3:
			CurrentFrame = new IntroFrame;
			break;
	}
}
