
#include <allegro.h>
#include <fstream.h>
#include <strstream.h>
#include <iomanip.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>

#include "synth.h"

// --- IN SITU PROFILER --------------------------------------------------

int profile_section = 0;
int profile_counter[2] = { 0, 0 };
int profile_enable = 0;

void timer()
{ if( profile_enable ) profile_counter[ profile_section ]++; }
END_OF_FUNCTION( timer )

// --- AUDIO UPDATE FUNCTION ---------------------------------------------

csequencer *seq[16];
int nseq = 0;
int enhance = 0;
int save = 0, last = 0;

void load_score_string( const char *text )
{
   istrstream is( text, strlen( text ) );
   is.width(80);

   nseq = 0;
   while( !is.eof() || nseq == 16 )
   {
      char buf[81];
      is >> buf;
      if( !strcmp( buf, "%" ) )
      {
         istrstream *sis = new istrstream( text, strlen( text ) );
         sis->seekg( is.tellg() );
         seq[ nseq++ ] = new csequencer( *sis );
      }
   }
}

// This is a workaround of a %$@(!! bug in gcc's
// file-stream code, because stream positioning
// does not work. I expected to pass file streams
// to the sequencer (that's all the reason I did
// it with streams, so I can pass string-streams
// and file-streams), but it doesn't work, so
// I load the text to memory and go from there:

void load_score_file( const char *filename )
{
   ifstream is( filename );
   int size = is.tellg();
   is.width(80);
   while( !is.eof() ) { char buf[81]; is >> buf; }
   size = is.tellg() - size;
   char *text = new char[ size + 1 ];

   is.close();
   is.open( filename );
   is.read( text, size );
   text[is.gcount()] = 0;
   load_score_string( text );
}

fstream record;
short recbuffer[ sb_times * sb_size * 2 ];

cdelay *effect = 0;
int effect_mix = 0;

void audio_update( short *sb, int times )
{
   short *pr = recbuffer;
   for( int n = 0; n < times; ++n )
   {
      sbuffer tmp = zero_sbuffer;
      for( int i = 0; i < nseq; ++i ) tmp = tmp + seq[i]->pace();

      if( effect )
         tmp = globalgain * ( 16384 - effect_mix / 2 ) / 10000 * tmp 
             + globalgain * ( effect_mix / 12 ) / 10000 * effect->pace( tmp );
      else
         tmp = globalgain * 16384 / 10000 * tmp;

      switch( enhance )
      {
         case 1:
         for( int u = 0; u < sb_size; ++u )
            { save = tmp.buf[u]; tmp.buf[u] += ( tmp.buf[u] - last ) >> 1; last = save; }
         break;

         case 2:
         for( int u = 0; u < sb_size; ++u )
            { save = tmp.buf[u]; tmp.buf[u] += tmp.buf[u] - last; last = save; }
         break;
      }

      if( !sb )
      for( int u = 0; u < sb_size; ++u )
      {
         if( tmp.buf[u] > 32767 ) tmp.buf[u] = 32767;
         if( tmp.buf[u] < -32768 ) tmp.buf[u] = -32768;
         *pr++ = (short) tmp.buf[u];
      }

      else
      for( int u = 0; u < sb_size; ++u )
      {
         if( tmp.buf[u] > 32767 ) tmp.buf[u] = 32767;
         if( tmp.buf[u] < -32768 ) tmp.buf[u] = -32768;
         *sb++ = ( *pr++ = (short) tmp.buf[u] ) + 32768;
      }
   }
   record.write( (char*) recbuffer, 2 * sb_times * sb_size );
};

// --- MAIN --------------------------------------------------------------

char cfg[] = "
[sound]
   quality = 1
   digi_voices = 1
";

int main( int argc, char **argv )
{
   cout << "\n*** Christian's virtual synthesizer ***\n"
        << "Version 1.3\n";

   char *inputfile = 0;
   char *outputfile = 0;
   int fin = false;
   int profile_average[2] = { 0, 0 };
   effect = 0;
   effect_mix = 0;

   bool quiet = false;
   int trans = 0;
   int reverb_preset = -1;
   int reverb_mix = -1;
   timestretch = 10000;
   int gain = 0;
   bool rms = false, gs = false, ts = false, trs = false,
        es = false, msrs = false, rps = false;

   for( int i = 1; i < argc; ++i )
   {
      if( !strcmp( argv[i], "-q" ) ) quiet = true;
      else if( !strncmp( argv[i], "-f", 2 ) ) { msrs = true; master_sample_rate = atoi( argv[i] + 2 ); }
      else if( !strncmp( argv[i], "-t", 2 ) ) { trs = true; trans = atoi( argv[i] + 2 ); }
      else if( !strncmp( argv[i], "-e", 2 ) ) { es = true; enhance = atoi( argv[i] + 2 ); }
      else if( !strncmp( argv[i], "-rp", 3 ) ) { rps = true; reverb_preset = atoi( argv[i] + 3 ); }
      else if( !strncmp( argv[i], "-rm", 3 ) ) { rms = true; reverb_mix = atoi( argv[i] + 3 ); }
      else if( !strncmp( argv[i], "-T", 2 ) ) { ts = true; timestretch = atoi( argv[i] + 2 ); }
      else if( !strncmp( argv[i], "-g", 2 ) ) { gs = true; gain = atoi( argv[i] + 2 ) ; }
      else if( !inputfile ) inputfile = argv[i];
      else if( !outputfile ) outputfile = argv[i];
   }
   if( gain > 20 ) gain = 20;
   if( gain < -20 ) gain = -20;
   globalgain = int( 10000 * pow( 10, double( gain ) / 20 ) );

   if( !inputfile )
   {
      cout << "Usage: synth [options] input [output]\n"
              "\n"
              "-q       quiet: do not produce realtime output\n"
              "-fx      set sample frequency to x Hz\n"
              "-tx      set global transpose to x semitones\n"
              "-Tx      set global tempo stretch to x/10000\n"
              "-gx      set global gain to x dB\n"
              "-ex      set enhancer 0 = off, 1 or 2\n"
              "-rpx     set reverbenator to preset 1..6\n"
              "-rmx     set reverbenator mix to 0..8\n"
              "input    input score file\n"
              "output   output wave file\n";
      return true;
   }

   else
   {
      ifstream is( inputfile );
      if( !is )
      {
         cout << "Unable to load file: "
              << inputfile << endl;
         return true;
      }

      is.width(80);
      char buf[81];

      while( !is.eof() )
      {
         is >> buf;
         if( !strcmp( buf, "@reverbenator" ) && !rps )
         {
            int a, b, c, d, e, f;
            double g, h;
            is >> a >> b >> c >> d >> e >> f >> g >> h;
            effect = new cdelay( a, b, c, d, e, f );
            effect->set_coeff( g, h );
            is >> effect_mix;
         }

         else if( !strcmp( "@gain", buf ) && !gs )
         {
            is >> gain;
            if( gain > 20 ) gain = 20;
            if( gain < -20 ) gain = -20;
            globalgain = int( 10000 * pow( 10, double( gain ) / 20 ) );
         }

         else if( !strcmp( "@tempostretch", buf ) && !ts ) is >> timestretch;
    	 else if( !strcmp( "@enhancer", buf ) && !es ) is >> enhance;
     	 else if( !strcmp( "@transpose", buf ) && !trs ) is >> trans;
         else if( !strcmp( "@samplerate", buf ) && !msrs ) is >> master_sample_rate;
      }

      if( rps )
      switch( reverb_preset )
      {
         case 1:
         effect = new cdelay( 180, 254, 402, 311, 440, 623 );
         effect->set_coeff( 500, 15000 );
         effect_mix = 22000;
         break;

         case 2:
         effect = new cdelay( 351, 445, 657, 771, 932, 1153 );
         effect->set_coeff( 2000, 10000 );
         effect_mix = 20000;
         break;

         case 3:
         effect = new cdelay( 1322, 1493, 1732, 1912, 1993, 2051 );
         effect->set_coeff( 3000, 16000 );
         effect_mix = 18000;
         break;

         case 4:
         effect = new cdelay( 742, 893, 932, 1229, 1393, 1598 );
         effect->set_coeff( 5500, 11000 );
         effect_mix = 16000;
         break;

         case 5:
         effect = new cdelay( 1393, 1727, 1865, 1943, 2175, 2257 );
         effect->set_coeff( 6000, 16500 );
         effect_mix = 16000;
         break;

	 case 6:
         effect = new cdelay( 100, 103, 111, 117, 121, 129 );
         effect->set_coeff( 4500, 8000 );
         effect_mix = 20000;
         break;
      }

      if( rms )
      {
         if( reverb_mix < 0 ) reverb_mix = 0;
         if( reverb_mix > 8 ) reverb_mix = 8;
         effect_mix = ( reverb_mix * reverb_mix + 8 * reverb_mix ) * 256;
      }

      load_score_file( inputfile );
   }

   if( outputfile ) record.open( outputfile, ios::out + ios::binary );

   if( trans > 15 ) trans = 15;
   if( trans < -15 ) trans = -15;

   // ------------------------------

   LOCK_VARIABLE( profile_enable );
   LOCK_VARIABLE( profile_section );
   LOCK_VARIABLE( profile_counter );
   LOCK_FUNCTION( timer );

   AUDIOSTREAM *stream = 0;
   short *buffer = 0;

   allegro_init();
   install_keyboard();
   install_timer();
   install_int_ex( timer, BPS_TO_TIMER( 1000 ) );
   set_config_data( cfg, strlen( cfg ) );
   reserve_voices( 1, 0 );

   if( !quiet || !master_sample_rate )
   {
      if( install_sound( -1 , 0 , 0 ) )
         { cout << "Could not install sound driver, baling out." << endl; return true; }
      set_volume( 255, 0 );
    
      if( !master_sample_rate )
      switch( digi_driver->id )
      {
         #ifdef DIGI_SB16
         case DIGI_SB16:
         master_sample_rate = 22727;
         break;
         #endif
    
         #ifdef DIGI_AUDIODRIVE
         case DIGI_AUDIODRIVE:
         master_sample_rate = 22729;
         break;
         #endif
    
         default:
         master_sample_rate = 22050;
      }
    
      cout << digi_driver->desc << endl << endl;
      if( master_sample_rate < 5000 ) master_sample_rate = 5000;

      stream = play_audio_stream(
                  sb_times * sb_size,
                  16,
                  0,
                  master_sample_rate,
                  255,
                  128 );

      if( !stream )
         { cout << "Could not install audiostream for live output, bailing out."
         << endl; return true; };
   }

   if( trans >= 0 ) for( int i = 0; i < 7; ++i )
      { notes[i] = 440 * 8 * 65536 / master_sample_rate * notes[i] / 10000 * semi[ trans ] / 10000; }
   else for( int i = 0; i < 7; ++i )
      { notes[i] = 440 * 8 * 65536 / master_sample_rate * notes[i] / 10000 * 10000 / semi[ -trans ]; }

   if( quiet ) remove_sound();

   // ---------------------------------

   cout << "Current CPU usage:    " << flush;
   while( !key[ 59 ] && !fin )
   {
      if( !quiet )
      { if( ( buffer = (short *) get_audio_stream_buffer( stream ) ) ) goto run; }

      else
      {
         buffer = 0;

         run:
         profile_enable = 0;
         if( profile_counter[0] + profile_counter[1] )
            cout << "\b\b\b\b" << setw(3) << 100 * profile_counter[0] / ( profile_counter[0] + profile_counter[1] )
                 << '%' << flush;
         else
            cout << "\b\b\b\b N/A" << flush;
         profile_average[0] += profile_counter[0]; profile_average[1] += profile_counter[1];
         profile_counter[0] = profile_counter[1] = 0;
         profile_enable = 1;
   
         profile_section = 0;
         audio_update( buffer, sb_times );
         free_audio_stream_buffer( stream );
   
         profile_section = 1;
         yield_timeslice();
   
         fin = true;
         for( int i = 0; i < nseq; ++i ) fin &= !seq[i]->play;
      }
   }

   if( profile_average[0] + profile_average[1] )
      cout << "\nAverage CPU usage: "
           << float( 10000 * profile_average[0] / ( profile_average[0] + profile_average[1] ) ) / 100
           << '%' << endl;

   return false;
}

END_OF_MAIN()

