
// Christian's virtual synthesizer
// (c) 2000 Christian Schler

#include <allegro.h>
#include <istream.h>
#include <strstream.h>
#include <string.h>
#include <math.h>

#include "synth.h"

int master_sample_rate;
int timestretch;
int globalgain;

sbuffer zero_sbuffer = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0  };
sbuffer unit_sbuffer = { 32768, 32768, 32768, 32768, 32768, 32768, 32768, 32768, 32768, 32768, 32768, 32768, 32768, 32768, 32768, 32768  };

int rate_values[16] = { 1, 2, 3, 4, 6, 10, 17, 29, 49, 83, 140, 240, 410, 700, 1200, 2000 };
int levela_values[16] = { 0, 830, 1115, 1400, 1900, 2400, 3125, 4100, 5550, 7000, 9500, 12000, 16000, 20000, 26000, 32700 };
int levelf_values[16] = { 1, 2, 4, 8, 16, 32, 64, 128, 256, 512, 1024, 2048, 4096, 8192, 16384, 32768 };
int qfactor_values[16] = { 0, 500, 700, 1000, 1500, 2100, 2600, 3100, 3400, 3600, 3800, 3900, 3950, 4000, 4030, 4050 };

int semi[16] =
   { 10000, 10596, 11225, 11892, 12599, 13348, 11414, 14983, 15874, 16818, 17818,
     18877, 20000, 21192, 22450, 23784 };

int notes[7] =
   { 10000, 11225, 11892, 13348, 14983, 15874, 17818 };

// --- GLOBAL UTILITY ----------------------------------------------------

int val( istream &is )
{
   char c;
   is >> c;
   c = c - '0';
   if( c > 9 ) c = c - 39;
   return  c;
}

// --- BASIC BUFFER OPERATIONS -------------------------------------------

sbuffer operator +( const sbuffer &A, const sbuffer &B ) return res;
{ for( int u = 0; u < sb_size; ++u ) res.buf[u] = ( A.buf[u] + B.buf[u] ) >> 0; }

sbuffer operator -( const sbuffer &A, const sbuffer &B ) return res;
{ for( int u = 0; u < sb_size; ++u ) res.buf[u] = ( A.buf[u] - B.buf[u] ) >> 0; }

sbuffer operator *( const sbuffer &A, const sbuffer &B ) return res;
{ for( int u = 0; u < sb_size; ++u ) res.buf[u] = ( A.buf[u] * B.buf[u] ) >> 15; }

sbuffer operator *( int A, const sbuffer  &B ) return  res;
{ for( int u = 0; u < sb_size; ++u )  res.buf[u] = ( A * B.buf[u] ) >> 15; }

// --- SINUS TABLE -------------------------------------------------------

csinus_table::csinus_table()
{
   for( int i = 0; i < 65536; ++i )
      y[i] = int( 262143.0 * sin( double( 2 * i ) * 3.14159265358979323846 / 65536 ) );
}

csinus_table sinus;

// --- BAND LIMITED IMPULSE TRAIN ----------------------------------------

sbuffer cBLIT::pace() return res
{
   if( freq )
   {
     #if 1
  
      int M = 16384 / freq;
      if( n ) if( M > n ) M = n;
      M = M * 2 + 1;

      int aa;
      if( !a ) aa = 4096 / M; else aa = a;

      int p = phase | 1,
          f = freq & 32766,
          P = M * p,
          F = M * f;

      if( ( p + ( f << sb_shift ) ) < 32768 )
      for( int u = 0; u < sb_size; ++u ) 
         res.buf[u] = aa * sinus.y[ ( P + u * F ) & 65535 ] / sinus.y[ p + u * f ] - aa;

      else
      for( int u = 0; u < sb_size; ++u ) 
      {
         if( p > 32767 ) { p &= 32767; P = M * p; }
         res.buf[u] = aa * sinus.y[ P & 65535 ] / sinus.y[ p ] - aa;
         p += f; P += F;
      }

      phase = ( phase + ( f << ( sb_shift ) ) ) & 65535;

     #else

      int M = 16384 / freq + 1;
      if( n ) if( M > n ) M = n;
      int sf = freq * 2048 >> 12;//fsqrt( freq << 8 ) >> 8;


      for( int u = 0; u < sb_size; ++u )
      {
         res.buf[u] = sf * ( sinus.y[( 2 * phase ) & 65535 ] 
                             - sinus.y[ ( 2 * M * phase ) & 65535 ]
                             + sinus.y[ ( 2 * ( M - 1 ) * phase ) & 65535 ] )
                      /
                      ( 524288 - 2 * sinus.y[ ( 16384 + 2 * phase ) & 65535 ] );

         phase = ( freq + phase ) & 65535;
      }

     #endif

   }


   else
      for( int u = 0; u < sb_size; ++u ) res.buf[u] = 0;
}

// --- ANALOG WAVEFORM GENERATOR -----------------------------------------

sbuffer cwave::pace() return res
{
   BLIT[0].n = BLIT[1].n = n;
   BLIT[1].freq = m * ( BLIT[0].freq = freq );
   BLIT[1].phase = ( m * BLIT[0].phase + width ) & 65535;

   if( !( type & 6 ) ) BLIT[0].a = BLIT[1].a = 0;
   else BLIT[1].a = BLIT[0].a = freq >> 2;

   if( type & 1 )
      res = BLIT[0].pace() - BLIT[1].pace();

   else
      res = BLIT[0].pace() + BLIT[1].pace();

   if( type & 6 )
   {
      int q = ( 65536 - 2 * freq ) >> 8;
      for( int u = 0; u < sb_size; ++u ) res.buf[u] = s = ( q * s >> 8 ) + res.buf[u];
      if( type & 4 ) for( int u = 0; u < sb_size; ++u ) res.buf[u] = s2 = ( q * s2 >> 8 ) + ( freq * res.buf[u] >> 12 );
   }
}

// --- ENVELOPE GENERATOR ------------------------------------------------

sbuffer cenvelope::pace( const sbuffer  &in ) return  res
{
   int delta =
      ( ( ay -= ar * ay >> 12 ) + ( ( dy += dr * ( sl - dy ) >> 12 ) >> 6 ) - y ) >> sb_shift;

   for( int u = 0; u < sb_size; ++u ) res.buf[u] = ( y += delta ) * in.buf[u] >> 15;
   if( !y ) active = false;
}

// --- FILTER ------------------------------------------------------------

sbuffer cfilter::pace( const sbuffer &in ) return res
{ for( int u = 0; u < sb_size; ++u )
     res.buf[u] = x = ( s += ( v = f * ( in.buf[u] - x )
        + int( (long long)(q) * (long long)(v) / 4096 ) ) ) >> 14; }

// --- MONOPHONIC SYNTH VOICE --------------------------------------------


cvoice_mono::cvoice_mono()
{
   memset( &freq, 0, sizeof( int ) * ( &pwf - &freq ) );
   trans = 10000;
   static char init[] = "iw 2 ip 00 ia ffff if ffff ic f0 iv 000";
   istrstream is( init, strlen( init ) );
   while( !is.eof() ) command( is );
};

void cvoice_mono::trig( int p )
{
   aenv.ar = aar;
   aenv.dr = adr;
   aenv.trig();
   aenv.kick( p );
   aenv.sets( asl );
   fenv.ar = far;
   fenv.dr = fdr;
   fenv.trig();
   fenv.kick( p );
   fenv.sets( fsl );
   vc = vd;
   vp = 0;
   pw = pws;
   pwc = 0;
}

void cvoice_mono::rel()
{
   aenv.ar = arr;
   aenv.dr = arr;
   aenv.sets( 0 );
   fenv.ar = frr;
   fenv.dr = frr;
   fenv.sets( 0 );
}

sbuffer cvoice_mono::pace() return res
{
   if( aenv.active )
   {
      if( vc )
         { wave.freq = freq; vc--; }
      else
         { wave.freq = freq + ( ( freq * vl >> 9 ) * sinus.y[ vp = ( vp + vf ) & 65535 ] >> 18 ); }

      pw = pwc < 0 ? ( pw -= pwm ) : ( pw += pwm );
      if( (++pwc) >= pwu ) pwc = - pwd;

      if( pwf ) wave.width = freq * pw / pwf;
      else wave.width = pw;

      filter.f = ff * fenv.pace() >> 15;
      filter.q = int( 4096 - ( fq * fsqrt( filter.f ) >> 16 ) );
      filter.f = filter.f * ( 4096 + filter.q ) >> 13;

      res = filter.pace( aenv.pace( wave.pace() ) );
   }

   else if( fenv.active )
   {
      filter.f = ff * fenv.pace() >> 15;
      filter.q = int( 4096 - ( fq * fsqrt( filter.f ) >> 16 ) );
      filter.f = filter.f * ( 4096 + filter.q ) >> 13;

      res = filter.pace( zero_sbuffer );
   }

   else
      res = zero_sbuffer;
}

void cvoice_mono::command( istream &is )
{
   char c;
   switch( is >> c, c )
   {
      case '+':
      { int p = 16384;
        if( is >> c, c == '+' ) p += 8192;
        else if( c == '-' ) p -= 8192;
        else is.putback( c );
        trig( p ); }
      break;

      case '-':
      rel();
      break;

      case 't':
      if( is >> c, c == '-' ) trans = 100000000 / semi[ val( is ) ];
      else { is.putback( c ); trans = semi[ val( is ) ]; }
      break;

      case 'i':
      switch( is >> c, c )
      {
         case 'a':
         aar = rate_values[ val( is ) ];
         adr = rate_values[ val( is ) ];
         asl = levela_values[ val( is ) ];
         arr = rate_values[ val( is ) ];
         break;

         case 'f':
         far = rate_values[ val( is ) ];
         fdr = rate_values[ val( is ) ];
         fsl = levelf_values[ val( is ) ];
         frr = rate_values[ val( is ) ];
         break;

         case 'c':
         ff = rate_values[ val( is ) ] * 32;
         fq = 8192 - 2 * qfactor_values[ val( is ) ];
         break;

         case 'h':
         is >> wave.n;
         break;

         case 'm':
         is >> wave.m;
         break;
    
         case 'p':
         pws = val( is ) * 2048;
         pwm = rate_values [ val( is ) ]  - 1;
         if( is >> c, c == '-' ) { pwu = 25 * val( is ); pwd = 25 * val( is ); }
         else { is.putback( c ); pwu = pwd = 0; }
         if( is >> c, c == '+' ) pwf = 100 * val( is );
         else { is.putback( c ); pwf = 0; }
         break;

         case 'v':
         vd = val( is ) * 100;
         vf = val( is ) * 30;
         vl = val( is ) * 6;
         break;

         case 'w':
         wave.type = val( is );
         break;
      }

      default:
      if( c >= 'a' && c <= 'g' )
      {
         freq = trans * notes[ c - 'a' ] / 10000;
         if( is >> c, c == '#' ) freq = semi[1] * freq / 10000;
         else is. putback ( c );
         if( is >> c,   c >= '0' && c < '9'  ) oct = '9' - c;
         else is. putback ( c );
         freq = freq >> oct;
      }
      break;
   }
}

// --- CHORD VOICE -------------------------------------------------------

void cchord::command( istream &is )
{
   char c;
   int p = is.tellg();
   switch( is >> c, c )
   {
      case '=':
      if( is >> c, c == '-' ) voice[0].freq = 10000 * freq / semi[ val( is ) ];
      else { is.putback( c ); voice[0].freq = semi[ val( is ) ] * freq / 10000; }
      if( is >> c, c == '-' ) voice[1].freq = 10000 * freq / semi[ val( is ) ];
      else { is.putback( c ); voice[1].freq = semi[ val( is ) ] * freq / 10000; }
      if( is >> c, c == '-' ) voice[2].freq = 10000 * freq / semi[ val( is ) ];
      else { is.putback( c ); voice[2].freq = semi[ val( is ) ] * freq / 10000; }
      break;

      case 'i':
      switch( is >> c, c )
      {
         case 'b':
         is >> boost;
         break;

         default:
         goto distribute;
      }
      break;

      default:
      if( c >= 'a' && c <= 'g' )
      {
         freq = notes[ c - 'a' ];
         if( is >> c, c == '#' ) freq = semi[1] * freq / 10000;
         else is. putback (c);
         if( is >> c, c >= '0' && c < '9' ) oct = '9' - c;
         else is. putback (c);
         freq = freq >> oct;
      }

      else
      {
         distribute:
         is.seekg( p ); voice[0].command( is );
         is.seekg( p ); voice[1].command( is );
         is.seekg( p ); voice[2].command( is );
      }
   }
}

sbuffer cchord::pace() return res
{
   res = voice[0].pace() + voice[1].pace() + voice[2].pace();

   if( boost )
   {
      for( int u = 0; u < sb_size; ++u )
      {
         res.buf[u] = boost * res.buf[u];
         res.buf[u] = 
            int( res.buf[u] > 0 ?
               32768LL * (long long) ( res.buf[u] ) / ( 32768LL + (long long) ( res.buf[u] ) )
             : 32768LL * (long long) ( res.buf[u] ) / ( 32768LL - (long long) ( res.buf[u] ) ) 
            ) >> 3;
      }
   }
}

// --- DRUM MACHINE ------------------------------------------------------

cdrums::cdrums()
{
   base_aenv.ar = 1000;
   base_aenv.dr = 300;
   base_fenv.ar = 400;
   base_fenv.dr = 100;
   base_fenv.sets( 1 );
   base_filter.q = 4093;
   base_kick = 16384;
   base_sens = 50;
   base_noise = 40000;
   snare_aenv.ar = 2000;
   snare_aenv.dr = 100;
   snare_filter.f = 50;
   snare_filter.q = 4085;
   snare_kick = 20000;
   snare_noise = 32000;
   hhat_aenv.ar = 2000;
   hhat_open_dr = 10;
   hhat_close_dr = 150;
   hhat_filter.f = 30000;
   hhat_filter.q = 500;
   hhat_kick = 12000;
}

void cdrums::command( istream &is )
{
   char c;
   switch( is >> c, c )
   {
      case 'b':
      base_aenv.kick( base_kick );
      base_fenv.kick( base_sens );
      break;

      case 's':
      snare_aenv.kick( snare_kick );
      break;

      case 'h':
      if( is >> c, c == 'o' ) hhat_aenv.dr = hhat_open_dr;
      else if( c == 'c' ) hhat_aenv.dr = hhat_close_dr;
      else is.putback( c );
      hhat_aenv.kick( hhat_kick );
      break;

      case 'i':
      switch( is >> c, c )
      {
         case 'b':
         is >> base_aenv.ar >> base_aenv.dr >> base_fenv.ar >> base_fenv.dr
            >> base_filter.q >> base_kick >> base_sens >> base_noise;
         break;

         case 's':
         is >> snare_aenv.ar >> snare_aenv.dr >> snare_filter.f
            >> snare_filter.q >> snare_kick >> snare_noise;
         break;

         case 'h':
         is >> hhat_aenv.ar >> hhat_close_dr >> hhat_open_dr
            >> hhat_filter.f >> hhat_filter.q >> hhat_kick;
         break;
      }
   }
}

sbuffer cdrums::pace() return res
{
   sbuffer nse;
   for( int u = 0; u < sb_size; ++u ) nse.buf[u] = ( ( rand() & 8191 ) - 4096 );

   res = zero_sbuffer;

   if( base_aenv.active || base_filter.x )
   {
      base_filter.f = base_fenv.pace();
      sbuffer aeb = base_aenv.pace( unit_sbuffer );
      res = res + base_filter.pace( aeb ) - aeb + base_noise * aeb * nse;
   }

   if( snare_aenv.active || snare_filter.x )
   {
      sbuffer aes = snare_aenv.pace( unit_sbuffer );
      res = res + 262144 * snare_filter.pace( aes * nse ) + snare_noise * aes * nse;
   }

   if( hhat_aenv.active )
   {
      sbuffer flh = hhat_filter.pace( nse );
      res = res + hhat_aenv.pace( flh - nse );
   }
}

// --- SEQUENCER TRACK ---------------------------------------------------

extern csequencer *seq[16];

sbuffer csequencer::pace() return res
{
   if( play )
   {
      count -= beat;
      countmodulo -= beatmodulo;
      if( countmodulo < 0 ) 
      { 
         countmodulo += master_sample_rate * 60;
         count--;
      }


      while( play && count < 0 )
      {
         char c;
         switch( is >> c, c )
         {
            case '%':
            play = false;
            break;

            case 'l':
            level = val( is ) * 4096;
            break;

            case 'r':
            is >> ring;
            break;

            case 'm':
            is >> mute;
            break;

            case '@':
            if( is >> c, c == 'M' ) { delete voice; voice = new cvoice_mono; }
            else if( c == 'C' ) { delete voice; voice = new cchord; }
            else if( c == 'D' ) { delete voice; voice = new cdrums; }
            else { is. putback ( c ); int a, b; is >> a >> b; 
                   beat = ( a * b * sb_size * timestretch / 10000 ) / ( master_sample_rate * 60 );
                   beatmodulo = ( a * b * sb_size * timestretch / 10000 ) % ( master_sample_rate * 60 ); 
		   count = countmodulo = 0; }
            break;

            case '{':
            is >> c;
            block_p[ c - 'A' ] = is.tellg();
            while( !is.eof() && c != '}' ) is >> c;
            break;

            case '}':
            is.seekg( return_p[ --return_count ] );
            break;
   
            default:
            if( c >= '1' && c <= '8' ) count += ( c - '0' );
            else if( c >= 'A' && c <= 'Z' ) { return_p[ return_count++ ] = is.tellg(); is.seekg( block_p[ c - 'A' ] ); }
            else { is. putback ( c ); voice->command( is ); }
         }
         
         if( is.eof() ) play = false;
      }
   }

   if( mute ) 
      { out = level * voice->pace(); res = zero_sbuffer; }

   else
   {
      if( !ring ) 
         res = out = level * voice->pace();

      else 
      { 
         out = voice->pace(); 
         for( int u = 0; u < sb_size; ++u ) 
            res.buf[u] = ( level * seq[ ring - 1 ]->out.buf[u] >> 17 ) * out.buf[u] >> 9; 
      }
   }
}

// ------------------------------------------------------------------

cdelay::cdelay( int an1, int an2, int an3, int an4, int an5, int an6 )
{
   buffer1 = new int[ n1 = an1 & 0xffffffff ];
   buffer2 = new int[ n2 = an2 & 0xffffffff ];
   buffer3 = new int[ n3 = an3 & 0xffffffff ];
   buffer4 = new int[ n4 = an4 & 0xffffffff ];
   buffer5 = new int[ n5 = an5 & 0xffffffff ];
   buffer6 = new int[ n6 = an6 & 0xffffffff ];
   

   memset( buffer1, 0, n1 * 4 );
   memset( buffer2, 0, n2 * 4 );
   memset( buffer3, 0, n3 * 4 );
   memset( buffer4, 0, n4 * 4 );
   memset( buffer5, 0, n5 * 4 );
   memset( buffer6, 0, n6 * 4 );

   i1 = i2 = i3 = i4 = i5 = i6 = c1 = c2 = c3 = c4 = c5 = c6 = 0;
   low1 = low2 = low3 = low4 = low5 = low6 = 0; pole = 16384;
}

cdelay::~cdelay()
{
   delete[] buffer1;
   delete[] buffer2;
   delete[] buffer3;
   delete[] buffer4;
   delete[] buffer5;
   delete[] buffer6;
}

sbuffer cdelay::pace( const sbuffer &in ) return res
{
   for( int u = 0; u < sb_size; ++u )
   {
      res.buf[u] =
         buffer1[i1] + buffer2[i2] + buffer3[i3] +
         buffer4[i4] + buffer5[i5] + buffer6[i6];

      low1 += pole * ( buffer1[i1] - low1 ) >> 14; 
      low2 += pole * ( buffer2[i2] - low2 ) >> 14; 
      low3 += pole * ( buffer3[i3] - low3 ) >> 14; 
      low4 += pole * ( buffer4[i4] - low4 ) >> 14; 
      low5 += pole * ( buffer5[i5] - low5 ) >> 14; 
      low6 += pole * ( buffer6[i6] - low6 ) >> 14; 
      buffer1[i1++] = ( low1 * c1 >> 14 ) + in.buf[u];      
      buffer2[i2++] = ( low2 * c2 >> 14 ) + in.buf[u]; 
      buffer3[i3++] = ( low3 * c3 >> 14 ) + in.buf[u];      
      buffer4[i4++] = ( low4 * c4 >> 14 ) + in.buf[u];
      buffer5[i5++] = ( low5 * c5 >> 14 ) + in.buf[u];
      buffer6[i6++] = ( low6 * c6 >> 14 ) + in.buf[u];
      if( i1 == n1 ) i1 = 0;
      if( i2 == n2 ) i2 = 0;
      if( i3 == n3 ) i3 = 0;
      if( i4 == n4 ) i4 = 0;
      if( i5 == n5 ) i5 = 0;
      if( i6 == n6 ) i6 = 0;
   }
}

void cdelay::set_coeff( double halflife, double pole )
{
   this->pole = int( pole );
   double f = pole < 16384 ? 1 : double( 32768 - pole ) / double( pole );
   c1 = int( 16384 * f * exp( - double( n1 ) / halflife * M_LN2 ) );
   c2 = int( 16384 * f * exp( - double( n2 ) / halflife * M_LN2 ) );
   c3 = int( 16384 * f * exp( - double( n3 ) / halflife * M_LN2 ) );
   c4 = int( 16384 * f * exp( - double( n4 ) / halflife * M_LN2 ) );
   c5 = int( 16384 * f * exp( - double( n5 ) / halflife * M_LN2 ) );
   c6 = int( 16384 * f * exp( - double( n6 ) / halflife * M_LN2 ) );
}

