/*
 *
 *   ^   |    sssss p   ddddd  fff  ggggg hhhh   iii  j   j    |   ^
 *  /|\  |    s     p   d     f   f   g   h   h i   i jj  j    |  /|\
 *   |   |    sss   p   ddd   f       g   hhhh  i   i j j j    |   |
 *   |  \|/   s     p   d     f   f   g   h   h i   i j  jj   \|/  |
 *   |   v    sssss ppp ddddd  fff    g   h   h  iii  j   j    v   |
 *
 *                           copyright 1999
 *                  Martijn Versteegh & Hein Zelle
 *
 */

/*
 *       user.cc
 *
 *       class implementation of a User (player)
 */

#include "user.h"

// include freelist.h here to prevent recursive include of user.h
#include "globals.h"
#include "usercomm.h"
#include "message.h"

User::User()
{
    freelist_index = SERVER_FREELIST_USER;

    player = 0;
    plug = 0;
    state = SHAKE_HANGUP;

    system_priviliges = 0;
    spent = 0;
    blocked = 0;
    vote = 0;
    weapon = 0;
    
    in_fork = 0;

    bounce_next = 0;
}

User::~User()
{
    clear();
}

void User::set(Plug *p)
{
    warning("User::set");
    
    system_priviliges = 0;
    spent = 0;
    blocked = 0;
    vote = 0;
    weapon = 0;
    player = 0;
    
    set_plug(p);
    state = SHAKE_START;
    
    // lists are initialized empty in constructor
    // afterwards they should be cleared by calling clear() when
    // the user is not needed anymore.
}

void User::clear()
{
    warning("User::clear()");
    
    player = 0;
    plug = 0;
    state = SHAKE_HANGUP;

    delete_objects(&send_q);
    delete_objects(&read_q);

    system_priviliges = 0;
    spent = 0;
    blocked = 0;
    vote = 0;
    weapon = 0;
    
    if (in_fork)
    {
	global_send_queue.remove(&send_q);
	in_fork = 0;
    }
}

void User::set_player(Splayer *p)
{
    player = p;
}

Splayer *User::get_player()
{
    return player;
}

void User::set_plug(Plug *p)
{
    plug = p;
}

Plug *User::get_plug()
{
    return plug;
}

void User::send_queue_packet(Packet *p)
{
    // users are only allowed to have their queues filled
    // when their state > SHAKE_MAP. if they have NOT received the
    // map yet, order of information would get screwed up:
    // example:
    //  - there is a nice little explosion here.  (queued)
    //  - player receives the map (directly through plug)
    //  - queue is flushed : hey! an explosion
    //
    // so between the explosion and the flushing of the queue,
    // the map may have changed but the user doesn't notice.
    
    if (state > SHAKE_MAP)
	send_q.push(p);
}

void User::send_queue()
{
    Packet *p;
    int nr_packets = 0;
    static int nummertje = 0;

//    warning("user::send_queue : blocked = %d", blocked);
    
    if (blocked)
    {
	// still waiting for ping answer
//	TRACE("User::send_queue() : queue is blocked");
	return;
    }
    
    while ((p = (Packet *)send_q.pop_tail()))
    {
	if (p->write_to(plug) < 0)
	{
	    warning("User::send_queue : connection dead!");
	    warning("User::send_queue : killing user");

	    delete_object(p);
	    hangup();
	    
	    return;
	}

	// clean up the packet
	delete_object(p);

	if (++nr_packets > MAX_PACKETS)
	{
	    // send directly through the plug: if the
	    // queue is very full we do not want the ping packet
	    // to wait for the queue to empty. that would defeat
	    // the purpose.
	    blocked = ++nummertje;

	    warning("sent %d packets, waiting for ping reply %d",
		    MAX_PACKETS, blocked);

	    Packet_info_num::write_to(plug,
				      ACTOR_SYSTEM,
				      SYS_MESSAGE,
				      System::PING_ARE_YOU_THERE,
				      blocked);
	    
	    break;
	}
    }
}

Packet *User::read_queue_packet()
{
    return (Packet *)read_q.pop_tail();
}

void User::read_queue()
{
    // read everything from the plug
    // add all packets to the read queue
    // read_packet should use recycled packets
    
    Packet *p;
    
    while ((p = read_packet(plug)))
    {
	// check for invalid packets
	// if we encounter one, hangup
	// this is necessary because the network traffic may have
	// become scrambled, not allowing recovery.
	// this is probably a major error, or a client purposely
	// sending garbage.
	if (p->err)
	{
	    warning("User::read_queue : invalid packet !!");
	    warning("Packet::error : %s", Packet::error);
	    warning("Notifying the client of invalid packet.");
	    Packet_info_num::write_to(plug,
				      ACTOR_SYSTEM,
				      SYS_MESSAGE,
				      System::INVALID_PACKET,
				      0);
	    warning("dump of illegal packet contents:");
	    print_packet(p);
	    
	    hangup();
	    delete_object(p);
	    
	    return;
	}

	// if bounce, bounce. Doh.
	if (bounce_next)
	{
	    p->write_to(plug);
	    delete_object(p);
	    bounce_next = 0;
	    continue;
	}
	
	// filter out hangup packets
	if (Packet_info_num::expect(p,
				    ACTOR_SYSTEM,
				    SYS_MESSAGE,
				    System::HANGUP)
	   )
	{
	    warning("hangup packet received: closing connection.");
	    hangup();
	    delete_object(p);

	    // don't continue reading!
	    return;
	}


        // filter out ping packets
	if (Packet_info_num::expect(p,
				    ACTOR_SYSTEM,
				    SYS_MESSAGE,
				    System::BOUNCE_NEXT_PACKET))
	{
	    warning("received bounce request: bouncing next packet.");
	    delete_object(p);
	    bounce_next = 1;
	    continue;
	}

	// filter out echo packets
	if (Packet_info_num::expect(p,
				    ACTOR_SYSTEM,
				    SYS_MESSAGE,
				    System::PING_YES_IM_HERE,
				    blocked))
	{
	    warning("ping packet %d received: unblocking plug", blocked);
	    
	    blocked = 0;
	    delete_object(p);
	    
	    continue;
	}
	
	read_q.push(p);
    }
}

void User::process_command_packet(Packet_command *pc)
{
    // check that the user has a player
    // if they are dead, the player doesn't exist.
    if (!player)
    {
	warning("User::process_packet :"
		" user is DEAD (no commands allowed)");
	return;
    }
	
    // check that the users player is the same as pc->actor_id
    if (!Packet_command::expect(pc, ACTOR_PLAYER, player->actor_id))
    {
	// different acting electron?
	// only works for pickup and drop commands
	// (meaning put in and get from)
	// these only work if player has an atom
	
	switch (pc->command)
	{
	    case COMMAND_PICKUP:
		do_put_in(player, pc);
		
		break;
	    case COMMAND_DROP:
		do_get_from(player, pc);
		
		break;
	    default:
		warning("server only understands command packets for "
			"the player (%d)", pc->command);
		print_packet(pc);
		break;
	}
	
	return;
    }
    
    switch (pc->command)
    {
	case COMMAND_DIR:
	    // move a player, with everything in it
	    do_move_dir(player, pc);
	    
	    break;
	case COMMAND_PICKUP:
	    // pick up an Object
	    do_pickup(player, pc);
	    
	    break;
	case COMMAND_DROP:
	    // drop an Object
	    do_drop(player, pc);
		
	    break;
	case COMMAND_ENTER:
	    // enter a vehicle
	    do_enter(player, pc);
		
	    break;
	case COMMAND_EXIT:
	    // exit a vehicle
	    do_exit(player, pc);

	    break;
	case COMMAND_TURN:
	    // turn (your vehicle)
	    do_turn(player, pc);

	    break;
	case COMMAND_FIRE_TO:
	    // fire to specified coordinates
	    do_fire_to(player, pc);
		
	    break;
	case COMMAND_FIRE_DIR:
	    // fire in specified direction (north, east, south, west)
	    do_fire_dir(player, pc);
		
	    break;
	case COMMAND_FIRE_AT:
	    // fire at the specified electron
	    do_fire_at(player, pc);
					  
	    break;
	case COMMAND_LAY_MINE:
	    // lay a mine at current position
	    do_lay_mine(player, pc);
		
	    break;
	case COMMAND_OPEN:
	    // open a door
	    do_open(player, pc);
		
	    break;
	case COMMAND_CLOSE:
	    // close a door
	    do_close(player, pc);
		
	    break;
	case COMMAND_LOCK:
	    // lock a door
	    do_lock(player, pc);
		
	    break;
	case COMMAND_UNLOCK:
	    // unlock a door
	    do_unlock(player, pc);
		
	    break;
	case COMMAND_VOTE:
	    // vote for a game
	    do_vote(player, pc);

	    break;
	case COMMAND_GAMETIME:
	    // tell the player the gametime
	    do_gametime(player, pc);

	    break;
	case COMMAND_WEAPON:
	    // choose a weapon
	    do_choose_weapon(player, pc);

	    break;
	case COMMAND_RIG:
	    // rig a vehicle
	    do_bomb_vehicle(player, pc);

	    break;
	case COMMAND_SELFDESTRUCT:
	    // blow up a vehicle
	    do_self_destruct(player, pc);

	    break;
	    
	default:
	    warning("unknown command packet %d", pc->command);
	    return;
    }
}


void User::process_packet(Packet *p)
{
    Electron *e = 0;
    
//    warning("packet being processed.");

    // no packet? keep playing
    if (!p)
	return;
    
    // for now, only expect command packets originating from
    // the player belonging to this connection
    if (Packet_command::expect(p))
    {
	// found a command packet
	Packet_command *pc = (Packet_command *)p;
//	warning("playing: packet_command received");

	// process the packet
	process_command_packet(pc);
	return;
    }
    else
    {
	// either an info packet, or an invalid packet
	
	// find the electron we're trying to set something for
	if (p->actor_type == ACTOR_SYSTEM)
	{
	    // dead users are allowed to send system messages.

	    if (p->actor_id == SYS_MESSAGE)
	    {
		system_message(p);
		return;
	    }
	    else
	    {
		e = sys;
	    }
	}
	else
	{	
	    // if a user is dead they are not allowed to set any
	    // variables
	    if (!player)
	    {
		warning("User::process_packet :"
			" user is DEAD (no variable setting allowed)");
		message(this, "Lie still. You are dead.");
		return;
	    }
	    
	    e = storage->find(p->actor_type, p->actor_id);
	}
	    
	if (!e)
	{
	    // not found? don't set any variables
	    warning("packet received for non-existing electron!");
	    print_packet(p);
	    return;
	}

	switch (p->get_type())
	{
	    case TYPE_INFO_NUM:
	    case TYPE_INFO_STRING:
	    case TYPE_INFO_VECT:

		warning("playing: info packet received");

		// check if the player is allowed to set this information
		if (!set_info_allowed(e, p))
		{
		    message(this, "You are not allowed to set that variable");
		    warning("unauthorized info packet sent by player");
		    print_packet(p);
			    
		    break;
		}

		warning("packet ok: setting variable");
		set_electron_var(e, p);

		// duplicate the packet. blergh.
		switch(p->get_type())
		{
		    case TYPE_INFO_NUM:
			::send_queue_packet(new_packet_info_num(
					    e,
					    ((Packet_info_num *)p)->var_id,
					    ((Packet_info_num *)p)->var_value)
					   );
			break;
		    case TYPE_INFO_VECT:
			::send_queue_packet(new_packet_info_vect(
					    e,
					    ((Packet_info_vect *)p)->var_id,
					    ((Packet_info_vect *)p)->var_x,
					    ((Packet_info_vect *)p)->var_y)
					   );
			break;
		    case TYPE_INFO_STRING:
			::send_queue_packet(new_packet_info_string(
				      e,
				      ((Packet_info_string *)p)->var_id,
				      ((Packet_info_string *)p)->get_string())
					   );
			break;
		}
		
		break;

	    default:
		warning("server got an unidentified packet");
	}
    }
}

int User::set_info_allowed(Electron *e, Packet *p)
{
    warning("User::set_info_allowed");
    
    switch (p->get_type())
    {
	case TYPE_INFO_NUM:
	    return set_int_allowed(e, ((Packet_info_num *)p)->var_id);
	    break;
	case TYPE_INFO_STRING:
	    return set_string_allowed(e, ((Packet_info_string *)p)->var_id);
	    break;
	case TYPE_INFO_VECT:
	    return set_vector_allowed(e, ((Packet_info_vect *)p)->var_id);
	    break;
    }

    return 0;
}

int User::is_owner(Electron *e)
{
    warning("User::is_owner");
    
    if (!player)
    {
	warning("User::is_owner: NULL player");
	return 0;
    }
    else
    {
	if (player == e)
	{
	    warning("player is owner of himself");
	    return 1;
	}
	
	// if you have an Object in your inventory, you own it.
	if (player->inv_contains_recursive(e))
	{
	    warning("player has Object in inventory and is the owner");
	    return 1;
	}

	// if you have the key to an Object in your inventory, you own it.
	if (e->is_group(GROUP_VEHICLE))
	{
	    Vehicle *v = (Vehicle *)e;

	    if (v->get_atom() != player->get_atom())
	    {
		// not in the same room? we're not the owner
		warning("Player is not in the same room as the vehicle");
		return 0;
	    }

	    // we're in the same room.
	    if (v->has_key())
	    {
		// vehicle has one or more keys
		
		if (v->owns_key(player))
		{
		    // we have a key
		    return 1;
		}
	    }
	    else
	    {
		// vehicle has no keys: everyone can use it
		
		return 1;
	    }
	}
    }
	    
    return 0;
}

int User::set_int_allowed(Electron *e, int var_id)
{
    // find our relationship with the electron
    // we either own it, or we don't

    int owner = is_owner(e);

    // see if we are allowed to set the variable
    return permission_granted(e->actor_type, TYPE_INFO_NUM, var_id,
			      owner, system_priviliges);
}

int User::set_string_allowed(Electron *e, int var_id)
{
    // find our relationship with the electron
    // we either own it, or we don't

    int owner = is_owner(e);

    // see if we are allowed to set the variable
    return permission_granted(e->actor_type, TYPE_INFO_STRING, var_id,
			      owner, system_priviliges);
}

int User::set_vector_allowed(Electron *e, int var_id)
{
    // find our relationship with the electron
    // we either own it, or we don't

    int owner = is_owner(e);

    // see if we are allowed to set the variable
    return permission_granted(e->actor_type, TYPE_INFO_VECT, var_id,
			      owner, system_priviliges);
}

void User::system_message(Packet *p)
{
    warning("processing system message");
//    print_packet(p);
    
    // ok, we received an info packet with
    // actor_type = ACTOR_SYSTEM
    // actor_id = SYS_MESSAGE

    Packet_info_num *pin;
    Packet_info_string *pis;
    Packet_info_vect *piv;
    
    switch (p->get_type())
    {
	case TYPE_INFO_NUM:
	    pin = (Packet_info_num *)p;
	    
	    switch (pin->var_id)
	    {
		case System::HANGUP:
		    // should never get here anymore.
		    // hangups are caught in read_queue
		    // they should be processed immediately.
		    ASSERT(0);
		    break;
		case System::CTS_SHUTDOWN:
		    // check this for validity!
		    // send a HANGUP to all clients, then shut down.
		    throw(1);
		default:
		    warning("received unexpected system message %d", pin->var_id);
		    break;
	    }

	    break;
	case TYPE_INFO_STRING:
	    pis = (Packet_info_string *)p;
	    break;
	case TYPE_INFO_VECT:
	    piv = (Packet_info_vect *)p;
	    break;
    }
    
}

// call hangup when a player is SHAKE_PLAYING
// it will remove the player object from the lattice,
// send a hangup packet to the client,
// and end with setting the users state to SHAKE_HANGUP

// SHAKE_HANGUP will disconnect, remove the player object,
// clean up queues etc and finally remove the user object
void User::hangup()
{
    int user_index;
    
    // the player is disconnecting
    // start with doing the same thing
    // find out which user index we have

    // dump freelist stats
    // do it before hanging up so we don't spam what comes after
    dump_freelist_stats();
    
    warning("user::hangup");
    
    for (user_index = 0; user_index < fb.get_nr_plugs(); user_index++)
    {
	if (fb[user_index] == plug)
	    break;
    }
    
    // remove the player from the game, and let the other players
    // know what's happening
    // first drop the whole inventory
    if (player)
    {
	warning("Removing player %d from the game", player->actor_id);
	move_to_top(player);
	destroy_electron(player);
	player = 0;
    }

    // setting SHAKE_HANGUP will do the rest (disconnect etc)
    warning("User %d: SHAKE_HANGUP", user_index);
    state = SHAKE_HANGUP;
}

// how much time did we spend
int User::time_spent() const
{
    return spent;
}

// used to reset spent
void User::set_time_spent(int ts)
{
    if (ts < 0)
    {
	warning("User::set_time_spent : ts = %d ??", ts);
	spent = 0;
    }

    spent = ts;
}

// spend an amount of time
void User::spend_time(int how_much)
{
    if (how_much < 0)
    {
	warning("User::set_time_spent : how_much = %d ??", how_much);
	spent = 0;
    }

    spent += how_much;
}

List *User::get_send_queue()
{
    return &send_q;
}

// functions to check if the plug is in the fork
void User::set_in_fork(int yesno)
{
    in_fork = yesno;
}

int User::get_in_fork()
{
    return in_fork;
}

User *new_user(Plug *p)
{
    User *u;

    u = (User *)new_object(SERVER_FREELIST_USER);

    if (!u)
    {
	u = new User;
    }

    u->set(p);

    return u;
}

void User::set_vote(int v)
{
    vote = v;
}

int User::get_vote() const
{
    return vote;
}

void User::set_weapon(int w)
{
    weapon = w;
}

int User::get_weapon() const
{
    return weapon;
}
