/*
 *
 *   ^   |    sssss p   ddddd  fff  ggggg hhhh   iii  j   j    |   ^
 *  /|\  |    s     p   d     f   f   g   h   h i   i jj  j    |  /|\
 *   |   |    sss   p   ddd   f       g   hhhh  i   i j j j    |   |
 *   |  \|/   s     p   d     f   f   g   h   h i   i j  jj   \|/  |
 *   |   v    sssss ppp ddddd  fff    g   h   h  iii  j   j    v   |
 *
 *                           copyright 1999
 *                  Martijn Versteegh & Hein Zelle
 *
 */

/*
 *        servertest.cc
 *
 *        first implementation of an electron test server
 *        main function of the server
 */

 
#include <string.h>
#include <signal.h>
#include "globals.h"
#include "parser/tools.h"

// argh. try to make the linker believe that it really has to call the
// functions in here, and therefore that it really should put them in the
// library. This is needed to prevent circular linking undefined references
// between libserver.a and libparser.a
//
// do not call this function :)
void fool_the_linker()
{
    Electron *e;
    e = get_electron_from_type(ACTOR_SYSTEM, 0);
    ASSERT(1);
}

void print_version(int port)
{
    printf("Game version : %s\n", GAME_VERSION_STRING);
    printf("Communication protocol version : %s\n", COMM_VERSION_STRING);
    printf("Release date : %s\n\n", RELEASE_DATE);
    printf("Maximum number of players : %d\n", MAX_PLAYERS);
    printf("Running on port : %d\n", port);
}

void print_logo()
{
    printf(logo);
}

// function to properly shutdown the server
void shutdown_server()
{
    // let the players know what's happening
    for (int i = 0; i < fb.get_nr_plugs(); i++)
    {
	if (fb[i])
	{
	    Packet_info_num::write_to(fb[i],
				      ACTOR_SYSTEM,
				      SYS_MESSAGE,
				      System::HANGUP,
				      0);
	}
    }

    // stop freelist checks from complaining:
    // we're shutting down so it's allowed to throw away objects while
    // their reference count is still > 0.
    // example: routeplanners contain lists of atoms. calling
    // shutdown_globals (delete lattice) before shutdown common
    // (delete storage) will cause an assert because the atoms in lattice
    // are still in the routeplanner's lists.

    going_down = 1;
    
    warning("shutting down map_actor_types");
    shutdown_map_actor_types();

    warning("shutting down globals");
    shutdown_globals();

    warning("shutting down common");
    shutdown_common();

    warning("server is done");
    exit(0);
}

// signal handler for SIGINT
void shutdown_handler(int sub)
{
    warning("Caught SIGINT, shutting down");
    shutdown_server();
}

// function to properly initialise the server
void init_server()
{
    // set both stdout& stderr to line buffered
    setvbuf(stdout, NULL, _IOLBF, BUFSIZ); 
    setvbuf(stderr, NULL, _IOLBF, BUFSIZ);

    // ignore SIGPIPE. 
    signal(SIGPIPE, SIG_IGN);
    signal(SIGINT, shutdown_handler);
    
    // initialise libraries
    if (init_common(SERVER_FREELIST_LAST - COMMON_FREELIST_LAST))
	fatal("init_common failed");
    if (init_globals())
	fatal("init_globals failed");
    if (init_map_actor_types())
	fatal("init_map_actor_types failed");
    
    sys->set_string_var(System::GAME_NAME, "Electron 0.1");
}

void usage()
{
    fprintf(stderr,
	    "usage: %s [options] [map]\n"
	    "options:\n"
	    "    -p port  (default 5000) : specify port to listen on\n"
	    "    -h, --help              : print this message\n"
	    "    map                     : load this map\n",
	    progname);
    exit(1);
}

char const *map_filename(char const *mapname)
{
    static char newname[1024];

    if (strchr(mapname, '/'))
    {
	// absolute path given.
	sstrncpy(newname, mapname, 1024);
    }
    else
    {
	// no absolute pathname: use electron_home to determine base.
	ssprintf(newname, 1024, "%s/maps/%s",
		 get_electron_home(),
		 mapname);
    }
	
    return newname;
}

int main(int argc, char **argv)
{
    char const *mapname = "test.map";
    long int port = 5000, tmp_port;
    char *strtol_error;
    
    progname = argv[0];

    // check program arguments

    for (int i = 1; i < argc; i++)
    {
	if (!strcmp(argv[i], "-h") || !strcmp(argv[i], "--help"))
	    usage();

	// check if it's a port option
	if (!strcmp(argv[i], "-p"))
	{
	    if (argc == ++i)
	    {
		// too few arguments
		usage();
	    }

	    // get the number out
	    tmp_port = strtol(argv[i], &strtol_error, 0);
	    if (strtol_error == argv[i])
	    {
		// error, invalid port number.
		usage();
	    }
	    port = tmp_port;
	    
	    continue;
	}
	
	// no? must be a mapname then
	mapname = argv[i];
    }

    mapname = map_filename(mapname);
    
    if (fb.init(MAX_PLAYERS + 1, port))
	exit(1);

    init_server();
    
    print_logo();
    print_version(port);

    if (!load_map(mapname))
    {
	warning("failed to load map '%s', exiting", mapname);
	return 0;
    }

    // run the game
    while (game_loop());

    // finished: shutdown nicely
    shutdown_server();
    
    return 0;
}
