/*
 *
 *   ^   |    sssss p   ddddd  fff  ggggg hhhh   iii  j   j    |   ^
 *  /|\  |    s     p   d     f   f   g   h   h i   i jj  j    |  /|\
 *   |   |    sss   p   ddd   f       g   hhhh  i   i j j j    |   |
 *   |  \|/   s     p   d     f   f   g   h   h i   i j  jj   \|/  |
 *   |   v    sssss ppp ddddd  fff    g   h   h  iii  j   j    v   |
 *
 *                           copyright 1999
 *                  Martijn Versteegh & Hein Zelle
 *
 */

/*
 *        tools.cc
 *
 *        utility functions for the parser
 *        used to fill the lattice and keep track of the atom and
 *        electron stack, and of variables that have been set
 *        most of these functions are called from the parser
 */

#include <ctype.h>
#include <string.h>
#include "tools.h"
#include "label.h"
#include "reference.h"
#include "common/lattice.h"
#include "common/act_type.h"
#include "common/actors/system.h"
#include "common/actors/vehicle.h"
#include "common/actors/key.h"
#include "common/actors/doorkey.h"
#include "common/actors/door.h"

List current_atom;
List current_electron;
List references;
List labels;

char *map_actor_types[ACTOR_LAST];

int map_name_set = 0;
int start_point_set = 0;
int vote_point_set = 0;
int comments_eaten = 0;

// fill the map_actor_types array
int init_map_actor_types()
{
    for (int i = 0; i < ACTOR_LAST; i++)
    {
	// string is allocated here!
	// delete in shutdown_map_actor_types
	map_actor_types[i] = underscore_name(i);
    }

    return 0;
}

// clean up the map_actor_types array
void shutdown_map_actor_types()
{
    for (int i = 0; i < ACTOR_LAST; i++)
    {
	delete [] map_actor_types[i];
	map_actor_types[i] = 0;
    }
}

// this function allocates a string!!
// it takes the type_name of the given actor_type, and replaces
// all spaces with 
char *underscore_name(int actor_type)
{
    char const *source;
    char *target;
    char *buf;

    source = global_actor_statics[actor_type]->type_name;
    buf = new char[strlen(source) + 1];
    target = buf;
    
    while (*source)
    {
	*target = (isspace(*source)) ? '_' : *source;

	target++;
	source++;
    }

    *target = '\0';
    
    return buf;
}

// set the reset / start point of the lattice
void set_start_point(Vector *v)
{
    if (!lattice)
    {
	fatal("set_start_point: lattice not created yet!");
    }

    lattice->set_reset(v->x, v->y);
    start_point_set++;
}

// set the voting room of the lattice
void set_vote_point(Vector *v)
{
    if (!lattice)
    {
	fatal("set_vote_point: lattice not created yet!");
    }

    lattice->set_vote(v->x, v->y);
    vote_point_set++;
}

// set the name of the map
void set_map_name(char const *str)
{
    int i = strlen(str);
    char buf[256];
    strncpy(buf, str + 1, i - 2);
    buf[i - 2] = '\0';
    sys->set_string_var(System::MAP_NAME, buf);

    TRACE("- map name: %s", buf);

    map_name_set++;
}

// make a new lattice with the given size
void set_lattice_size(Vector *v)
{
    if (lattice)
    {
	fatal("set_lattice_size : lattice already exists!");
    }

    lattice = new Lattice(v->x, v->y);
}

int check_file()
{
    int succes = 1;
    int rx = 0;
    int ry = 0;
    
    // check that both current_atom and current_electron are empty
    // if not, we had unmatched braces

    if (current_atom.get_head() || current_electron.get_head())
    {
	warning("check_file: mismatched braces detected!");
	succes = 0;
    }

    // check that the mapname was set
    if (!map_name_set)
    {
	warning("check_file: no map name set");
	succes = 0;
    }

    if (map_name_set > 1)
    {
	warning("check_file: map name set %d times!", map_name_set);
	succes = 0;
    }
    
    if (!start_point_set)
    {
	warning("check_file: no start point set");
	succes = 0;
    }
    
    if (start_point_set > 1)
    {
	warning("check_file: start point set %d times!", start_point_set);
	succes = 0;
    }

    lattice->get_reset(&rx, &ry);
    Vector v(rx, ry);

    // check that the start point is valid
    if (!check_range(&v))
    {
	warning("check_file: start point <%d, %d> out of bounds",
		rx, ry);
	succes = 0;
    }

    if (!lattice->get(rx, ry))
    {
	warning("check_file: invalid start point <%d, %d> (no atom)",
		rx, ry);
	succes = 0;
    }

    if (!vote_point_set)
    {
	warning("check_file: no vote point set");
	succes = 0;
    }
    
    if (vote_point_set > 1)
    {
	warning("check_file: vote point set %d times!", vote_point_set);
	succes = 0;
    }

    lattice->get_vote(&rx, &ry);
    v.x = rx;
    v.y = ry;

    // check that the vote point is valid
    if (!check_range(&v))
    {
	warning("check_file: vote point <%d, %d> out of bounds",
		rx, ry);
	succes = 0;
    }

    if (!lattice->get(rx, ry))
    {
	warning("check_file: invalid vote point <%d, %d> (no atom)",
		rx, ry);
	succes = 0;
    }
    
    return succes;
}

// convert a given string into an actor_type, using the
// map_actor_types array
int string_to_type(char const *str)
{
    for (int i = 0; i < ACTOR_LAST; i++)
    {
	if (!strcmp(str, map_actor_types[i]))
	{
	    return i;
	}
    }

    return -1;
}

// check if the given coordinates fall within lattice boundaries
int check_range(Vector *v)
{
    return (v->x >= 0 && v->x < lattice->get_w()
	    &&
	    v->y >= 0 && v->y < lattice->get_h());
}

// start a new atom at the given coordinates
void begin_atom(Vector *v)
{
    // check_range has already been called
    // simply add the new atom

    Atom *a = new_atom();
    lattice->set(a, v->x, v->y);

    // push the atom onto the stack
    current_atom.push(a);
}

// finish the current atom
void end_atom()
{
    Atom *tmpa;
    
    // pop the atom from the stack
    tmpa = (Atom *)current_atom.pop();

    // check if the atom is in any lists still
    ASSERT(tmpa->list_refcount == 0);
}

// create a new electron with specified type, and put it either
// in current_electron (if there is one) or in current_atom
void begin_electron(int actor_type)
{
    // validity of actor_type has already been checked
    // simply make the new electron

    Electron *tmp;
    Electron *parent;
    Atom *where;

    // make the electron, and add it to storage
    tmp = get_electron_from_type(actor_type, Electron::fid());
    storage->lowlevel_add(tmp);

    // put it in the right parent (atom or electron)
    // check if we have a parent electron
    parent = (Electron *)current_electron.get_head();
    if (parent)
    {
	// add the electron to the inventory of it's parent
	// can't do this directly because we cannot pass electrons
	// around in the parser
	parent->get_atom()->add(tmp, parent);
    }
    else
    {
	where = (Atom *)current_atom.get_head();
	ASSERT(where);     // must exist, otherwise it's a big bug

	// make this the top level Object (no parent)
	where->add(tmp, NULL);   
    }

    // push the electron onto the stack, in case it gets an inventory
    current_electron.push(tmp);
}

// finish the current electron
void end_electron()
{
    Electron *e;
    
    // pop the electron from the stack
    e = (Electron *) current_electron.pop();

    // call the map_finish function of the electron to let it know
    // we are done loading. used for example to add all the pads
    // in the inventory of a repair_vehicle to it's pads list
    e->map_finish();
}

// check for valid actor type
int check_type(int actor_type)
{
    return (actor_type >= 0 && actor_type < ACTOR_LAST);
}

// add a label to the current electron
void add_label(char const *str)
{
    // check that the current electron exists
    Electron *e;
    Label *l;

    e = (Electron *)current_electron.get_head();
    ASSERT(e);

    l = new Label(e, str);
    labels.push(l);
}

// add a requested reference to the current electron
void add_reference(char const *str)
{
    // check that the current electron exists
    Electron *e;
    Reference *r;

    e = (Electron *)current_electron.get_head();
    ASSERT(e);

    r = new Reference(e, str);
    references.push(r);
}

// connect a key with corresponding target
// target can either be a door or a vehicle
// check if they really are key and vehicle, or key and door
int connect(Electron *target, Electron *key)
{
    Door *d = 0;
    Doorkey *dk = 0;
    
    Vehicle *v = 0;
    Key *k = 0;
    
    if (!target->is_group(GROUP_VEHICLE))
    {
	if (!target->actor_type == ACTOR_DOOR)
	{
	    // not a vehicle and not a door? fault
	    return 0;
	}
	else
	{
	    d = (Door *)target;

	    if (key->actor_type != ACTOR_DOORKEY)
	    {
		return 0;
	    }
	    else
	    {
		dk = (Doorkey *)key;
	    }
	}
    }
    else
    {
	v = (Vehicle *)target;

	if (key->actor_type != ACTOR_KEY)
	{
	    return 0;
	}
	else
	{
	    k = (Key *)key;
	}
    }

    if (k && v)
    {
	// check that the key is not already connected
	if (k->get_vehicle())
	{
	    warning("can't connect key to %s %d: "
		    "it is already connected to %s %d.",
		    target->type_name, target->actor_id,
		    k->get_vehicle()->type_name,
		    k->get_vehicle()->actor_id);
	    return 0;
	}
	
	// connect key to vehicle and vv
	v->add_key(k);
	k->set_vehicle(v);
    }
    else
    if (dk && d)
    {
	// check that the key is not already connected
	if (dk->get_door())
	{
	    warning("can't connect doorkey to %s %d: "
		    "it is already connected to %s %d.",
		    target->type_name, target->actor_id,
		    dk->get_door()->type_name,
		    dk->get_door()->actor_id);
	    return 0;
	}

	// connect doorkey to door and vv
	d->add_key(dk);
	dk->set_door(d);
    }
    else
    {
	return 0;
    }
    
    return 1;
}

// trie to tie all references to all labels
int fix_references()
{
    int succes = 1;
    Reference *r;
    Label *l;

    // loop over all references, pop them from the List as we go
    references.reset();
    while ((r = (Reference *)references.pop()))
    {
	// loop over all labels and compare the reference to the labels
	labels.reset();
	while ((l = (Label *)labels.get()))
	{
	    if (!strcmp(l->get_name(), r->get_name()))
	    {
		// we have a match, try to connect them
		if (!connect(l->get_e(), r->get_e()))
		{
		    warning("failed to connect %s to %s (%s)",
			    r->get_e()->type_name,
			    l->get_name(), l->get_e()->type_name);
		    succes = 0;
		}

		// get rid of the reference
		delete r;
		r = 0;
		break;
	    }
	    
	    labels.next();
	}

	// if r still exists, we didn't find a matching label
	if (r)
	{
	    warning("fix_references: undefined reference %s in %s",
		    r->get_name(), r->get_e()->type_name);
	    delete r;
	    r = 0;
	    succes = 0;
	}
	
	references.next();
    }

    // delete all labels
    labels.destroy();

    return succes;
}
