/*
 *
 *   ^   |    sssss p   ddddd  fff  ggggg hhhh   iii  j   j    |   ^
 *  /|\  |    s     p   d     f   f   g   h   h i   i jj  j    |  /|\
 *   |   |    sss   p   ddd   f       g   hhhh  i   i j j j    |   |
 *   |  \|/   s     p   d     f   f   g   h   h i   i j  jj   \|/  |
 *   |   v    sssss ppp ddddd  fff    g   h   h  iii  j   j    v   |
 *
 *                           copyright 1999
 *                  Martijn Versteegh & Hein Zelle
 *
 */

/*
 *        move.cc
 *
 *        functions to move things
 */

#include "globals.h"
#include "usercomm.h"
#include "message.h"

void print_pos(Electron *e)
{
    warning("%s at position [%d, %d]", e->type_name, e->gx(), e->gy());
}

int do_move_dir(Splayer *pl, Packet_command *pc)
{
    Vehicle *v;
    int ret;
    
//    print_pos(pl);
    
    // check if the player is inside a vehicle
    if ((v = pl->get_vehicle()))
    {
	ret = do_move_vehicle_dir(pl, pc, v);

	// always print messages, even if move succeeded
	check_error_message(pl);
	
	if (!ret)
	{
	    message(pl, "Vehicle move failed.");
	}
	
//	print_pos(pl);
	return ret;
    }
    
    // no? move the player. (walk)
    ret = do_move_player_dir(pl, pc);
    if (!ret)
    {
	message(pl, "Move failed.");
    }
    
//    warning("player went to:");
//    print_pos(pl);

    return ret;
}

// 0: move failed
// 1: move was succesfull
int do_move_player_dir(Splayer *pl, Packet_command *pc)
{
    int
	x,
	y,
	ret;

    pl->get_pos(&x, &y);

    // we're walking
    switch (pc->arg1)
    {
	case DIR_N:
	    y--;
	    break;
	case DIR_E:
	    x++;
	    break;
	case DIR_S:
	    y++;
	    break;
	case DIR_W:
	    x--;
	    break;
	default:
	    warning("command_dir: invalid direction %d", pc->arg1);
	    return 0;
	    break;
    }

    ret = do_move_electron(pl, x, y, 20);

    // only spend time if the move succeeded
    if (ret)
    {
	pl->spend_time(20);
    }
    
    return ret;
}

int do_move_vehicle_dir(Splayer *pl, Packet_command *pc, Vehicle *v)
{
    int ret;
    int x, y;

    v->get_pos(&x, &y);
    
    // check if player has the key for the vehicle
    // assuming that the user is valid!
    if (!pl->get_user()->is_owner(v))
    {
	message(pl, "You don't control that vehicle");
	return 0;
    }
    
    // ask the vehicle to move
    // the vehicle has to call the queue functions by itself
    // vehicle returns the time spent or 0 if it didn't move
    // watch it: both the player and the vehicle may have died in
    // the mean time!
    ret = v->move_dir(pc->arg1, pl);

    // only spend time if the move succeeded (partially)
    if (ret)
    {
	pl->spend_time(ret);
    }
    
    return ret;
}

int do_move_electron(Electron *e, Atom *a, int time)
{
    int xold = e->gx();
    int yold = e->gy();

    // assume this atom is valid to go to (it exists
    // and we are allowed to enter it)
    // use get_target_atom if you want to check that
    
    // are we allowed to leave our parent
    if (e->parent)
    {
	if (!e->parent->can_be_entered(e))
	{
	    return 0;
	}
    }

    // check if we are allowed to enter the target atom
    if (!a->top_level_object->can_be_entered(e))
	return 0;
    
//    warning("do_move_electron: electron starts at");
//    print_pos(e);
    
    // activate leave triggers
    trigger_leave(e, e->get_atom());

    // check if we're still alive
    if (e->dead)
	return 0;
    
    // let the lattice move the electron
    lattice->move_electron(e, a->top_level_object);

//    warning("do_move_electron: electron is now at");
//    print_pos(e);

    // need to write to the clients what happens!
    send_queue_packet(new_packet_command(e,
					 COMMAND_FROM_TO,
					 xold, yold,
					 e->gx(), e->gy(), time));

    
    // activate enter triggers after the correct time
    evl->add_event(new_enter_event(e, a), time);
    
    return 1;
}

int do_move_electron(Electron *e, int x, int y, int time)
{
    Atom *a = get_target_atom(e, x, y);

    if (!a)
	return 0;   // no atom found, or we can't enter it

    return do_move_electron(e, a, time);
}

Atom *get_target_atom(Electron *e, int x, int y)
{
    Atom *a = 0;

    // check if the atom exists (catches invalid coordinates)
    a = lattice->get(x, y);
    if (!a)
    {
//	warning("get_target_atom: %d, %d, NULL atom (no error)", x, y);
	return 0;
    }

    // check if the atom has a top level Object
    if (!a->top_level_object)
    {
	warning("get_target_atom: no top level Object");
	return 0;
    }

    // check if we are allowed to enter the atom
    if (!a->top_level_object->can_be_entered(e))
    {
	warning("get_target_atom: %s can not be entered by %s",
		a->top_level_object->type_name, e->type_name);
	return 0;
    }

    // check if we are allowed to enter a room with a cyclewall
    if (check_cyclewall(e, a))
    {
	warning("get_target_atom: cyclewall in the way");
	return 0;
    }
    
    return a;
}

int check_cyclewall(Electron *e, Atom *a)
{
    List *l = a->get(LIST_MISC);
    Electron *tmp;
    
    l->reset();
    while((tmp = (Electron *)l->get()))
    {
	if (tmp->actor_type == ACTOR_CYCLEWALL)
	    return 1;

	l->next();
    }

    return 0;
}

int do_turn_vehicle(Splayer *pl, Packet_command *pc, Vehicle *v)
{
    int spent;
    int dir = pc->arg1;
    
    // mag jij dat
    if (!pl->get_user()->is_owner(v))
    {
	message(pl, "You don't control that vehicle");
	return 0;
    }

    spent = v->turn(dir);

    if (spent > 0)
    {
	pl->spend_time(spent);
    }

    return spent;
}

int do_turn(Splayer *pl, Packet_command *pc)
{
    Vehicle *v;
    int ret;

    // check if the player is inside a vehicle
    if ((v = pl->get_vehicle()))
    {
	ret = do_turn_vehicle(pl, pc, v);

	// always check for messages
	check_error_message(pl);

	if (!ret)
	{
	    message(pl, "Vehicle turn failed.");
	}
	
	return ret;
    }

    // no? do something else
    // players can't turn yet
    message(pl, "Players can't turn (yet).");
    
    return 0;
}

// compute coordinates of the atom to the <dir>
int get_dir_coordinates(Electron *e, int dir, int *x, int *y)
{
    int
	tx,
	ty;

    e->get_pos(&tx, &ty);

    // find target coordinates
    switch (dir)
    {
	case DIR_N:
	    ty--;
	    break;
	case DIR_E:
	    tx++;
	    break;
	case DIR_S:
	    ty++;
	    break;
	case DIR_W:
	    tx--;
	    break;
	default:
	    warning("get_dir_coordinates: invalid direction %d", dir);
	    return 0;
	    break;
    }

    *x = tx;
    *y = ty;

    return 1;
}


int random_push(Electron *e)
{
    int
	x = e->gx(),
	y = e->gy(),
	options = 0,
	result;
    
    Atom *exits[4];
    
    warning("random_push: pushing %s at <%d, %d>", e->type_name, x, y);
    
    // let get_target_atom do the checks for the target atom
    // if this succeeds, it has a top level Object and is (in principle)
    // enterable for us. get_target_atom also checks for cyclewalls in a room
    
    if ((exits[options] = get_target_atom(e, x + 1, y)))
    {
	options++;
    }

    if ((exits[options] = get_target_atom(e, x, y + 1)))
    {
	options++;
    }
    
    if ((exits[options] = get_target_atom(e, x - 1, y)))
    {
	options++;
    }
    
    if ((exits[options] = get_target_atom(e, x, y - 1)))
    {
	options++;
    }
    
    warning("random_push: there were %d options", options);
    
    if (!options)
	return 0;
    
    result = (int) ((double)options * rand() / (RAND_MAX + 1.0));
    
    warning("random_push: moving %s to option %d", e->type_name, result);
    
    // now that we've picked a room: move the electron
    do_move_electron(e, exits[result], 10);
    
    warning("random_push: %s is now at <%d, %d>",
	    e->type_name, e->gx(), e->gy());
    
    return 1;
}

// check availability of atoms around us:
// 0 is left top, 4 is below us, 9 is right bottom
Atom *check_adjacent_atom(Electron *parent, Electron *to_drop, int counter)
{
    int x, y, myx, myy;

    parent->get_pos(&myx, &myy);

    x = myx - 1 + (counter % 3);
    y = myy - 1 + (counter / 3);

    return get_target_atom(to_drop, x, y);
}
