/*
 *
 *   ^   |    sssss p   ddddd  fff  ggggg hhhh   iii  j   j    |   ^
 *  /|\  |    s     p   d     f   f   g   h   h i   i jj  j    |  /|\
 *   |   |    sss   p   ddd   f       g   hhhh  i   i j j j    |   |
 *   |  \|/   s     p   d     f   f   g   h   h i   i j  jj   \|/  |
 *   |   v    sssss ppp ddddd  fff    g   h   h  iii  j   j    v   |
 *
 *                           copyright 1999
 *                  Martijn Versteegh & Hein Zelle
 *
 */

/*
 *        inout.cc
 *
 *        functions for entering and leaving vehicles and the like
 */

#include "globals.h"
#include "usercomm.h"
#include "message.h"
#include "common/trigger.h"
#include "inout.h"

// when dropping things or picking them up, try to send only
// COMMAND_DROP and COMMAND_ENTER command packets to the client.

// players ALWAYS have a parent, otherwise these functions should NOT
// be called. players ALWAYS have a valid lattice position (atom) too.

// arg1 = vehicle actor_type
// arg2 = vehicle actor_id
void do_enter(Splayer *pl, Packet_command *pc)
{
    Electron *parent = pl->parent;
    Electron *to_enter;
    Atom *a;

    TRACE("enter command received");
    
    // find the vehicle we are talking about
    a = pl->get_atom();
    to_enter = a->find(pc->arg1, pc->larg2());

    if (!to_enter)
	return;

    TRACE("player %s tries to enter vehicle %s, id %d",
	    pl->get_string_var(Splayer::NAME),
	    to_enter->type_name,
	    to_enter->actor_id);
    
    // see if the vehicle is in the player's parent
    if (!parent->inv_contains(to_enter))
	return;
    
    // found the vehicle. see if we may enter it.
    if (!to_enter->can_be_entered(pl))
    {
	// vehicle may have set an error message (tank)
	check_error_message(pl);
	return;
    }

    // call leave trigger
    trigger_leave_electron(pl, parent);

    // check if we're still alive
    if (pl->dead || parent->dead)
	return;

    // we're allowed to enter it. now do it.
    parent->inv_del(pl);
    to_enter->inv_add(pl);

    TRACE("enter succesfull");
    
    // add the action to the send queue
    send_queue_packet(new_packet_command(to_enter,
					 COMMAND_PICKUP,
					 pl,
					 0));

    // call enter trigger
    trigger_enter_electron(pl, to_enter);
}

// try to leave the parent, no matter what it is
void do_exit(Splayer *pl, Packet_command *pc)
{
    Electron *parent = pl->parent;
    Electron *to_enter;

    TRACE("exit command received");
    
    // check if we are allowed to leave the parent
    // if we can't enter the parent, we can't leave it either (for now).
    if (!parent->can_be_entered(pl))
    {
	// vehicle may have set an error message (tank)
	check_error_message(pl);
	return;
    }
    
    // now find the parent's parent
    to_enter = parent->parent;
    if (!to_enter)
	return;

    TRACE("player %s tries to leave vehicle and enter '%s', id %d",
	    pl->get_string_var(Splayer::NAME),
	    to_enter->type_name,
	    to_enter->actor_id);
    
    // check if we may enter the parent's parent
    if (!to_enter->can_be_entered(pl))
    {
	// parent may have set an error message (tank)
	check_error_message(pl);
	return;
    }

    // call leave trigger
    trigger_leave_electron(pl, parent);

    // check if we're still alive
    if (pl->dead || parent->dead)
	return;
    
    // ok. we may leave the parent, and we may enter it's parent
    parent->inv_drop(pl);

    TRACE("exit succesfull");
    
    // let the world know what happened
    send_queue_packet(new_packet_command(parent,
					 COMMAND_DROP,
					 pl,
					 0));

    // call enter trigger
    trigger_enter_electron(pl, to_enter);
}

void do_drop(Splayer *pl, Packet_command *pc)
{
    Electron *parent = pl->parent;
    Electron *to_drop;
    Atom *a = pl->get_atom();

    // find the Object to drop
    to_drop = a->find(pc->arg1, pc->larg2());
    if (!to_drop)
	return;
    
    // check if the parent will receive the Object
    if (!parent->can_be_entered(to_drop))
    {
	// parent may have set an error message (tank)
	check_error_message(pl);
	
	return;
    }
    
    // call leave trigger
    trigger_leave_electron(to_drop, pl);

    // check if we're still alive
    if (pl->dead || to_drop->dead)
	return;

    // drop the Object (only if we are really carrying it)
    pl->inv_drop(to_drop);

    // let the world know what happened
    send_queue_packet(new_packet_command(pl,
					 COMMAND_DROP,
					 to_drop,
					 0));

    trigger_enter_electron(to_drop, parent);
}

void do_pickup(Splayer *pl, Packet_command *pc)
{
    Electron *parent = pl->parent;
    Electron *to_pickup;
    Atom *a = pl->get_atom();

    // find the item to pickup
    to_pickup = a->find(pc->arg1, pc->larg2());

    if (!to_pickup)
    {
	message(pl, "You don't see that here");
	TRACE("player tried to pick up an Object that is not in the atom");
	return;
    }
    
    // see if the item is in our parents inventory
    if (!parent->inv_contains(to_pickup))
    {
	message(pl, "You can't reach it");
	TRACE("pickup: parent does not contain to_pickup");
	return;
    }

    // see if we can pick it up
    if (!pl->can_be_entered(to_pickup))
    {
	message(pl, "You can't pick that up");
	return;
    }

    // call leave trigger
    trigger_leave_electron(to_pickup, parent);

    // check if we're still alive
    if (pl->dead || to_pickup->dead || parent->dead)
	return;

    // douglas adams.. Hot potato
    // don't pick it up - don't pick it up - don't pick it up
    // quick, pass it on - quick, pass it on - quick, pass it on
    parent->inv_del(to_pickup);
    pl->inv_add(to_pickup);

    // let the world know what happened
    send_queue_packet(new_packet_command(pl,
					 COMMAND_PICKUP,
					 to_pickup,
					 0));

    // call enter trigger
    trigger_enter_electron(to_pickup, pl);
}

// function to create the inventory structure at the client side.
// we start with all loose electrons lying in the room.
// now we need to pick them up in the right order: the client will only
// pick things up if (to_pickup->parent == picking_up->parent) ie if
// the acting electron and target electron are at the same level.
// this is most economically done by starting at the deepest inventory
// level and working in the direction of the parent.
//
// use the version below which gets a queue instead of a user
// as it's argument.

void hot_potato(User *u, Electron *top)
{
    hot_potato(top, u->get_send_queue());
}

#if 0

void hot_potato(User *u, Electron *top)
{
    List *inv = top->inv();
    
    // first do all our children, recursively
    // this makes sure we start at the deepest level
    if (inv)
    {
	Electron *e;

	for (inv->reset();
	     (e = (Electron *)inv->get());
	     inv->next())
	{
	    // recursively pass it on to all our children
	    hot_potato(u, e);
	}
    }

    // only let ourselves be picked up if we are inside something
    // that is not a room. if our parent is a room (or we are a room
    // ourselves), then the client
    // has already got the inventory structure right.

    if (!top->parent || !top->parent->parent) 
	return;

    // tell the client that we are picked up by our parent
//    TRACE("hot_potato: %s picks up %s",
//	  top->parent->type_name,
//	  top->type_name);
    
    u->send_queue_packet(new_packet_command(top->parent,
					    COMMAND_PICKUP,
					    top,
					    0));
}

#endif

// function to create the inventory structure at the client side.
// we start with all loose electrons lying in the room.
// now we need to pick them up in the right order: the client will only
// pick things up if (to_pickup->parent == picking_up->parent) ie if
// the acting electron and target electron are at the same level.
// this is most economically done by starting at the deepest inventory
// level and working in the direction of the parent.
//
// this one writes to a queue instead of a user (more general)
void hot_potato(Electron *top, List *q)
{
    List *inv = top->inv();
    
    // first do all our children, recursively
    // this makes sure we start at the deepest level
    if (inv)
    {
	Electron *e;

	for (inv->reset();
	     (e = (Electron *)inv->get());
	     inv->next())
	{
	    // recursively pass it on to all our children
	    hot_potato(e, q);
	}
    }

    // only let ourselves be picked up if we are inside something
    // that is not a room. if our parent is a room (or we are a room
    // ourselves), then the client
    // has already got the inventory structure right.

    if (!top->parent || !top->parent->parent) 
	return;

    // tell the client that we are picked up by our parent
//    TRACE("hot_potato: %s picks up %s",
//	  top->parent->type_name,
//	  top->type_name);
    

    q->push(new_packet_command(top->parent,
			       COMMAND_PICKUP,
			       top, 0));
}

// move_to_top and drop_all are used when electrons die.
// trigger functions are NOT called in these to prevent effects.
// if this is wanted anyway later on, add calls to trigger_leave_electron
// and trigger_enter_electron in move_to_top and drop_all.

void move_to_top(Electron *e)
{
    Atom *a = e->get_atom();
    
    // move the player to the top level Object
    while (e->parent != a->top_level_object)
    {
	TRACE("move_to_top: %s (%d) drops %s (%d)",
		e->parent->type_name, e->parent->actor_id,
		e->type_name, e->actor_id);
	

	// order is important here!
	// send it to the client first, otherwise our parent pointer
	// has changed!
	send_queue_packet(new_packet_command(e->parent,
					     COMMAND_DROP,
					     e,
					     0));
	e->parent->inv_drop(e);
    }
}

void drop_all(Electron *e)
{
    List *inv;
    Electron *item;
    
    // drop the inventory, if possible
    if (e->parent)
    {
	inv = e->inv();
	if (inv)
	{
	    for (inv->reset(); (item = (Electron *)inv->get()); inv->reset())
	    {
		TRACE("drop_all: %s (%d) drops %s (%d)",
			e->type_name, e->actor_id,
			item->type_name, item->actor_id);
	
		// let everybody know what's happening
		send_queue_packet(new_packet_command(e,
						     COMMAND_DROP,
						     item,
						     0));

		// drop the item and all it's contents
		e->inv_drop(item);
	    }
	}
    }
    else
    {
	// panic.. no parent to drop things to
	TRACE("drop_all : no parent to drop inventory to!");
    }
}

// find the acting electron in our players atom
Electron *find_acting_electron(Splayer *pl, Packet_command *pc)
{
    Atom *a = pl->get_atom();
    Electron *e;
    
    ASSERT(a);

    e = a->find(pc->actor_type, pc->actor_id);

    if (!e)
    {
	char buf[256];
	sprintf(buf, "%s %d not found",
		global_actor_statics[pc->actor_type]->type_name,
		pc->actor_id);
	message(pl, buf);
	TRACE("find_acting_electron: %s", buf);
    }

    return e;
}

// find the target electron in our players room
// arg1 + arg2 = actor_type
// arg3 + arg4 = actor_id
// that's how it should be
// for now use arg1 = actor_type, arg2 = actor_id
Electron *find_target_electron(Splayer *pl, Packet_command *pc)
{
    Atom *a = pl->get_atom();
    Electron *e;
    
    ASSERT(a);
    
    e = a->find(pc->arg1, pc->larg2());

    if (!e)
    {
	char buf[256];
	sprintf(buf, "%s %d not found",
		global_actor_statics[pc->actor_type]->type_name,
		pc->actor_id);
	message(pl, buf);
	TRACE("find_target_electron: %s", buf);
    }

    return e;
}

void do_put_in(Splayer *pl, Packet_command *pc)
{
    Electron
	*acting = 0,
	*target = 0;

    acting = find_acting_electron(pl, pc);
    target = find_target_electron(pl, pc);

    if (!acting || !target)
    {
	return;
    }

    // check validity for a pickup
    // this is too simple: for now you can only put things in eachother
    // if they are both in your inventory

    if (acting->parent != pl || target->parent != pl)
    {
	TRACE("do_put_in : acting and parent are not both in inventory");
	message(pl, "Both items must be in inventory");
	return;
    }

    if (!acting->can_be_entered(target))
    {
	TRACE("do_put_in : that doesn't fit");
	message(pl, "You can't put that in there");
	return;
    }

    // call leave triggers
    trigger_leave_electron(target, pl);

    // check if we're still alive
    if (pl->dead || target->dead | acting->dead)
	return;
    
    // take target from our inventory and put it in acting
    pl->inv_del(target);
    acting->inv_add(target);

    // let the world know what happened
    send_queue_packet(new_packet_command(acting,
					 COMMAND_PICKUP,
					 target,
					 0));

    // call enter triggers
    trigger_enter_electron(target, acting);
}

void do_get_from(Splayer *pl, Packet_command *pc)
{
    Electron
	*acting = 0,
	*target = 0;

    acting = find_acting_electron(pl, pc);
    target = find_target_electron(pl, pc);

    if (!acting || !target)
    {
	return;
    }

    // check validity for a drop
    // this is too simple: for now you can only take one thing from another
    // if acting is in our inventory and target is in acting's inventory

    if (acting->parent != pl)
    {
	TRACE("do_get_from : acting is not in inventory");
	message(pl, "The container is not in inventory");
	return;
    }

    if (target->parent != acting)
    {
	TRACE("do_get_from : target is not in acting's inventory");
	message(pl, "The Object is not in the container");
	return;
    }

    if (!acting->can_be_entered(target))
    {
	TRACE("do_get_from : it can't come out");
	message(pl, "You can't take it from there");
	return;
    }

    if (!pl->can_be_entered(target))
    {
	TRACE("do_get_from : player can't carry target");
	message(pl, "You can't carry that");
	return;
    }

    // call leave triggers
    trigger_leave_electron(target, acting);

    // check if we're still alive
    if (acting->dead || target->dead || pl->dead)
	return;
    
    // take target from acting's inventory and put it in acting
    acting->inv_del(target);
    pl->inv_add(target);

    // let the world know what happened
    send_queue_packet(new_packet_command(acting,
					 COMMAND_DROP,
					 target,
					 0));

    // call enter triggers
    trigger_enter_electron(target, pl);
}
