/*
 *
 *   ^   |    sssss p   ddddd  fff  ggggg hhhh   iii  j   j    |   ^
 *  /|\  |    s     p   d     f   f   g   h   h i   i jj  j    |  /|\
 *   |   |    sss   p   ddd   f       g   hhhh  i   i j j j    |   |
 *   |  \|/   s     p   d     f   f   g   h   h i   i j  jj   \|/  |
 *   |   v    sssss ppp ddddd  fff    g   h   h  iii  j   j    v   |
 *
 *                           copyright 1999
 *                  Martijn Versteegh & Hein Zelle
 *
 */

/*
 *     handshake.cc
 *
 *     the handshake to say hello to a new player
 *     this file implements a state machine, where the state indicates
 *     how far the handshake has been finished. this is necessary because
 *     the game may not lock up when a new player logs in, so each time
 *     the server has to wait for an answer from the new player, the
 *     player advances to a new state and the server runs another
 *     game_`loop.
 */

#include "globals.h" 
#include "handshake.h"
#include "events.h"
#include "inout.h"
#include "send_q.h"

int error_packet(Packet const *p, int user)
{
    if (p->err)
    {
	warning("check_packet: error in packet (type %d)", p->get_type());
	warning("Packet::error : %s", Packet::error);
	users[user]->state = SHAKE_HANGUP;
    }

    return p->err;
}

// send the number of players, and the name of each player
void send_player_info(int connection)
{
    int nr_players = 0;

    // temporary: count number of players
    // this should be replaced with the size of the player array,
    // or something similar

    for (int i = 0; i < fb.get_nr_plugs(); i++)
    {
	if (fb[i] && i != connection)
	    nr_players++;
    }

    if (nr_players != sys->int_vars[System::NUM_PLAYERS] - 1)
	warning("nr_players %d != %d", nr_players, sys->int_vars[System::NUM_PLAYERS] - 1);
    
    warning("sending the number of players : %d", sys->int_vars[System::NUM_PLAYERS] - 1);
    // send the number of players
    Packet_info_num::write_to(fb[connection],
			      ACTOR_SYSTEM,
			      SYS_VAR,
			      System::NUM_PLAYERS,
			      sys->int_vars[System::NUM_PLAYERS] - 1);    
	    
    warning("sending the names of the players");
    // send the name of each player
    for (int i = 0; i < fb.get_nr_plugs(); i++)
    {
	if (fb[i] && i != connection)
	{
	    char buf[80];
	    sprintf(buf, "player in socket %d", i);
	    warning("sending name %s", buf);
	    Packet_info_string::write_to(fb[connection],
					 ACTOR_SYSTEM,
					 SYS_MESSAGE,
					 System::STC_PLAYER_NAME,
					 buf);
	}
    }
}


void send_map_info(int connection)
{
    // send the name of the map
    warning("sending map name");
    Packet_info_string::write_to(fb[connection],
				 ACTOR_SYSTEM,
				 SYS_VAR,
				 System::MAP_NAME,
				 sys->get_string_var(System::MAP_NAME));

    warning("sending map size: <%d, %d>", lattice->get_w(), lattice->get_h());
    // send the size of the map
    Packet_info_vect::write_to(fb[connection],
			       ACTOR_SYSTEM,
			       SYS_VAR,
			       System::MAP_SIZE,
			       lattice->get_w(), lattice->get_h());
}

void init_client_electron(Electron *e, int connection)
{
    create_client_electron(e, users[connection]->get_send_queue());
    
    // turn vehicles the right way round
    if (e->is_group(GROUP_VEHICLE))
    {
	users[connection]->send_queue_packet(
	    new_packet_command(e,
			       COMMAND_TURN,
			       ((Vehicle *)e)->facing,
			       0, 0, 0, 0));
    }
}

void init_client_atom(Atom *a, int connection)
{
    Electron *e;
    List *l;

    for (int ml = 0; ml < LIST_LAST; ml++)
    {
	l = a->get(ml);
	l->reset();

	while ((e = (Electron *)l->get()))
	{
//	    warning("sending atom (%d, %d)", i, j);
	    
	    init_client_electron(e, connection);
	    l->next();
	}
    }

    // fix the inventory structure at the client side
    hot_potato(users[connection], a->top_level_object);
}

// send lattice should send all electons in the lattice,
// including all info variables.

void send_lattice(int connection)
{
    Atom *a;
    
    warning("sending map size: <%d, %d>", lattice->get_w(), lattice->get_h());
    // send the size of the map
    users[connection]->send_queue_packet(
        new_packet_info_vect(ACTOR_SYSTEM,
			     SYS_VAR,
			     System::MAP_SIZE,
			     lattice->get_w(),
			     lattice->get_h()));

    for (int i = 0; i < lattice->get_w(); i++)
    {
	for (int j = 0; j < lattice->get_h(); j++)
	{
	    a = lattice->get(i,j);

	    if (a)
	    {
		init_client_atom(a, connection);
	    }
	}
    }
}


// return value of net_shake is boolean
// 1 = succes, 0 = error
int net_shake(int connection, int state)
{
    Packet const *p;
    Splayer *pl;
    int vx, vy;          // vote point

    switch (state)
    {
	case SHAKE_PLAYING:
	    // handshake is done, keep the state the same
	    // read commands, queue answers
	    users[connection]->state = playing(connection);
	    break;

	case SHAKE_HANGUP:
	    // delete the player object
	    if (users[connection]->get_player())
	    {
		warning("Deleting player object");
		delete_object(users[connection]->get_player());
	    }

	    // close connection: makes the plug invalid
	    warning("Pulling plug %d. Bye!", connection);
	    fb.close_connection(connection);
	    users[connection]->set_plug(0);
	    
	    // delete the user object
	    warning("Deleting user object");
	    delete_object(users[connection]);
	    users[connection] = 0;

	    // decrease number of players
	    warning("Decreasing number of players");
	    sys->int_vars[System::NUM_PLAYERS]--;
	    break;

	case SHAKE_START:
	    warning("there are %d plugs", fb.get_nr_plugs());

	    sys->int_vars[System::NUM_PLAYERS]++;
		
	    if (connection == fb.get_nr_plugs() - 1)
	    {
		// the last connection was used up: pull the plug
		warning("we're full");
	
		Packet_info_num::write_to(fb[connection],
					  ACTOR_SYSTEM,
					  SYS_MESSAGE,
					  System::STC_I_AM_FULL_GET_LOST,
					  0);
		users[connection]->state = SHAKE_HANGUP;
		break;
	    }
	    else
	    {
		warning("sending version number");
	
		Packet_info_num::write_to(fb[connection],
					  ACTOR_SYSTEM,
					  SYS_VAR,
					  System::VERSION_INT,
					  sys->int_vars[System::VERSION_INT]);
	    }

	    users[connection]->state = SHAKE_VERSION;
	    break;
	    
	case SHAKE_VERSION:
	    p = read_packet_static(fb[connection]);

	    if (!p)
	    {
		// nothing to read? try again next gameloop
		users[connection]->state = SHAKE_VERSION;
		break;
	    }

	    if (error_packet(p, connection)
		|| 
		!Packet_info_num::expect(p, ACTOR_SYSTEM, SYS_VAR,
					 System::VERSION_INT,
					 sys->int_vars[System::VERSION_INT]))
	    {
		warning("Error reading the network version number.");
		users[connection]->state = SHAKE_HANGUP;
		break;
	    }

	    // check to send ONLY players with SHAKE_PLAYING
	    // to avoid crashes.
	    // this is informative only for the client:
	    // not vital information
	    
            // send the player info
	    send_player_info(connection);

	    // send the map statistics
	    send_map_info(connection);
	    
	    users[connection]->state = SHAKE_INTERESTED;
	    break;
	    
	case SHAKE_INTERESTED:
	    // we're getting married, or are we?
	    p = read_packet_static(fb[connection]);

	    if (!p)
	    {
		// nothing to read? try again next gameloop
		users[connection]->state = SHAKE_INTERESTED;
		break;
	    }
	    
	    // see if the client likes the game
	    // if we don't get SYS_YES, we assume they didn't like it.
	    if (error_packet(p, connection)
		||
		!Packet_info_num::expect(p, ACTOR_SYSTEM, SYS_MESSAGE,
					 System::CTS_INTERESTED_YN, SYS_YES))
	    {
		warning("Client on socket %d didn't like the game. Bye!",
			connection);
		users[connection]->state = SHAKE_HANGUP;
		break;
	    }
	    warning("Client on socket %d likes the game. Welcome.",
		    connection);


	    users[connection]->state = SHAKE_PLAYER_NAME;
	    break;

	case SHAKE_PLAYER_NAME:
	    // read the players name
	    p = read_packet_static(fb[connection]);

	    if (!p)
	    {
		// nothing to read? try again next gameloop
		users[connection]->state = SHAKE_PLAYER_NAME;
		break;
	    }

	    // see if the client sends it's name
	    if (error_packet(p, connection)
		||
		!Packet_info_string::expect(p,
					    ACTOR_SYSTEM,
					    SYS_MESSAGE,
					    System::CTS_MY_NAME,
					    0))
	    {
		warning("Client on socket %d did not send their name. Bye!",
			connection);
		users[connection]->state = SHAKE_HANGUP;
		break;
	    }


	    // assign a new player object to the user.  after this, it
	    // is NO LONGER ALLOWED to just set the state to
	    // SHAKE_HANGUP: the player object has to be removed from
	    // the storage list first. In the game, the hangup()
	    // function will take care of this.
	    pl = new_actor<Splayer>(Electron::fid());
	    storage->lowlevel_add(pl);

	    // connect user and player, both ways
	    users[connection]->set_player(pl);
	    pl->set_user(users[connection]);

	    // set player name
	    pl->set_string_var(Splayer::NAME,
			       ((Packet_info_string *)p)->get_string());
	    
	    warning("Client on socket %d : Welcome %s!",
		    connection,
		    pl->get_string_var(Splayer::NAME));

	    // get lattice vote point
	    // put player here, not on the reset point.
	    // reset point is used for games, vote point for logging in
	    lattice->get_vote(&vx, &vy);
    
	    // add the player to the lattice
	    lattice->add(users[connection]->get_player(), vx, vy);
	    
	    // let the other players know what's happening
	    warning("telling all players about our new player %d", connection);
	    create_client_electron(users[connection]->get_player(),
				   &global_send_queue);

	    users[connection]->state = SHAKE_MAP;
	    break;

	case SHAKE_MAP:
	    // send the map to the player
	    // this sends everything: position + vars of all electrons
	    // including all other players and yourself
	    // to prevent network problems (lots of data at once)
	    // the information is sent through the queues.
	    //
	    // need to turn the send_q on here, because otherwise
	    // send_queue_packet won't work. (used by send_lattice)

	    // add the send_q to the global fork so the global
	    // send_queue_packet function will work too
	    // FROM THIS POINT ON THE USERS SEND QUEUE IS ACTIVE!!!
	    global_send_queue.add(users[connection]->get_send_queue());
	    // let the user object know it is now in the fork
	    users[connection]->set_in_fork(1);
	    users[connection]->state = SHAKE_PLAYING;

	    send_lattice(connection);
	    //
	    // !!!!!!!!!!! IMPORTANT !!!!!!!!!!!
	    //
	    // everything after this point HAS to be sent through the queues!
	    
	    // finally, tell the client who they are
	    users[connection]->send_queue_packet(
             		       new_packet_info_num(
		                     ACTOR_SYSTEM,
				     SYS_MESSAGE,
				     System::STC_YOU_ARE,
				     users[connection]->get_player()->actor_id
						  )
					       );
	    
            // give the player their first brake Event
	    evl->add_event(new_brake_event(connection), 1);

	    // turn on the clients send_q
	    // already done above
	    
	    break;

//      DO NOT ADD SHAKE STATES HERE:
//      if a shake_state is added here, the queue code has to be
//      adapted: only when a user is SHAKE_PLAYING is full queue usage allowed,
//      if the user has SHAKE_MAP < state < SHAKE_PLAYING, the queues
// 	are allowed to be filled but NOT flushed, AND the global queue
//      should be copied to the personal queue
//      until the user reaches SHAKE_PLAYING	    
	    
	default:
	    // no valid state? error.
	    users[connection]->state = SHAKE_HANGUP;
	    break;
    }

    // if we manage to get here, something went wrong. each state
    // should return a valid state.
    return 1;
}
