/*
 *
 *   ^   |    sssss p   ddddd  fff  ggggg hhhh   iii  j   j    |   ^
 *  /|\  |    s     p   d     f   f   g   h   h i   i jj  j    |  /|\
 *   |   |    sss   p   ddd   f       g   hhhh  i   i j j j    |   |
 *   |  \|/   s     p   d     f   f   g   h   h i   i j  jj   \|/  |
 *   |   v    sssss ppp ddddd  fff    g   h   h  iii  j   j    v   |
 *
 *                           copyright 1999
 *                  Martijn Versteegh & Hein Zelle
 *
 */

/*
 *     gameloop.cc
 *
 *     main game loop of the Electron server
 */

// include some stuff for sched_yield and gettimeofday
#include <sched.h>
#include <sys/time.h>
#include <unistd.h>

#include "globals.h"
#include "handshake.h"

Packet const *get_packet(int con)
{
    Packet const *p;
    
    // this loop stops the game until another packet arrives.
    // NOT TO BE USED FOR MULTIPLE PLAYERS!
    while (1)
    {
        p = read_packet_static(fb[con]);
        if (p)
        {
            if (p->err)
            {
                return 0;
            }
            else
                return p;
        }
    }

    return p;
}

void ping_client(int connection)
{
    int i = 0;
    
    warning("let's start pinging");
    
    while (1)
    {
	warning("ping %d", i++);
	Packet_command::write_to(fb[connection], 0, 0, 0, 0, 0, 0, 0);
	if (get_packet(connection))
	    warning("got a packet from the client");
    }
}

// fills start with now
// fills end with now + interval
void set_interval(timeval *start, timeval *end, timeval const *interval)
{
    // fill start with the current time
    gettimeofday(start, 0);

    // fill end with start + interval (take care of usec overflow)
    end->tv_sec = start->tv_sec + interval->tv_sec;
    end->tv_sec += (start->tv_usec + interval->tv_usec) / 1000000;
    end->tv_usec = (start->tv_usec + interval->tv_usec) % 1000000;
}

// 1: end_time < now
// 0: end_time >= now
int time_to_spare(timeval const *end_time)
{
    static timeval now_time;

    gettimeofday(&now_time, 0);

    if (
	now_time.tv_sec < end_time->tv_sec
	||
	(
	 now_time.tv_sec == end_time->tv_sec
	 &&
	 now_time.tv_usec < end_time->tv_usec
	)
       )
    {
	return 1;
    }

    return 0;
}

// 1 = success
int state_machine()
{
    // enter the state machine
    for (int i = 0; i < fb.get_nr_plugs(); i++)
    {
	if (fb[i])
	{
	    // if the socket exists, check if it's already playing
	    // if not, shake hands

	    if (users[i]->state <= SHAKE_PLAYING)
	    {
		int state;
		state = users[i]->state;

		// net_shake can change the users state!
		net_shake(i, users[i]->state);

		// the user can become invalid in net_shake: CHECK!
		
		if (users[i] && state != users[i]->state)
		{
		    warning("plug %d: state %d", i, users[i]->state);
		}
	    }		    
	}
    }

    return 1;
}

int game_loop()
{
    int	new_connection;
    static timeval tshort = {0, 10000};
    static timeval tlong  = {0, 100000};
    static timeval tstartlong;
    static timeval tendlong;
//    static timeval tstartshort;
//    static timeval tendshort;
    
//    static int i = 0;

    set_interval(&tstartlong, &tendlong, &tlong);
    
    new_connection = fb.check_new_connection();

//    warning("%d, new_connection: %d", i++, new_connection);
    
    if (new_connection >= 0)
    {
	// initialise the user object
	// new connection, start of the handshake

	// don't give the user a player yet. this happens
	// within the handshake routine, as late as possible.
	users[new_connection] = new_user(fb[new_connection]);
    }

    // run the state machine
    // read all network info from the plugs
    // treat users that are logging in (state machine < SHAKE_PLAYING)
    state_machine();
    
    // handle all events
    try
    {
	// advance the clock 10 pulses. the server only does 10 gameloops
	// per second, but the clock has 100 ticks per second
	evl->tick(10);

	// advance the game clock, if there is one
	if (game && game->get_clock())
	{
	    game->get_clock()->tick(10);
	}
    }
    catch(int shutdown)
    {
	// if a shutdown int is thrown, shut down the game
	warning("server shutting down: exception %d", shutdown);
	return 0;
    }
	
    // empty all send queues
    send_queues();
    
    // give the system a break
    // game_loop should run only 10 times per second
    // if we have time to spare, we do the network loop 100 times per second
    // until our time is up.

    // clear the garbage list
    empty_garbage();
    
    while (time_to_spare(&tendlong))
    {
/*	
	// time to spare: start doing extra network reads
	set_interval(&tstartshort, &tendshort, &tshort);

	while (time_to_spare(&tendshort))
	{
	    // give the cpu a break
	    // usleep(1000);  // unreliable
	    // sched_yield();
	}
*/

	tshort.tv_sec = 0;
	tshort.tv_usec = 10000;
	
	select(0, 0, 0, 0, &tshort);
    }

//    warning("game loop %d ! :-)", i++);
    
    return 1;
}
