/*
 *
 *   ^   |    sssss p   ddddd  fff  ggggg hhhh   iii  j   j    |   ^
 *  /|\  |    s     p   d     f   f   g   h   h i   i jj  j    |  /|\
 *   |   |    sss   p   ddd   f       g   hhhh  i   i j j j    |   |
 *   |  \|/   s     p   d     f   f   g   h   h i   i j  jj   \|/  |
 *   |   v    sssss ppp ddddd  fff    g   h   h  iii  j   j    v   |
 *
 *                           copyright 1999
 *                  Martijn Versteegh & Hein Zelle
 *
 */

/*
 *          twoteam.cc
 *
 *          class implementation of a Twoteam war game
 */

#include "twoteam.h"
#include "server/message.h"
#include "server/usercomm.h"

// allocate a new twoteam game
Game *new_twoteam()
{
    Twoteam *tw;

    tw = (Twoteam *) new_object(SERVER_FREELIST_TWOTEAM_GAME);

    if (!tw)
    {
	tw = new Twoteam();
    }

    tw->set();
    return tw;
}

Twoteam::Twoteam()
{
    freelist_index = SERVER_FREELIST_TWOTEAM_GAME;
    game_type = TWOTEAM_GAME;
    set();
}

void Twoteam::set()
{
    // request 2 teams for playing
    Game::set(0, 0, 0, 2);
}

void Twoteam::set(Svotemachine *vm, Sweaponmachine *wm, Clock *cl)
{
    // request two teams for playing
    Game::set(vm, wm, cl, 2);
}

void Twoteam::clear()
{
    Game::clear();
}

int Twoteam::init_game()
{
    // Split the players in two teams
    // this should be done by first sorting them by level or experience
    // then splitting them one-on-one for equal division.
    // Since, however, we do not know the level or experience of a
    // player, just split them up without further thinking.

    int t = 0;
    char buf[256];

    // count players while composing teams.
    // not very nice if we end up having too little, but the
    // game will clean itself up
    
    // select all available players
    for (int u = 0; u < MAX_PLAYERS; u++)
    {
	if (users[u] && users[u]->state == SHAKE_PLAYING)
	{
	    Splayer *pl = users[u]->get_player();

	    // now add to the proper team
	    teams[t % 2]->push(pl);

	    // let everybody know what happened
	    sprintf(buf, "%s %s (%d) is on team %d",
		    pl->type_name, pl->get_string_var(Player::NAME),
		    pl->actor_id, t % 2);
	    message_all(buf);

	    // found a valid player
	    t++;
	}
    }

    // check if we have enough players
    if (t < 2)
    {
	message_all("Two players is the minimum for a two team game");
	return 0;
    }

    return 1;
}

// custom die function for players.
// handle personal and team scores where applicable
// when a player dies, remove them from the game.
// move to a safe room, but leave them in the team
// so they can still be scored and printed
//
// pay attention: k can be NULL !!!!
int Twoteam::die_player(Splayer *pl, Electron *k)
{
    int rx, ry;
    Atom *a;
    Splayer *killer;
    char buf[256];
    Team *kt = 0;
    Team *plt = 0;

    // do something fancy

    if (k)
    {
	sprintf(buf, "Twoteam: %s %d killed by %s %d",
		pl->type_name, pl->actor_id,
		k->type_name, k->actor_id);
    }
    else
    {
		sprintf(buf, "Twoteam: %s %d died",
			pl->type_name, pl->actor_id);
    }

    message_all(buf);
	
    // get the players team
    plt = pl->get_team();

    if (k && k->actor_type == ACTOR_PLAYER)
    {
	killer = (Splayer *)k;
	kt = killer->get_team();
	ASSERT(kt && plt);
    
	warning("kt->score: %d, plt->score: %d", kt->score, plt->score);
    
	// killed a player, raise kill count
	if (k != pl)
	{
	    // penalise for killing teammates
	    if (kt == plt)
	    {
		killer->int_vars[Player::KILLS]--;
		kt->score--;
		
		sprintf(buf,
			"You killed player %d\n"
			"That was your teammate!\n"
			"You have %d kills and died %d times",
			pl->actor_id,
			killer->int_vars[Player::KILLS],
			killer->int_vars[Player::DEATHS]);
	    }
	    else
	    {
		killer->int_vars[Player::KILLS]++;
		kt->score++;
		
		sprintf(buf,
			"You killed player %d\n"
			"You have %d kills and died %d times",
			pl->actor_id,
			killer->int_vars[Player::KILLS],
			killer->int_vars[Player::DEATHS]);
	    }
	}
	else
	{
	    sprintf(buf, "You committed suicide.");
	}
	
	message(killer, buf);
    }

    // we died, raise our death count
    pl->int_vars[Player::DEATHS]++;

    if (k)
    {
        sprintf(buf,
		"You were killed by %s %d\n"
		"You have %d kills and died %d times",
		k->type_name,
		k->actor_id,
		pl->int_vars[Player::KILLS],
		pl->int_vars[Player::DEATHS]);
    }
    else
    {
        sprintf(buf,
		"You died.\n"
		"You have %d kills and died %d times",
		pl->int_vars[Player::KILLS],
		pl->int_vars[Player::DEATHS]);
    }
    
    message(pl, buf);
    
    // drop all items into the top level Object
    move_to_top(pl);
    drop_all(pl);


    // changed: move to vote room when dead.
    
    // do not use do_move_electron(e, x, y, time)
    // this checks for cyclewalls!
    
    lattice->get_reset(&rx, &ry);

    warning("moving player %d to vote room [%d, %d]",
	    pl->actor_id, rx, ry);

    a = lattice->get(rx, ry);
    do_move_electron(pl, a, 0);

    // stop running
    pl->set_running(0);
    
    // reset player variables (h, m, d)
    pl->reset();
    pl->write_vars(packet_to_queue, &global_send_queue);

    // let the dying continue
    return 1;
}

void Twoteam::stop()
{
    char buf[1024];
    
    sprintf(buf,
	    "Game stopped.\n"
	    "Team %d score: %d\n"
	    "Team %d score: %d",
	    0, teams[0]->score,
	    1, teams[1]->score);
    message_all(buf);

    Game::stop();
}
