/*
 *
 *   ^   |    sssss p   ddddd  fff  ggggg hhhh   iii  j   j    |   ^
 *  /|\  |    s     p   d     f   f   g   h   h i   i jj  j    |  /|\
 *   |   |    sss   p   ddd   f       g   hhhh  i   i j j j    |   |
 *   |  \|/   s     p   d     f   f   g   h   h i   i j  jj   \|/  |
 *   |   v    sssss ppp ddddd  fff    g   h   h  iii  j   j    v   |
 *
 *                           copyright 1999
 *                  Martijn Versteegh & Hein Zelle
 *
 */

/*
 *          nogame.cc
 *
 *          class implementation of Nogame: a dummy game
 *          to use when no game is active
 */

#include "nogame.h"
#include "server/message.h"
#include "server/usercomm.h"
#include "server/globals.h"
#include "server/game/vote.h"
#include "server/eventlis.h"

// allocate a nogame game
Game *new_nogame()
{
    Nogame *n;

    n = (Nogame *) new_object(SERVER_FREELIST_NO_GAME);

    if (!n)
    {
	n = new Nogame();
    }

    n->set();
    return n;
}

Nogame::Nogame()
{
    freelist_index = SERVER_FREELIST_NO_GAME;
    game_type = NO_GAME;
    set();
}

void Nogame::set()
{
    // request 0 teams for playing
    Game::set(0, 0, 0, 0);
    voting_started = 0;
    ev = 0;
}

void Nogame::set(Svotemachine *vm, Sweaponmachine *wm, Clock *cl)
{
    warning("Nogame::set : called with machines? ignored!");
    
    // ignore machines, and request 0 teams
    Game::set(0, 0, 0, 0);
    voting_started = 0;
}

void Nogame::clear()
{
    Game::clear();
    voting_started = 0;

    // if we have a game_event set, stop it.
    if (ev)
    {
	ev->stop();
	ev = 0;
    }
}

int Nogame::init_game()
{
    // this one has to be overloaded, even for nogame
    // print a message that no game is active, otherwise do nothing

    // reset votes
    reset_votes();
    
    message_all("No game is currently running, waiting for votes.");

    return 1;
}

// overload these 2 to return 0 so it doesn't really happen
int Nogame::die_player(Splayer *pl, Electron *k)
{
    warning("Nogame::die_player");
    message_all("No game is currently running, dying is not possible.");
    return 0;
}

int Nogame::damage_player(Splayer *pl, Splayer *bully)
{
    warning("Nogame::damage_player");
    message_all("No game is currently running, damage is not possible.");
    return 0;
}

// someone just voted. See if we have to start a game_event
void Nogame::vote(Splayer *pl)
{
    int nr_votes = 0;
    
    // nogame version; check if we have to start a game
    if (!voting_started)
    {
	char buf[80];
	
	voting_started = 1;

	// if we have an event set here, something is very wrong.
	ASSERT(!ev);

	ev = new_game_event(Game_event::VOTE, 0);
	evl->add_event(ev, duration);

	ssprintf(buf, 80, "Game will start in %d seconds. Vote now!",
		 duration/100);
	message_all(buf);
    }

    // check if all players have voted
    // count number of votes

    for (int g = 0; g < GAME_LAST; g++)
    {
	nr_votes += game_type_list[g].votes;
    }

    if (nr_votes >= sys->int_vars[System::NUM_PLAYERS])
    {
	// everyone has voted. Don't wait anymore.
	message_all("Everyone has voted, game starts in 5 seconds.");

	ev->stop();        // make sure it doesn't go off later

	// set a new (fast) event (5 second)
	ev = new_game_event(Game_event::VOTE, 0);
	evl->add_event(ev, 500);
    }
}

void Nogame::announce()
{
    // do nothing so this can be called safely
}

void Nogame::start_new_game()
{
    int g = 0;
    int max = 0;
    
    // find the game with the most votes

    // skip nogame (==0)
    for (g = 1; g < GAME_LAST; g++)
    {
	if (game_type_list[g].votes > game_type_list[max].votes)
	    max = g;
    }

    if (!max)
    {
	message_all("No votes were brought out, waiting for new votes.");
    }

    // select the new game, and replace the current game with it
    game = game_type_list[max].fun();
    ASSERT(game);   // this MUST work, otherwise rethink strategy

    // initialize game
    if (!game->init_game())
    {
	char buf[80];
	ssprintf(buf, 80, "%s failed to initialize.",
		 game_type_list[max].name);
	message_all(buf);
	warning(buf);
	
	// delete game we just created
	delete_object(game);
	game = 0;
	
	// select nogame after all
	set_no_game();
    }

    // it's safe to call this for all games
    game->announce();

    // finally, delete the calling instance of no_game: this
    delete_object(this);
}
