/*
 *
 *   ^   |    sssss p   ddddd  fff  ggggg hhhh   iii  j   j    |   ^
 *  /|\  |    s     p   d     f   f   g   h   h i   i jj  j    |  /|\
 *   |   |    sss   p   ddd   f       g   hhhh  i   i j j j    |   |
 *   |  \|/   s     p   d     f   f   g   h   h i   i j  jj   \|/  |
 *   |   v    sssss ppp ddddd  fff    g   h   h  iii  j   j    v   |
 *
 *                           copyright 1999
 *                  Martijn Versteegh & Hein Zelle
 *
 */

/*
 *          game.cc
 *
 *          class implementation of Game
 *          Game is to be a baseclass for Gametype, which will contain
 *          specific information for each game type.
 *
 *          Game will contain all data and functions that the game types
 *          have in common.
 */

#include "server/s_freelist.h"
#include "server/globals.h"
#include "server/game/game.h"
#include "server/game/vote.h"
#include "server/game/nogame.h"
#include "server/game/tag.h"
#include "server/game/twoteam.h"
#include "server/message.h"

game_type game_type_list[GAME_LAST] =
{
//   id              name            loadfunc    votes  duration (0.01 seconds)
    {NO_GAME,       "no game",       new_nogame,   0,   60*100}, // vote time
    {TAG_GAME,      "tag game",      new_tag,      0,   300*100},
    {TWOTEAM_GAME,  "twoteam game",  new_twoteam,  0,   300*100},
};

Game::Game()
{
    // initialise game_type to 0 before calling set: it needs a valid value
    // and because derived classes may set it, it cannot be set in Game::set
    game_type = 0;
    set();
}

// this is called twice: once from the constructor of Game, once
// from the set() function of the derived class. Can't bother to fix it, now.
void Game::set()
{
    // no arguments: set everything to 0
    startpoints.clear();

    vote_area[0] = vote_area[1] = 0;

    teams = 0;
    nr_teams = 0;

    vote_machine = 0;
    weapon_machine = 0;

    clock = 0;
    running = 0;
    duration = game_type_list[game_type].duration;
}

void Game::set(Svotemachine *vm, Sweaponmachine *wm, Clock *cl, int n_teams)
{
    startpoints.clear();

    vote_area[0] = vote_area[1] = 0;

    if (teams)
    {
	for (int i = 0; i < nr_teams; i++)
	{
	    delete_object(teams[i]);
	    teams[i] = 0;
	}
	delete teams;
	teams = 0;
    }

    nr_teams = n_teams;
    teams = new Team * [n_teams];
    for (int i = 0; i < nr_teams; i++)
    {
	teams[i] = new_team();
    }

    vote_machine = vm;
    weapon_machine = wm;
    clock = cl;

    duration = 0;
    running = 0;
    duration = game_type_list[game_type].duration;    
}

void Game::clear()
{
    startpoints.clear();

    vote_area[0] = vote_area[1] = 0;

    if (teams)
    {
	for (int i = 0; i < nr_teams; i++)
	{
            delete_object(teams[i]);            // recycle teams
	    teams[i] = 0;
	}
	delete teams;
	teams = 0;
    }
    nr_teams = 0;

    vote_machine = 0;
    weapon_machine = 0;

    if (clock)
    {
        delete_object(clock);                   // recycle clock
	clock = 0;
    }

    duration = 0;
    running = 0;
}

// find the first weaponmachine
Sweaponmachine *Game::find_weapon_machine()
{
    return (Sweaponmachine *)storage->find_type(ACTOR_WEAPONMACHINE);
}

// find the first votemachine
Svotemachine *Game::find_vote_machine()
{
    return (Svotemachine *)storage->find_type(ACTOR_VOTEMACHINE);
}

void Game::announce()
{
    char buf[80];
    
    ASSERT(!clock);

    // set clock, weapon machine and vote machine
    clock = new_clock(duration);
    weapon_machine = find_weapon_machine();
    vote_machine = find_vote_machine();
    
    sprintf(buf, "starting a new %s, duration %d seconds",
	    game_type_list[game_type].name, duration/100);
    message_all(buf);
    
    // start the clock. game starts when START event kicks in
    clock->start();
}

void Game::start()
{
    Splayer *pl;

    // load weapons for all players if possible
    if (weapon_machine)
	weapon_machine->load_weapons();
    
    for (int i = 0; i < nr_teams; i++)
    {
	for (teams[i]->reset();
	     (pl = (Splayer *)teams[i]->get());
	     teams[i]->next())
	{
	    pl->set_running(1);
	}
    }

    message_all("The game has started.");
    
    running = 1;
}

void Game::stop()
{
    Splayer *pl;

    for (int i = 0; i < nr_teams; i++)
    {
	for (teams[i]->reset();
	     (pl = (Splayer *)teams[i]->get());
	     teams[i]->next())
	{
	    pl->set_running(0);
	}
    }

    clock->stop();

    message_all("The game has ended.");
    
    running = 0;
    set_no_game();
}

void Game::warn()
{
    message_all("Warning: game ends in 10 seconds.");
}

void Game::check()
{
    // default function, overload for different behaviour
    // check the teams, if only one team has live members, stop
    Splayer *pl;

    int live_teams  = 0;
    
    for (int i = 0; i < nr_teams; i++)
    {
	for (teams[i]->reset();
	     (pl = (Splayer *)teams[i]->get());
	     teams[i]->next())
	{
	    if (pl->get_running())
	    {
		live_teams++;         // found a live player
		break;
	    }
	}
    }

    warning("check: %d live teams found\n", live_teams);
    
    if (live_teams < 2)
    {
	// stop the game, one team cannot play against itself
	stop();
    }
}

int Game::get_running() const
{
    return running;
}

int Game::die_player(Splayer *pl, Electron *k)
{
    // let the player die. overload this to 0 if you don't want
    // them to die
    return 1;
}

int Game::damage_player(Splayer *pl, Splayer *bully)
{
    // let the damage dealing continue
    // overload this to 0 if you don't want them to get damaged
    return 1;
}


// global function to delete active game (if any) and
// to install a Nogame

void set_no_game()
{
    if (game)
    {
	// stop only if running
	if (game->get_running())
	    game->stop();

        // delete old game
	delete_object(game);
    }

    // set nogame
    game = new_nogame();
    game->init_game();
}

Clock *Game::get_clock()
{
    return clock;
}

// someone just voted. See if we have to start a game_event
void Game::vote(Splayer *pl)
{
    // default version: don't do anything

    message(pl, "You cannot vote while a game is running.");
}

void Game::start_new_game()
{
    warning("Game::start_new_game : BUG : should only be called for no_game,"
	    "this should never be reached.");
    ASSERT(0);
}
     
