/*
 *
 *   ^   |    sssss p   ddddd  fff  ggggg hhhh   iii  j   j    |   ^
 *  /|\  |    s     p   d     f   f   g   h   h i   i jj  j    |  /|\
 *   |   |    sss   p   ddd   f       g   hhhh  i   i j j j    |   |
 *   |  \|/   s     p   d     f   f   g   h   h i   i j  jj   \|/  |
 *   |   v    sssss ppp ddddd  fff    g   h   h  iii  j   j    v   |
 *
 *                           copyright 1999
 *                  Martijn Versteegh & Hein Zelle
 *
 */

/*
 *           create.cc
 *
 *           utility functions to create electrons
 *           (or to create them at the client side)
 */

#include "send_q.h"
#include "globals.h"
#include "usercomm.h"
#include "message.h"

// send a new electron to the client, plus all it's information
// this actually creates a new actor at the client side
// one version for sending directly to a Plug
// one version for sending through a queue (List)

void create_client_electron(Electron *e, Plug *p)
{
    Packet_command::write_to(p,
			     e->actor_type,
			     e->actor_id,
			     COMMAND_FROM_TO,
			     -1, -1, e->gx(), e->gy());

    e->write_vars(packet_to_plug, p);
}

void create_client_electron(Electron *e, List *q)
{
    Packet_command *pc = new_packet_command(e, COMMAND_FROM_TO,
					    -1, -1, e->gx(), e->gy(),
					    0);

    q->push(pc);
    e->write_vars(packet_to_queue, q);
}

void print_inv(Electron *e)
{
    warning(" ----- inventory of %s %d ----- ", e->type_name, e->actor_id);

    List *inv = e->inv();

    if (inv)
    {
	Electron *item;
	
	inv->reset();

	while ((item = (Electron *)inv->get()))
	{
	    print_inv(item);
	    inv->next();
	}
    }

    warning(" ----- inventory end %s %d ----- ", e->type_name, e->actor_id);
}

void destroy_electron(Electron *e)
{
    // remove the electron from the lattice and from storage
    // let the world know, and delete it

//    Electron *r = e->get_atom()->top_level_object;
    
    drop_all(e);
    
    send_queue_packet(new_packet_command(e,
					 COMMAND_FROM_TO,
					 e->gx(),
					 e->gy(),
					 -1, -1, 0));

//    print_inv(r);
    
    lattice->remove(e, e->gx(), e->gy());
    storage->lowlevel_remove(e);

//    print_inv(r);

    // use garbage instead of freelist
    // this is a bit safer in loops over linked lists where objects die
    // during the loop.
    // delete_object(e);
    garbage_electron(e);

    warning("destroy_electron finished");
}

void die_electron(Electron *e, Electron *killer)
{
    if (e->is_group(GROUP_VEHICLE))
    {
	// check if there are players in the inventory
	// if so, let them know their vehicle blew up
	
	Electron *pl;
	List *l = e->inv();

	if (l)
	{
	    l->reset();
	    while ((pl = (Electron *)l->get()))
	    {
		if (pl->actor_type == ACTOR_PLAYER)
		{
		    char buf[80];
		    sprintf(buf, "Your %s just blew up.", e->type_name);
		    message((Splayer *)pl, buf);
		}
		   
		l->next();
	    }
	}
    }

    warning("%s (%d) died", e->type_name, e->actor_id);
    destroy_electron(e);
}

void die_player(Splayer *pl, Electron *k)
{
    int rx, ry;
    Atom *a;
    Splayer *killer;
    char buf[256];
    
    lattice->get_reset(&rx, &ry);
    a = lattice->get(rx, ry);
    
    if (k && k->actor_type == ACTOR_PLAYER)
    {
	killer = (Splayer *)k;
	// killed a player, raise kill count
	if (k != pl)
	{
	    killer->int_vars[Player::KILLS]++;
	    sprintf(buf,
		    "You killed player %d\n"
		    "You have %d kills and died %d times",
		    pl->actor_id,
		    killer->int_vars[Player::KILLS],
		    killer->int_vars[Player::DEATHS]);
	}
	else
	{
	    sprintf(buf, "You committed suicide.");
	}
	
	message(killer, buf);
    }

    // we died, raise our death count
    pl->int_vars[Player::DEATHS]++;

    if (k)
    {
        sprintf(buf,
		"You were killed by %s %d\n"
		"You have %d kills and died %d times",
		k->type_name,
		k->actor_id,
		pl->int_vars[Player::KILLS],
		pl->int_vars[Player::DEATHS]);
    }
    else
    {
        sprintf(buf,
		"You died.\n"
		"You have %d kills and died %d times",
		pl->int_vars[Player::KILLS],
		pl->int_vars[Player::DEATHS]);
    }
    
    message(pl, buf);
    
    // drop all items into the top level Object
    move_to_top(pl);
    drop_all(pl);

    warning("moving player %d to reset room [%d, %d]",
	    pl->actor_id, rx, ry);
    
    // move the player to the reset room
    // do not use do_move_electron(e, x, y, time)
    // this checks for cyclewalls!
    do_move_electron(pl, a, 0);

    // reset player variables
    pl->reset();
    pl->write_vars(packet_to_queue, &global_send_queue);
}

// toss a dead electron in the garbage list
void garbage_electron(Electron *e)
{
    e->clear();
    garbage->lowlevel_add(e);
}

// clean the garbage atom (delete_object): return how many were in it
int empty_garbage()
{
    List *to_clear;
    Electron *to_delete;
    int total = 0;
    
    for (int i = 0; i < LIST_LAST; i++)
    {
	to_clear = garbage->get(i);
	while((to_delete = (Electron *)to_clear->pop()))
	{
	    delete_object(to_delete);
	    total++;
	}
    }

    return total;
}
