/*
 *
 *   ^   |    sssss p   ddddd  fff  ggggg hhhh   iii  j   j    |   ^
 *  /|\  |    s     p   d     f   f   g   h   h i   i jj  j    |  /|\
 *   |   |    sss   p   ddd   f       g   hhhh  i   i j j j    |   |
 *   |  \|/   s     p   d     f   f   g   h   h i   i j  jj   \|/  |
 *   |   v    sssss ppp ddddd  fff    g   h   h  iii  j   j    v   |
 *
 *                           copyright 1999
 *                  Martijn Versteegh & Hein Zelle
 *
 */

/*
 *          sweaponmachine.cc
 */

#include "server/globals.h"
#include "sweaponmachine.h"
#include "server/message.h"

load_weapon_struct weapon_table[WEAPON_LAST] =
{
    { WEAPON_NONE,       0,                load_none },
    { WEAPON_LIGHTCYCLE, ACTOR_LIGHTCYCLE, load_lightcycle },
    { WEAPON_TANK,       ACTOR_TANK,       load_tank },
    { WEAPON_LAUNCHER,   ACTOR_LAUNCHER,   load_launcher },
    { WEAPON_HOVER,      ACTOR_HOVER,      load_hover },
};

void Sweaponmachine::set(int _actor_id)
{
    Weaponmachine::set(_actor_id);
}

void Sweaponmachine::clear()
{
    Weaponmachine::clear();
}

int Sweaponmachine::select_weapon(Splayer *pl, Packet_command *pc)
{
    char buf[80];
    int wtype = pc->arg1;

    // check for valid weapon type
    if (wtype <= WEAPON_NONE || wtype >= WEAPON_LAST)
    {
	message(pl,  "Invalid weapon type. Valid choices:");

	for (int w = WEAPON_NONE + 1; w < WEAPON_LAST; w++)
	{
	    if (w == pl->get_user()->get_weapon())
	    {
		sprintf(buf, "[%2d] %s *", w,
			global_actor_statics[weapon_table[w].actor_type]
			->type_name);
	    }
	    else
	    {
		sprintf(buf, "[%2d] %s", w,
			global_actor_statics[weapon_table[w].actor_type]
			->type_name);		
	    }
	    message(pl, buf);
	}
	return 0;
    }

    // valid type found: select the weapon.
    // it will be loaded by load_weapons() when the game starts.

    pl->get_user()->set_weapon(wtype);
    return 1;
}

int load_none(Splayer *pl)
{
    // do nothing: player stays where it is and doesn't receive a weapon.
    return 1;
}

int load_lightcycle(Splayer *pl)
{
    // give the player a lightcycle
    Slightcycle *lc;
    int retval;
    
    lc = new_actor<Slightcycle>(Electron::fid());
    storage->lowlevel_add(lc);

    retval = load_vehicle(pl, lc);
    return retval;
}

int load_launcher(Splayer *pl)
{
    Slauncher *l;
    Sred_ball *rb;
    Syellow_ball *yb;
    Sblue_ball *bb;
    int retval;

    // load a launcher
    l = new_actor<Slauncher>(Electron::fid());
    storage->lowlevel_add(l);

    // load three balls
    rb = new_actor<Sred_ball>(Electron::fid());
    bb = new_actor<Sblue_ball>(Electron::fid());
    yb = new_actor<Syellow_ball>(Electron::fid());
    storage->lowlevel_add(rb);
    storage->lowlevel_add(bb);
    storage->lowlevel_add(yb);

    // this creates (client side) the launcher and the balls
    // they are left on the floor in the player's room.
    retval = load_held_weapon(pl, rb);
    retval = load_held_weapon(pl, bb);
    retval = load_held_weapon(pl, yb);
    retval = load_held_weapon(pl, l);

    // put the launcher in the player
    lattice->move_electron(l, pl);
    
    // put the balls in the launcher
    lattice->move_electron(rb, l);
    lattice->move_electron(bb, l);
    lattice->move_electron(yb, l);

    // fix player inventory only once!
    hot_potato(pl, &global_send_queue);
    
    return retval;
}

int load_tank(Splayer *pl)
{
    // load a tank, then call load_vehicle with the new tank
    Stank *t;
    int retval;
    
    t = new_actor<Stank>(Electron::fid());
    storage->lowlevel_add(t);
    
    retval = load_vehicle(pl, t);
    return retval;
}

int load_hover(Splayer *pl)
{
    // load a hover, then call load_vehicle with the new hover
    Shover *h;
    int retval;
    
    h = new_actor<Shover>(Electron::fid());
    storage->lowlevel_add(h);
    
    retval = load_vehicle(pl, h);
    return retval;
}

int load_held_weapon(Splayer *pl, Electron *e)
{
    // give the player a held weapon
    Atom *a;

    // get player's atom and add the weapon
    a = pl->get_atom();
    ASSERT(a);
    a->add(e);        // put object in player's room
    
    // tell the players what happened
    create_client_electron(e, &global_send_queue);

    return 1;
}

int load_vehicle(Splayer *pl, Vehicle *v)
{
    // give the player a key.
    // connect the key to the vehicle
    // place the vehicle in the map.
    // move the player to the vehicle

    Atom *a, *old_a;
    Skey *k;

    // new key
    k = new_actor<Skey>(Electron::fid());

    // connect key and vehicle
    v->add_key(k);
    k->set_vehicle(v);

    // give the key to the player
    a = pl->get_atom();
    ASSERT(a);
    a->add(k, pl);

    // add the vehicle to the lattice
    // make sure the vehicle can enter the room it's put in
    a = lattice->get_random_atom(v);
    ASSERT(a);
    a->add(v); // put the vehicle in the atom. top_level_object becomes parent.

    // move the player into the vehicle (was next to the vehicle before)
    old_a = pl->get_atom();
//    lattice->move_electron(pl, v->parent);
    lattice->move_electron(pl, v);
    
    // let the client know the player moved
    send_queue_packet(new_packet_command(pl,
					 COMMAND_FROM_TO,
					 old_a->x, old_a->y,
					 a->x, a->y,
					 0));

    // let the client know of the new electrons
    create_client_electron(k, &global_send_queue);
    create_client_electron(v, &global_send_queue);
    
    // fix the inventory structure of the player at the client side
//    hot_potato(pl, &global_send_queue);    
    hot_potato(v, &global_send_queue);
    
    return 1;
}

int Sweaponmachine::load_weapons()
{
    // game is starting: hand out the weapons.

    // run by all players, check their selection and go.
    
    Splayer *pl;
    int wtype;

    // select all available players
    for (int u = 0; u < MAX_PLAYERS; u++)
    {
	if (users[u] && users[u]->state == SHAKE_PLAYING)
	{
	    pl = users[u]->get_player();
	    ASSERT(pl);

	    // don't check weapontype, this has already been done.
	    wtype = users[u]->get_weapon();

	    if (!weapon_table[wtype].load_weapon(pl))
	    {
		warning("Sweaponmachine::load_weapons : failed to load weapon type %d"
			"for player %d.", wtype, pl->actor_id);
	    }
	}
    }

    return 1;
}
