/*
 *
 *   ^   |    sssss p   ddddd  fff  ggggg hhhh   iii  j   j    |   ^
 *  /|\  |    s     p   d     f   f   g   h   h i   i jj  j    |  /|\
 *   |   |    sss   p   ddd   f       g   hhhh  i   i j j j    |   |
 *   |  \|/   s     p   d     f   f   g   h   h i   i j  jj   \|/  |
 *   |   v    sssss ppp ddddd  fff    g   h   h  iii  j   j    v   |
 *
 *                           copyright 1999
 *                  Martijn Versteegh & Hein Zelle
 *
 */

/*
 *       stank.cc
 *
 *       server version of the Tank
 */

#include "stank.h"
#include "server/globals.h"
#include "server/usercomm.h"
#include "server/fire.h"
#include "server/message.h"

void Stank::set(int _actor_id)
{
    Tank::set(_actor_id);
    driving = 0;
    drive_time = 60;
    reloading = 0;
    reload_time = 500;
    fire_range = 20.0;
    flight_time = 150;
    target_range = 10.0;
    turn_time = 100;
    facing = 2;
}

void Stank::clear()
{
    Tank::clear();
    driving = 0;
    reloading = 0;
}

int Stank::get_reloading()
{
    return reloading;
}

void Stank::set_reloading(int yesno)
{
    reloading = yesno;
}

int Stank::get_driving()
{
    return driving;
}

void Stank::set_driving(int yesno)
{
    driving = yesno;
}

void Stank::die(Electron *killer)
{
    die_electron(this, killer);
}

int Stank::move_dir(int dir, Electron *pl)
{
    int
	tx,
	ty;

    int
	ret,
        spent = 0;

    if (get_driving())
    {
	error_message = "Tank is already moving.";
	return 0;
    }

    TRACE("Stank::move_dir : facing = %d, dir = %d",
	    facing, dir);

    // if a tank is facing the wrong way, don't move
    if (facing != dir)
    {
	error_message = "Tank is facing the wrong way.";
	return 0;
    }

    // compute target coordinates
    if (!get_dir_coordinates(this, dir, &tx, &ty))
	return 0;
    
    // try and move the tank
    TRACE("Stank::move_dir: try moving 1 room in direction %d", dir);

    // moving takes 0.5 seconds
    ret = do_move_electron(this, tx, ty, drive_time);

    if (ret)
    {
	// we moved, add some spent time (less than drive time!)
	spent += 20;
    }

    if (spent > 0)
    {
	// only set the 'driving' flag if it's taking some time
	set_driving(1);
	// set an Event to make the tank 'stop driving'
	evl->add_event(new_tank_event(this, Tank_event::DRIVE), drive_time);
    }

    TRACE("Stank::move_dir : drive time %d, user time %d",
	    drive_time, spent);
    
    return spent;
}

// overload fire_at. tank doesn't really have fire_at, but since it
// can't distinguish between several players in a single room anyway,
// we may as well compute the coordinates and shoot at the room. saves
// the client a lot of headaches.
int Stank::fire_at(Electron *bokje, Electron *pl)
{
    Atom *target;

    target = bokje->get_atom();
    ASSERT(target);

    return fire_to(target, pl);
}

// fire a grenade at the specified atom
int Stank::fire_to(Atom *a, Electron *pl)
{
    double dist;
    int flt;
    Grenade *gr;
    Atom *my_atom;

    TRACE("Stank::fire_to");
    
    if (int_vars[AMMO] <= 0)
    {
	error_message = "The tank is out of ammo.";
	return 0;
    }
    
    // check reloading
    if (get_reloading())
    {
	error_message = "Tank is still reloading.";
	return 0;
    }

    // check target range
    my_atom = get_atom();
    dist = distance(a, my_atom);
    if (dist > fire_range)
    {
	error_message = "Target out of range.";
	return 0;
    }

    flt = (int) max(flight_time * (dist / fire_range), 1);
    
    // create grenade
    int_vars[AMMO]--;
    gr = new_actor<Sgrenade>(Electron::fid());
    gr->set_d(get_d());    // copy damage of the tank
    storage->lowlevel_add(gr);
    lattice->add(gr, my_atom->x, my_atom->y);

    // move grenade
    // use distance to compute flight time
    TRACE("creating grenade for client at [%d,%d]", gr->gx(), gr->gy());
    create_client_electron(gr, &global_send_queue);

    TRACE("sending grenade move from [%d,%d] to [%d,%d]",
					 my_atom->x,
					 my_atom->y,
					 a->x,
					 a->y);
	    
    // move the grenade
    lattice->move_electron(gr, a->top_level_object);    
    send_queue_packet(new_packet_command(gr,
					 COMMAND_FROM_TO,
					 my_atom->x,
					 my_atom->y,
					 a->x,
					 a->y,
					 flt));

    // set explosion Event
    evl->add_event(new_explode_event(gr, pl), flt);
    
    // set reload Event
    set_reloading(1);
    evl->add_event(new_tank_event(this, Tank_event::RELOAD), reload_time);
    
    return 0;
}

int Stank::turn(int dir)
{
    int ret;
    int spent = 0;
    
    if (get_driving())
    {
	error_message = "Tank is already moving";
	return 0;
    }

    // invalid directions are taken % 4 to fix any problems
    // this happens in Vehicle::turn
    ret = Vehicle::turn(dir);

    if (ret > 0)
    {
	// we turned, add some spent time (less than turn time!)
	spent = 20;

	// only send the command to the client if we really turned
	send_queue_packet(new_packet_command(this,
					     COMMAND_TURN,
					     dir, 0, 0, 0,
					     ret));
	
	// only set the 'driving' flag if it's taking some time
	set_driving(1);
	// set an Event to make the tank 'stop driving'
	evl->add_event(new_tank_event(this, Tank_event::DRIVE), ret);
    }

    return spent;
}

void Stank::door(int d) const
{
    error_message = "The hatch is closed";
    return;
}

void Stank::set_int_var(int var_id, int val)
{
    Tank::set_int_var(var_id, val);

    write_vars(packet_to_queue, &global_send_queue);
}
