/*
 *
 *   ^   |    sssss p   ddddd  fff  ggggg hhhh   iii  j   j    |   ^
 *  /|\  |    s     p   d     f   f   g   h   h i   i jj  j    |  /|\
 *   |   |    sss   p   ddd   f       g   hhhh  i   i j j j    |   |
 *   |  \|/   s     p   d     f   f   g   h   h i   i j  jj   \|/  |
 *   |   v    sssss ppp ddddd  fff    g   h   h  iii  j   j    v   |
 *
 *                           copyright 1999
 *                  Martijn Versteegh & Hein Zelle
 *
 */

/*
 *          rvfunc.cc
 *
 *          class implementation of Rvfunc
 *
 *          these are the 'server functions' of Repair_vehicle
 *          because this is a baseclass within the server, it is
 *          allowed to call server-specific functions here.
 */


#include "rvfunc.h"
#include "common/actors/pad.h"
#include "server/usercomm.h"
#include "server/send_q.h"
#include "server/globals.h"

Rvfunc::Rvfunc()
{
    re = 0;
    route = 0;
    rp = 0;
    pulse_time = 100;
}

void Rvfunc::set()
{
    Repair_event *temp_ev;

    // create a repair event for this repair vehicle
    temp_ev = new_repair_event(this);
    set_event(temp_ev);
    evl->add_event(temp_ev, pulse_time);
}

void Rvfunc::clear()
{
    clear_event();
}

void Rvfunc::set_event(Repair_event *_re)
{
    if (re)
    {
	re->stop();
    }
	 
    re = _re;
}

// unset the repair event
void Rvfunc::clear_event()
{
    if (re)
    {
	re->stop();
	re = 0;
    }
}

// take a step towards our target
// -1 : something went wrong
// 0  : continue with next step
// 1  : arrival at destination (route list empty)
int Rvfunc::move()
{
    Atom *a;
    int ret;
    
    ASSERT(route);
    
    a = (Atom *)route->pop();

    if (a)
    {
	ret = do_move_electron(this, a, pulse_time);
	if (!ret)
	{
	    // road is blocked
	    delete route;
	    route = 0;
	    return -1;
	}
	else
	{
	    return 0;
	}
    }
    else
    {
	// out of atoms in our route list.
	delete route;
	route = 0;

//	TRACE("%s : arrival at destination", type_name);
	return 1;
    }
}

// we're going to move. collect all pads.
void Rvfunc::fold()
{
    Pad *p;

    pads.reset();

    while ((p = dynamic_cast<Pad *>(pads.get())))
    {
	// check if we already have the pad in inventory
	if (inv_contains_recursive(p))
	{
	    pads.next();
	    continue;
	}

	// if we're not in the same room, tell the client
	// the pad is moving across the lattice
	if (get_atom() != p->get_atom())
	{
	    send_queue_packet(new_packet_command(p,
						 COMMAND_FROM_TO,
						 p->gx(), p->gy(),
						 gx(), gy(), 0));
	}
	
	// move the pad to our inventory from whereever it is
	lattice->move_electron(p, this);

	// tell the client about the pickup
	send_queue_packet(new_packet_command(this,
					     COMMAND_PICKUP,
					     p, 0));
	
	pads.next();
    }
}

// we've stopped: lay down our pads
void Rvfunc::unfold()
{
    Pad *p;
    int counter = 0;
    int pos = 1;            // start at top
    Atom *where;

    pads.reset();
    
    // count to 9
    // skip rooms that are not appropriate
    // stop if we run out of pads

    while ((p = dynamic_cast<Pad *>(pads.get())))
    {
	// do 8 pads at most
	if (++counter == 9)
	    break;

	// try to put the pad in the first available room
	// check if room is available
	if ((where = check_adjacent_atom(this, p, pos % 9)))
	{
	    // found a valid atom
	    // drop the pad there
	    
	    do_move_electron(p, where, 0);
	    pads.next();
	}

	pos += 2;     // start at top, skip one every time
	              // this will first make a cross around the vehicle,
                      // then it will fill up the corners

	if (pos == 13)  // 13 % 9 = 4 = center
	    pos += 2;   // skip the center position
    }
}


// event fired
void Rvfunc::pulse()
{
    ASSERT(state >= NOTHING
	   &&
	   state < STATE_LAST);

    switch (state)
    {
	case NOTHING:
	    
	    // find a target atom
	    find_target();

	    // stand until the route is computed
	    state = STANDING;

	    break;
	    
	case MOVING:

	    // move returns -1 (error: stop), 0 (continue) or 1 (arrival)
	    if (move())
	    {
		// we have arrived at the target,
		// or we cannot reach it
		if (target == get_atom())
		{
		    // we arrived at our target: unfold
		    unfold();
		    timer = stand_time;

//		    TRACE("%s : stopping for %d seconds",
//			  type_name, stand_time * pulse_time / 100);
		}
		
		// in either case: stop and start computing a new route
		state = STANDING;
		find_target();
	    }

	    break;
	    
	case STANDING:

	    timer--;
	    
	    // if we don't have a route yet, compute 100 steps
	    if (!route)
	    {
		for (int i = 0; i < 100; i++)
		{
		    if (rp->do_search(this))
		    {
			route = rp->last_route();
			delete_object(rp);
			rp = 0;

			if (!route)
			{
			    // search was unsuccesful: try again
//			    TRACE("%s : route not found : trying again.",
//				  type_name);
			    find_target();
			}
                        else
			{
//			    TRACE("%s : route found to <%d, %d>",
//				  type_name,
//				  target->x, target->y);
			}
			
			break;
		    }
		}
	    }
	    else
	    {
		if (timer < 0)
		{
		    fold();
//		    TRACE("%s : moving to <%d, %d>",
//			  type_name,
//			  target->x, target->y);
		    state = MOVING;
		}
	    }

	    break;
	    
	default:
	    fatal("Rvfunc::pulse : invalid state %d", state);
	    break;
    }

    // add a new event
    re = new_repair_event(this);
    evl->add_event(re, pulse_time);
}

static int route_cost(Electron *what, Atom *from, Atom *to)
{
    // first check if it's a door: special case
    // doors are expensive

    if (to->top_level_object->actor_type == ACTOR_DOOR)
    {
	if (!to->top_level_object->can_be_entered(what))	
	    return 200;       // closed / locked door
	else
	    return 50;        // open door
    }
    
    
    if (!to->top_level_object->can_be_entered(what))
    {

        // a door might be closed,
	// so try to find a way around it if possible

	
	return 0xFFF;
    }
    
    return 1;
}

// pick a new target room to go to
void Rvfunc::find_target()
{
    Atom *a = 0;

    // find a random room.
    a = lattice->get_random_atom();

    target = a;
//    TRACE("%s : computing route to <%d, %d>", type_name, a->x, a->y);

    rp = new_routeplanner();
    rp->init_search(get_atom(), target, route_cost);
}

