/*
 *
 *   ^   |    sssss p   ddddd  fff  ggggg hhhh   iii  j   j    |   ^
 *  /|\  |    s     p   d     f   f   g   h   h i   i jj  j    |  /|\
 *   |   |    sss   p   ddd   f       g   hhhh  i   i j j j    |   |
 *   |  \|/   s     p   d     f   f   g   h   h i   i j  jj   \|/  |
 *   |   v    sssss ppp ddddd  fff    g   h   h  iii  j   j    v   |
 *
 *                           copyright 1999
 *                  Martijn Versteegh & Hein Zelle
 *
 */

/*
 *       shover.cc
 *
 *       server version of the Hover
 */

#include "shover.h"
#include "server/globals.h"
#include "server/usercomm.h"
#include "server/fire.h"
#include "server/message.h"
#include "server/events/hover_ev.h"

void Shover::set(int _actor_id)
{
    Hover::set(_actor_id);
    drive_time = 20;
    recharge_time = 100;          // time between battery charges
    reload_time = 100;            // time between shots
    fire_energy = 100;            // 10 shots -> empty
    charge_energy = 25;           // 40 recharges -> full
    target_range = 10.0;
    facing = 2;
    
    // not reloading, no reload event
    reloading = 0;
    he_reload = 0;

    // give the hover it's own recharge event
    // the event will spawn new events until the hover is removed
    // from the game, see Shover::clear
    he_recharge = new_hover_event(this, Hover_event::RECHARGE);
    evl->add_event(he_recharge, recharge_time);
}

void Shover::clear()
{
    Hover::clear();
    reloading = 0;

    // stop the current hover Event
    if (he_recharge)
    {
	he_recharge->stop();
	he_recharge = 0;
    }
    if (he_reload)
    {
	he_reload->stop();
	he_reload = 0;
    }
}

int Shover::get_reloading()
{
    return reloading;
}

void Shover::set_reloading(int yesno)
{
    reloading = yesno;

    // if we are set to non-reloading, clear the reload event 
    if (!yesno)
	he_reload = 0;
}

void Shover::die(Electron *killer)
{
    die_electron(this, killer);
}

int Shover::move_dir(int dir, Electron *pl)
{
    int
	tx,
	ty;

    int
	ret,
        spent = 0;

    get_pos(&tx, &ty);
    
    // find target coordinates
    switch (dir)
    {
	case DIR_N:
	    ty--;
	    break;
	case DIR_E:
	    tx++;
	    break;
	case DIR_S:
	    ty++;
	    break;
	case DIR_W:
	    tx--;
	    break;
	default:
	    warning("Shover::move_dir: invalid direction %d", dir);
	    return 0;
	    break;
    }

    // try and move the hover
    warning("Shover::move_dir: try moving 1 room in direction %d", dir);
    ret = do_move_electron(this, tx, ty, drive_time);

    if (ret)
    {
	// we moved, add some spent time (less than drive time!)
	spent += drive_time;
    }

    return spent;
}

// fire a laser beam at the specified target
int Shover::fire_at(Electron *target, Electron *pl)
{
    Atom *source;
    Atom *dest;
    Slaserbeam *sl;
    int beam_time;
    
    double dist;

    TRACE("Shover::fire_at");
    
    if (get_reloading())
    {
	error_message = "Laser is still reloading.";
	return 0;
    }

    if (int_vars[BATTERY] < fire_energy)
    {
	error_message = "Not enough energy to fire the laser.";
	return 0;
    } 
    
    // determine validity of target
    source = get_atom();
    dest = target->get_atom();

    dist = distance(source, dest);
    if (dist > target_range)
    {
	error_message = "Target out of range.";
	return 0;
    }

    // create a laser beam from source to target
    sl = new_actor<Slaserbeam>(Electron::fid());
    sl->set_d(get_d());          // copy damage from hover
    sl->set_source(pl);          // set source electron
    sl->set_target(target);      // set target electron
    storage->lowlevel_add(sl);
    get_atom()->add(sl);         // add laser beam to atom

    // send the laserbeam to the client
    warning("creating laserbeam for client at <%d, %d>",
	    gx(), gy());
    create_client_electron(sl, &global_send_queue);
    
    // add an explode Event
    beam_time = get_d() / 5;
    TRACE("beam time: %d", beam_time);
    evl->add_event(new_explode_event(sl, pl), beam_time);

    // empty the battery
    set_int_var(BATTERY, max(int_vars[BATTERY] - fire_energy, 0));
    
    set_reloading(1);
    he_reload = new_hover_event(this, Hover_event::RELOAD);
    evl->add_event(he_reload, reload_time);
    
    return 0;
}

void Shover::recharge()
{
    Hover_event *nhe;

    set_int_var(BATTERY, min(int_vars[BATTERY] + charge_energy, 1000));

    // send battery state to the clients
    write_vars(packet_to_queue, &global_send_queue);

//    warning("hover battery recharging: %d", int_vars[BATTERY]);

    // add a new hover event
    nhe = new_hover_event(this, Hover_event::RECHARGE);
    evl->add_event(nhe, recharge_time);
    he_recharge = nhe;
}

int Shover::lay_mine(Electron *pl)
{
    Smine *sm;
    User *u = ((Splayer *)pl)->get_user();
    
    ASSERT(int_vars[MINES] >= 0);
    
    if (!int_vars[MINES])
    {
	message(u, "The mine bay is empty.");
	return 0;
    }

    // use a mine
    int_vars[MINES]--;
    
    warning("hover laying a mine at [%d, %d]", gx(), gy());

    // create a mine
    sm = new_actor<Smine>(Electron::fid());
    // set damage if desired
    // sm->set_d(200);
    storage->lowlevel_add(sm);
    get_atom()->add(sm);

    // send the mine to OWNER client only
    // make the mine invisible
    // make sure explode and remove events are only sent to those
    // clients that can see the mine

    create_client_electron(sm, u->get_send_queue());
    message(u, "Mine dropped and armed.");
    
    return 1;
}

void Shover::set_int_var(int var_id, int val)
{
    Hover::set_int_var(var_id, val);

    write_vars(packet_to_queue, &global_send_queue);
}
