#!/usr/bin/perl -w

use diagnostics;

$usage = "<name>\n" .
         "where <name> is the uncapitalized name of the new actor. \n";

sub main
{
    &usage unless (@ARGV == 1);

    # global variables!
    $bare = shift(@ARGV);
    $bare = lc($bare);
    $one_cap = ucfirst($bare);
    $all_cap = uc($bare);
    $type_name = $bare;
    $type_name =~ s/_/ /g;
    $common_file = "$bare";
    $server_file = "s$bare";
    $client_file = "c$bare";
    $prefix = "/home/files/electron";
    $common_prefix = "$prefix/common";
    $server_prefix = "$prefix/server";
    $client_prefix = "$prefix/client";
    
    print "Creating new actor '$bare'\n";
#    print "bare:     $bare\n" .
#	  "one_cap:  $one_cap\n" .
#	  "all_cap:  $all_cap\n" .
#          "typename: $type_name\n" .
#	  "common_file: $common_file\n" .
#	  "server_file: $server_file\n" .
#          "client_file: $client_file\n";

    &check_exists();
    $baseclass = &get_base_class();
    print "base class for $one_cap: " . ucfirst($baseclass) . "\n";

    $hmd = &get_hmd();
    
    $copyright = &read_file("$common_prefix/header.h");
#    print "copyright string read:\n";
#    print $copyright;
    
    &create_headers();
    &create_sources();
    &fix_common();
    &fix_server();
    &fix_client();
    &fix_makefiles();
}

sub get_base_class
{
    my($answer);
    
    print "Enter a base class (Electron): ";
    $answer = <STDIN>;
    chomp($answer);

    if ($answer eq "")
    {
	$answer = "electron";
    }
    else
    {
	$answer = lc($answer);
    }

    $answer;
}

sub get_hmd
{
    my($answer);

    print "Does $one_cap have Hit, Multiplier and Damage (y/n)? ";
    $answer = <STDIN>;
    chomp($answer);

    if ($answer eq "y")
    {
	return 1;
    }

    0;
}
    
# pass a filename as arg0
sub open_file_write
{
    my($file, $handleref);

    $file = $_[0];
    if (!open(FILE, ">$file"))
    {
	warn("could not open $file");
	return 0;
    }

    $handleref = \*FILE;
    $handleref;
}

sub check_exists
{
    my ($answer);
    
    if (-e "$common_prefix/actors/$common_file.h")
    {
	print("$one_cap already exists. Continue (y/n)? ");
	$answer = <STDIN>;
	chomp($answer);
	
	exit (1) unless ($answer eq "y");
    }
}

sub read_file
{
    my(@str, $file, $result);

    $file = $_[0];

    if (!open(HANDLE, "<$file"))
    {
	&error("Could not open $file");
    }

    @str = <HANDLE>;
    $result = join("", @str);

    close (HANDLE);
    
    $result;
}

sub create_headers
{
    &create_common_header();
    &create_server_header();
    &create_client_header();
}

sub create_sources
{
    &create_common_source();
    &create_server_source();
    &create_client_source();
}

# arguments: path (including slash), base filename, .extension
# create a file skeleton, containing the copyright header
sub create_file_skeleton
{
    my ($file, $cstr, $fn);
    
    $fn = $_[1] . $_[2];
    $file = open_file_write($_[0] . $fn);
    
    if (!$file)
    {
	return $file;
    }
    
    $cstr = $copyright;
    $cstr =~ s/__REPLACE__/$fn/g;
    
    print $file $cstr;
    print $file "\n\n";
    $file;
}

# arg0: filename
sub header_define
{
    my($fn, $str);

    $fn = $_[0];
    $fn =~ s/\./_/g;
    $fn = uc($fn);
    $fn = "_$fn" . "_";

    $str = "#ifndef $fn\n#define $fn\n\n";
    $str;
}

sub create_client_header
{
    my ($file);
    
    print "creating client header file\n";
    $file = &create_file_skeleton("$client_prefix/actors/", $client_file, ".h");
    return unless ($file);
    
    print $file &header_define($client_file . ".h") . "\n";
    print $file "#include \"common/actors/$common_file.h\"\n\n";
    print $file "class C$bare : public $one_cap\n";
    print $file "{\n";
    print $file "public:\n";
#    print $file "    ALLOW_FREELIST;\n";     # not anymore
    print $file "};\n";
    print $file "\n";
    print $file "#endif\n";

    close($file);
}

sub create_client_source
{
    my ($file);
    
    print "creating client source file\n";
    $file = &create_file_skeleton("$client_prefix/actors/", $client_file, ".cc");
    return unless ($file);

    print $file "#include \"$client_file" . ".h\"\n\n";
    
    close($file);
}

sub create_server_header
{
    my ($file);
    
    print "creating server header file\n";
    $file = &create_file_skeleton("$server_prefix/actors/", $server_file, ".h");
    return unless ($file);
    
    print $file &header_define($server_file . ".h") . "\n";
    print $file "#include \"common/actors/$common_file.h\"\n\n";
    print $file "class S$bare : public $one_cap\n";
    print $file "{\n";
    print $file "public:\n";
#    print $file "    ALLOW_FREELIST;\n";     # not anymore
    print $file "};\n";
    print $file "\n";
    print $file "#endif\n";

    close ($file);
}

sub create_server_source
{
    my ($file);
    
    print "creating server source file\n";
    $file = &create_file_skeleton("$server_prefix/actors/", $server_file, ".cc");
    return unless ($file);

    print $file "#include \"$server_file" . ".h\"\n\n";
    
    close ($file);
}

sub create_common_header
{
    my ($file, $str, $bla, $basec);

    print "creating common header file\n";
    $file = &create_file_skeleton("$common_prefix/actors/", $common_file, ".h");
    return unless ($file);

    $str = &header_define($common_file . ".h");

    print $file $str . "\n";

    if ($baseclass eq "electron")
    {
	print $file "#include \"common/electron.h\"\n\n";
    }
    else
    {
	print $file "#include \"common/actors/$baseclass.h\"\n\n";
    }
    
    print $file "class $one_cap : public ". ucfirst($baseclass) . "\n";
    print $file "{\n";
    print $file "public:\n";
    print $file "    enum _int_vars\n" .
	         "    {\n";
    if ($hmd)
    {
	print  $file "        HMD_ENUM,\n" .
	             "        INTVARS_LAST\n" .
		     "    };\n" .
		     "\n";
    }
    else
    {
	print  $file "        INTVARS_LAST = 0\n" .
		     "    };\n" .
		     "\n";
    }
    print  $file "    enum _string_vars\n" .
	         "    {\n" .
		 "        STRINGVARS_LAST = 0\n" .
		 "    };\n" .
		 "\n" .
		 "    enum _vector_vars\n" .
		 "    {\n" .
		 "        VECTORVARS_LAST = 0\n" .
		 "    };\n" .
		 "\n";

    print  $file "    $one_cap();\n" .
		 "\n" .
		 "    // public constructor / destructor\n" .
		 "    void set(int actor_id = 0);\n" .
		 "    void clear();\n" .
		 "\n";
		 
    if ($hmd)
    {
	print  $file "    HMD_PROTO;\n" .
	             "    WRITE_HMD_PROTO;\n";
    }

    print  $file "    VARS_PROTO;\n";
    print  $file "\n" .
		 "    int int_vars[INTVARS_LAST];\n" .
		 "    Vector vector_vars[VECTORVARS_LAST];\n" .
		 "\n" .
		 "protected:\n" .
                 "    char *string_vars[STRINGVARS_LAST];\n";
    
    print  $file "};\n";
    print  $file "\n";
    print  $file "#endif\n";

    close ($file);
}

sub create_common_source
{
    my ($file);
    
    print "creating common source file\n";
    $file = &create_file_skeleton("$common_prefix/actors/", $common_file, ".cc");
    return unless ($file);

    print $file "#include \"" . $common_file . ".h\"\n" .
	        "\n" .
		"$one_cap" . "::$one_cap()\n" .
		"{\n" .
		"    set(-1);\n" .
		"}\n" .
		"\n" .
		"void $one_cap" . "::set(int _actor_id)\n" .
		"{\n";
    print $file "    " . ucfirst($baseclass) .
	        "::set(ACTOR_$all_cap, _actor_id);\n" .
                "    type_name = \"$type_name\";\n" .
                "    freelist_index = COMMON_FREELIST_$all_cap;\n";

    if ($hmd)
    {
	print $file "    set_hmd(1000, 1000, 100);\n";
    }

    print $file "}\n" .
	        "\n" .
		"void $one_cap" . "::clear()\n" .
		"{\n" .
		"    " . ucfirst($baseclass) . "::clear();\n" .
		"}\n" .
		"\n";

    print $file "// macro functions for write_vars and write_hmd\n" .
                "VARS_FUNCTION($one_cap)\n";

    if ($hmd)
    {
	print $file "WRITE_HMD_FUNCTION($one_cap)\n";
    }
    
    close ($file);
}

sub fix_makefiles
{
    my($repstring, $temp_file, $contents);

    print "adding $one_cap to the makefiles\n";
    
#   add an entry for the actor to actors.inc
    $tmp_file = &open_file_write("/tmp/new_actor.tmp");
    return unless ($tmp_file);

    $contents = &read_file("$prefix/actors.inc");
    $repstring = " \\\n" .
	         "             $bare\n" .
                 "# END_OF_ACTORS";
    $contents =~ s~\n# END_OF_ACTORS~$repstring~;

    print $tmp_file $contents;
    close ($tmp_file);
    rename("/tmp/new_actor.tmp", "$prefix/actors.inc");
}

sub fix_common
{
    my($contents, $tmp_file, $includestring, $tablestring, $casestring);

    print "adding $one_cap to the common library\n";

#   add the actor type to act_type.h
    $tmp_file = &open_file_write("/tmp/new_actor.tmp");
    return unless ($tmp_file);

    $contents = &read_file("$common_prefix/act_type.h");
    $contents =~ s/,\n\s+ACTOR_LAST/,\n    ACTOR_$all_cap,\n    ACTOR_LAST/;

    print $tmp_file $contents;
    close ($tmp_file);
    rename("/tmp/new_actor.tmp", "$common_prefix/act_type.h");

#   add the include file to act_type.cc
    $tmp_file = &open_file_write("/tmp/new_actor.tmp");
    return unless ($tmp_file);

    $contents = &read_file("$common_prefix/act_type.cc");
    $includestring = "#include \"actors/$common_file" . ".h\"\n" .
	             "// END_OF_INCLUDES";
    $contents =~ s~// END_OF_INCLUDES~$includestring~;

    print $tmp_file $contents;
    close ($tmp_file);
    rename("/tmp/new_actor.tmp", "$common_prefix/act_type.cc");

#   add an entry for the new actor in translate_actor_types
    $tmp_file = &open_file_write("/tmp/new_actor.tmp");
    return unless ($tmp_file);

    $contents = &read_file("$common_prefix/act_type.cc");
    if ($hmd)
    {
	$tablestring = "    {ACTOR_$all_cap,    LIST_????,    " .
	               "{\"sss\", \"\", \"\"}},\n" .
		       "};\n" .
                       "// END_OF_TABLE";
    }
    else
    {
	$tablestring = "    {ACTOR_$all_cap,    LIST_????,    " .
	               "{\"\", \"\", \"\"}},\n" .
		       "};\n" .
                       "// END_OF_TABLE";
    }
    
    $contents =~ s~};\n// END_OF_TABLE~$tablestring~;
    print $tmp_file $contents;
    close ($tmp_file);

    rename("/tmp/new_actor.tmp", "$common_prefix/act_type.cc");

#   add a case label to init_global_actor_statics
    $tmp_file = &open_file_write("/tmp/new_actor.tmp");
    return unless ($tmp_file);

    $contents = &read_file("$common_prefix/act_type.cc");
    $casestring = "            case ACTOR_$all_cap" . ":\n" .
                  "                e = new $one_cap;\n" .
                  "                break;\n" .
                  "// END_OF_CASE";
    $contents =~ s~// END_OF_CASE~$casestring~;

    print $tmp_file $contents;
    close ($tmp_file);
    rename("/tmp/new_actor.tmp", "$common_prefix/act_type.cc");

#   add the actor type to freelist.h
    $tmp_file = &open_file_write("/tmp/new_actor.tmp");
    return unless ($tmp_file);

    $contents = &read_file("$common_prefix/freelist.h");
    $contents =~ s/,\n\s+COMMON_FREELIST_LAST/,\n    COMMON_FREELIST_$all_cap,\n    COMMON_FREELIST_LAST/;

    print $tmp_file $contents;
    close ($tmp_file);
    rename("/tmp/new_actor.tmp", "$common_prefix/freelist.h");
}

sub fix_server
{
    my ($tmp_file, $contents, $includestring, $casestring);

    print "adding S$bare to the server\n";

#   add the include file to server/actors/sactors.h
    $tmp_file = &open_file_write("/tmp/new_actor.tmp");
    return unless ($tmp_file);

    $contents = &read_file("$server_prefix/actors/sactors.h");
    $includestring = "#include \"$server_file" . ".h\"\n" .
	             "// END_OF_INCLUDES";
    $contents =~ s~// END_OF_INCLUDES~$includestring~;

    print $tmp_file $contents;
    close ($tmp_file);
    rename("/tmp/new_actor.tmp", "$server_prefix/actors/sactors.h");

#   add a case label to get_electron_from_type
    $tmp_file = &open_file_write("/tmp/new_actor.tmp");
    return unless ($tmp_file);

    $contents = &read_file("$server_prefix/actors/snew_act.cc");
    $casestring = "        case ACTOR_$all_cap" . ":\n" .
                  "            e = new_actor<S$bare>(id);\n" .
                  "            break;\n" .
                  "// END_OF_CASE";
    $contents =~ s~// END_OF_CASE~$casestring~;

    print $tmp_file $contents;
    close ($tmp_file);
    rename("/tmp/new_actor.tmp", "$server_prefix/actors/snew_act.cc");
}

sub fix_client
{ 
    my ($tmp_file, $contents, $includestring, $casestring);

    print "adding C$bare to the server\n";

#   add the include file to client/actors/sactors.h
    $tmp_file = &open_file_write("/tmp/new_actor.tmp");
    return unless ($tmp_file);

    $contents = &read_file("$client_prefix/actors/cactors.h");
    $includestring = "#include \"$client_file" . ".h\"\n" .
	             "// END_OF_INCLUDES";
    $contents =~ s~// END_OF_INCLUDES~$includestring~;

    print $tmp_file $contents;
    close ($tmp_file);
    rename("/tmp/new_actor.tmp", "$client_prefix/actors/cactors.h");

#   add a case label to get_electron_from_type
    $tmp_file = &open_file_write("/tmp/new_actor.tmp");
    return unless ($tmp_file);

    $contents = &read_file("$client_prefix/actors/cnew_act.cc");
    $casestring = "        case ACTOR_$all_cap" . ":\n" .
                  "            e = new_actor<C$bare>(id);\n" .
                  "            break;\n" .
                  "// END_OF_CASE";
    $contents =~ s~// END_OF_CASE~$casestring~;

    print $tmp_file $contents;
    close ($tmp_file);
    rename("/tmp/new_actor.tmp", "$client_prefix/actors/cnew_act.cc");
}

sub usage
{
    print STDERR "usage: $0 $usage\n";
    exit(1);
}

sub warn
{
    print STDERR "$0: $_[0]\n";
}

sub error
{
    &warn($_[0]);
    exit(1);
}

&main;
