/*
 *
 *   ^   |    sssss p   ddddd  fff  ggggg hhhh   iii  j   j    |   ^
 *  /|\  |    s     p   d     f   f   g   h   h i   i jj  j    |  /|\
 *   |   |    sss   p   ddd   f       g   hhhh  i   i j j j    |   |
 *   |  \|/   s     p   d     f   f   g   h   h i   i j  jj   \|/  |
 *   |   v    sssss ppp ddddd  fff    g   h   h  iii  j   j    v   |
 *
 *                           copyright 1999
 *                  Martijn Versteegh & Hein Zelle
 *
 */

/*
 *   plug.cc
 *
 *   class implementation of the plug.
 */

#include <sys/socket.h>
#include <sys/time.h>
#include <netdb.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdlib.h>
#include <errno.h>
#include "plug.h"
#include "common/utils.h"


#define PLUG_MAX_READ_RETRY 100

Plug::Plug(char const *n_adress, int n_port)
{
    file_des = -1; // set it to uninitialised

    hostname = sstrdup(n_adress);
    port = n_port;
    listening = opened = 0;

}

Plug::Plug(int _file_des, char const *_hostname, int _port, sockaddr_in _address)
{
    file_des = _file_des;
    hostname = sstrdup(_hostname);
    port = _port;
    address = _address; //!
    opened = 1;
    listening = 0;
}


Plug::~Plug()
{
    if (file_des >= 0)
    {
	close();
    }

    if (hostname)
        free(hostname);
}


int Plug::init_sockaddr(const char *hostname, int port)
{
       hostent *hostinfo;
     
       address.sin_family = AF_INET;
       address.sin_port = htons ((short)port);

       if (sizeof(short) != 2)
          warning("Plug::init_sockaddr( ): sizeof(short) != 2, could cause problems");

       
       hostinfo = gethostbyname (hostname);
       if (hostinfo == NULL)
       {
           warning("Plug:init_sockaddr(  ): host %s.\n", hostname);
           return -1;
       }
       address.sin_addr = *(struct in_addr *) hostinfo->h_addr;

       return 0;
}

int Plug::open()
{
    int result = 0;

    if (listening)
        fatal("Plug::open(): tried to open a plug used for listening");
    
    if (init_sockaddr(hostname, port))
       return -1;
    
    file_des = socket(PF_INET, SOCK_STREAM, 0);

    if (file_des <0)
    {
        warning("Plug::open(): error creating socket");
        return -1;
    }

    warning("Plug::open() using fd %d for new connection", file_des);

//    fcntl(file_des, F_SETFL, O_NONBLOCK);       // set it to nonblocking mode
    // reuse adress. needed for listen socket to split off a new socket
    // using accept() on the same port
    setsockopt(file_des, SOL_SOCKET, SO_REUSEADDR, NULL, 0);

    //! add SO_LINGER here if necessary
    //! yes probably necessary: server sends HANGUP and is not expecting
    //! an answer, you want to make sure (reasonably) that the hangup
    //! actually gets sent. set the timeout value for the linger struct
    //! to something like 0.1 second
    
    if (connect(file_des, (sockaddr *)&address , sizeof(address)))
    {
        if (errno == EINPROGRESS)
        {
        /*
                  //for now we will accept it
                  // is not yet ready, but will probably be ready later
                  // so we don't wait here, but just start writing
                  // honping that's ok ;-)
            FD_ZERO(&read_set);
            FD_SET(file_des, &read_set);
            tv.tv_sec = 1;
            tv.tv_usec = 0;
            select(file_des + 1, &read_set, NULL, NULL, &tv);
          */
            warning("Plug::open() : connection not established immediately...");

            int tmp = can_write(900000);

            warning("Plug::open() : can_write returned %d", tmp);
            if (tmp <= 0)
            {
                warning("Plug::open() :could not connect");
                close();
                return -1;
            }
        }
        else
        {

            warning("Plug::open() : could not connect to server");
            // cleanup
            close (); //! may not be right on this half opened socket
            
            return -1;
        }
    }

    warning("Plug::open() : created connection on socket %d", file_des);

    opened = -1;  // mark plug as in use for read/write
    total_read_bytes = total_write_bytes = 0;
    
    return result;
}

int Plug::close()
{
    if (!opened && !listening)
    {
       warning("Plaug::close() : attempted to close plug when it was not open (non-serious error)");
    }
    opened = listening = 0;
    ::close(file_des);
    file_des = -1;
    return 1;
}


int Plug::get_port() const
{
    return port;
}

char const *Plug::get_adress() const
{
    return hostname;
}

int Plug::read(char *buffer, int bytes)
{
    int retval;
    int total_bytes = bytes;

    ASSERT(bytes > 0);
    
    if (!opened && !listening)
    {
        warning("Plug::read() : attempting to read from a closed connection");
        return 0;
    }
    
    retval = can_read(0);

    if (!retval)
        return 0;

    if (retval < 0)
    {
        warning("Plug::read() : error %d in select()", retval);
	ASSERT(0);
        return retval;
    }

    // can_read says we can read
    
    if (listening)
        fatal("Plug::read() : tried to read from a plug used for listening");

    while (bytes > 0)
    {
        while ((retval = ::read(file_des, buffer, bytes)) < 0)
        {
	    if (errno == EAGAIN) 
            {
                // EAGAIN is the same as EWOULDBLOCK,           
                // both mean there is no data ready at the moment,
                // but we can retry safely
                continue;
            }
            else
            {
                // real error
                warning("Plug::read() : error reading from socket %d (error code %d)",
			file_des, errno);
//              warning("closing connection");
//              close();

		return retval;
            }
        }

	ASSERT(retval <= bytes);
	
        // we read something (0 or more bytes, no error)
        if (retval != bytes)
        {
            // if it was the first try, there is really nothing to read
            if (!retval && bytes == total_bytes)
                return 0;
            else
            {
		int retval2;

                // didn't read enough: retry
                warning("Plug::read : read only %d/%d bytes, "
                        "retrying the rest",
                        retval, bytes);

		// wait 100 milliseconds before retrying, otherwise there's no use.
		retval2 = can_read(100000);

		if (retval2 < 0)
		{
		    warning("Plug::read() : error %d in select()", retval);
		    ASSERT(0);
		    return retval;
		}

		if (!retval2)
		{
		    // nothing to read yet
		    // don't retry

		    warning("Plug::read : still nothing to read after 0.1 seconds, aborting.");
		    return -1;
		}
		
	    }
        }

        buffer += retval;
        bytes -= retval;
    }

    total_read_bytes += total_bytes;

    return total_bytes;
}

int Plug::write(char const *buffer, int bytes)
{
    int ret = 0;
    int total_bytes = bytes;
    int retval;



    if (listening)
        fatal("Plug::write() : tried to write to a plug used for listening");

    if (!opened)
    {
        warning("Plug::write() : attempting to write to a closed connection");
        return 0;
    }
    
    retval = can_write(200000);

    if (retval <= 0)
    {
        warning("Plug::write() : error (timeout) !!!");

        // return -1 here
        // otherwise the rest of the program (notably the server)
        // doesn't realize the connection should be closed.
        // this happens if we return 0 instead of -1
        // this statement used to be
        // return retval;
        
        return -1;
    }

    while (bytes) // we may need to write in multiple passes when ever compiled on a system with small output buffers
    {
        while((ret = ::write(file_des, buffer, bytes)) < 0)
        {
            if (errno != EAGAIN)
            {
                warning("Plug::write() : could not write to file descriptor %d (error %d)", file_des, errno);
//                warning("closing connection");
//                close();
                return -1;
            }
            warning("Plug::write() : write on filedescriptor %d got EAGAIN error, retrying", file_des);
        }
        if (ret != bytes)
        {
            // not enough? try again with the rest
//           warning("on filedescriptor %d: could write only %d of %d bytes, retrying the rest",
//                   file_des, ret, bytes);
        }
        buffer += ret;
        bytes -= ret;
    }
    total_write_bytes += total_bytes;
    return total_bytes;
}

int Plug::start_listening()
{
    int ret;
    int i;
    
    if (opened)
       fatal("Plug: trying to listen on a plug set up for reading/writing");

    file_des = socket(AF_INET, SOCK_STREAM, 0);

    if (file_des < 0)
    {
        warning("Plug::start_listening():error creating socket");
        return -1;
    }

    fcntl(file_des, F_SETFL, O_NONBLOCK); // set non-blocking mode, i.e. do not wait if there's no data waiting on reads etc.
    setsockopt(file_des, SOL_SOCKET, SO_REUSEADDR, &i, sizeof(i));  // make it re-use the same port

//    sl.l_onoff = 1;
//    sl.l_linger = 2;     // 2 second timeout on close
//    setsockopt(file_des, SOL_SOCKET, SO_LINGER, &sl, sizeof(sl));
    
    if (init_sockaddr(hostname, port))
    {
        warning("Plug::start_listening():could not find adress");
	::close(file_des);
	file_des = -1;
        return -1;
    }

    // socket is set up for connecting
    ret = bind(file_des, (sockaddr *)&address, sizeof(sockaddr_in));
    
    if (ret)
    {
        warning("Plug::start_listening():error binding the socket to it's address");
	::close(file_des);
	file_des = -1;
	return -1;
    }

    // back log of 5 connections trying to get in at the same time
    ret = listen(file_des, 5);
    
    if (ret)
    {
        warning("Plug::start_listening():error making the socket listen");
	::close(file_des);
	file_des = -1;
        return -1;
    }

    listening = -1;

    return 0;
}


Plug *Plug::check_new_connection()
{
    int retval;
    int new_file_des;
    sockaddr_in new_address;
    int size = sizeof(struct sockaddr_in);

    if (opened)
       fatal("Plug::check_new_connection() : trying to check new connection on a plug set up for reading/writing");

    if (!listening)
       fatal("Plug::check_new_connection() : trying to check new connection on a plug not set up for listening");
       
    retval = can_read(0);

    if (!retval)  // if select returns 0 there is no new connection found.
    {
        return NULL;
    }

    if (retval < 0) // an error has occurred during select
    {
        fatal("Plug::check_new_connection(): error %d!!!", errno);
    }

    // retval > 0 so there is a connection, we need to accept it.

    warning("Plug::check_new_connection(): new connection!");
            
    
    new_file_des = accept(file_des, (struct sockaddr *)&new_address,
                           (socklen_t *)&size);

    if (new_file_des < 0)
    {
        warning("Plug::check_new_connection(): error accepting a new connection");
        return NULL;
    }

    //! fill in hostname and port number from the new_address
    return new Plug(new_file_des, "undefined", -1, new_address);
}

int Plug::can_write(int usec) const
{
    struct timeval tv; // timeout value
    fd_set what_set;   // the set of filedescripors to check for read/write
    int retval;
    
    tv.tv_sec = 0;     // fill the timeout struct
    tv.tv_usec = usec; // usec has to be < 1000000

    // fill the set
    FD_ZERO(&what_set);  
    FD_SET(file_des, &what_set);

    while ((retval = select(file_des + 1, NULL, &what_set, NULL, &tv)) < 0)
    {
        if (errno == EINTR) // EINTR is not a real error
        {
            tv.tv_sec = 0;
            tv.tv_usec = usec;
            FD_ZERO(&what_set);
            FD_SET(file_des, &what_set);
            continue;
        }
        
        // now we do have a real error
        return -1;
    }

//    warning("can_write: file_des = %d, select returned %d", file_des, retval);

    return FD_ISSET(file_des, &what_set);;
}

int Plug::can_read(int usec) const
{
    struct timeval tv; // timeout value
    fd_set what_set;   // the set of filedescripors to check for read/write
    int retval;
    
    tv.tv_sec = 0;     // fill the timeout struct
    tv.tv_usec = usec; // usec has to be < 1000000

    // fill the set
    FD_ZERO(&what_set);  
    FD_SET(file_des, &what_set);

    while ((retval = select(file_des + 1, &what_set, NULL, NULL, &tv)) < 0)
    {
        if (errno == EINTR) // EINTR is not a real error
        {
            tv.tv_sec = 0;
            tv.tv_usec = usec;
            FD_ZERO(&what_set);
            FD_SET(file_des, &what_set);
            
            continue;
        }
        
        // now we do have a real error
        return retval;
    }

    return retval;
}

int Plug::is_open()
{

   if (opened || listening)
      return -1;

   return 0;
}
