#include "srchrout.h"
#include "common.h"
#include "lattice.h"
#include "utils.h"

#define LOWER2BITS (1 | (1<<1))
#define THIRDBIT (1<<2)

static const int xdirs[] = {0,1,0,-1};
static const int ydirs[] = {-1,0,1,0};

Routeplanner *new_routeplanner()
{
   Routeplanner *n = (Routeplanner *)new_object(COMMON_FREELIST_ROUTEPLANNER);//! freelist

   if (!n)
      n = new Routeplanner;

   n->set();

   return n;
}


Routeplanner::Routeplanner()
{
   work_array = NULL;
   w = h = -1;
   found = busy = 0;

   to_do = new List(__FILE__);

   freelist_index = COMMON_FREELIST_ROUTEPLANNER;
}

Routeplanner::~Routeplanner()
{
   if (work_array)
      del_work_array();

   delete to_do;
}

void Routeplanner::check_work_array()
{
   if (!lattice)
       fatal("cannot use routeplanners without lattice");

   if (w != lattice->get_w() || h != lattice->get_h())
   {
      if (work_array)
          del_work_array();
          
      new_work_array();
   }
}

void Routeplanner::new_work_array()
{
   ASSERT(!work_array);

   w = lattice->get_w();
   h = lattice->get_h();

   work_array = new unsigned short *[w];
   for (int i=0;i<w;i++)
       work_array[i] = new unsigned short[h];
   
}

void Routeplanner::del_work_array()
{
   ASSERT(work_array);

   for (int i=0;i<w;i++)
       delete [] work_array[i];

   delete [] work_array;

   w = h = -1;
   work_array = NULL;
}

void Routeplanner::clear_work_array()
{

   check_work_array();
   for (int i=0;i<w;i++)
   {
      for (int j=0;j<h;j++)
      {
         work_array[i][j] = 0xFFF0; //! use memset?
      }
   }
}

void Routeplanner::init_search(Atom *from, Atom *to, int (*cost)(Electron *what, Atom *a, Atom *b))
{
   if (busy)
      warning("Routeplanner got new search while still searching");


   calc_cost = cost;
   busy = -1;
   found = 0;
   longesttried = 0;
   iterations = 0;
   
   clear_work_array();

   to_do->clear();

   to_do->push(from);

   work_array[from->x][from->y] = 0;

   target = to;
   startpoint = from;
   
}

int Routeplanner::do_search(Electron *what)
{
   Atom *cur = (Atom *)(to_do->pop());
   Atom *next;
   Atom *a;

   iterations++;
   
   int xc,yc,x,y, cost;
   unsigned short curdist, totdist;

   
   if (!cur)
   {
//     printf(" DO_S: EMPTY\n");
      busy = 0;
      if (!found)
         return -2;
      else
         return -1;
   }



//   printf(" DO_S: pop %d,%d\n",cur->x,cur->y);

   xc = cur->x;
   yc = cur->y;

   curdist = work_array[xc][yc] >> 4;

   // mark this position as not being in the list
   work_array[xc][yc] &= ~THIRDBIT;

   if (curdist > longesttried)
       longesttried = curdist;

   if (cur == target)
   {
      busy = 0;
      found = -1;
//      printf(" DO_S: FOUND\n");
      to_do->clear();
      return -1;
   }

   for (int i=0;i<4;i++)
   {
      x = xc + xdirs[i];
      y = yc + ydirs[i];

      next = lattice->get(x, y);
//      printf(" lat %d,%d:%p \n",x,y,next);

      if (next)
      {
         if (calc_cost)
             cost = calc_cost(what, cur, next);
         else
             cost = 1;
             
         totdist = (cost + curdist);
//         printf("  check %d,%d , totdist %d, wa %d\n",next->x,next->y, totdist,(work_array[x][y] >> 4));

         if (totdist < (work_array[x][y] >> 4))
         {

//            printf("     push %d,%d \n",next->x,next->y);
//!            if (work_array[xc][yc] & THIRDBIT)
//!                to_do->pop(next); // prevent it from being twice in the list
            if (work_array[x][y] & THIRDBIT)
                to_do->pop(next); // prevent it from being twice in the list

            work_array[x][y] = i + (totdist << 4);
//!            work_array[xc][yc] |= THIRDBIT;
            work_array[x][y] |= THIRDBIT;

            // now for the inserting we bias the distances in
            // the globally right direction by adding the
            // estimated distance to the endpoint
            // before comparing
            // must make sure the estimate is never longer than the
            // actual distance
            totdist += dist_estimate(next, target);

            // wind the list until it points at the first element
            // with a greater cost than the current cost
            to_do->reset();
            while((a = (Atom *)(to_do->get())))
            {
               if ((work_array[a->x][a->y] >> 4) + dist_estimate(a, target) > totdist)
                   break;
            
               to_do->next();
            }

            // insert the location in the list
            to_do->insert(next);

         }
         
      }
      
   }


   return 0;
}

List *Routeplanner::last_route()
{
   List *ret = new List;
   Atom *a = target;
   int fromdir;

   int cx,cy,x,y;
   

   if (!found)
       return NULL;

   while(a != startpoint)
   {
      ret->push(a);
      cx = a->x;
      cy = a->y;
      fromdir = work_array[cx][cy] & LOWER2BITS;

      ASSERT(fromdir <4);
      ASSERT(fromdir >=0);
      
      x = cx - xdirs[fromdir];
      y = cy - ydirs[fromdir];
      a = lattice->get(x,y);

      ASSERT(a);
   }


   ret->push(startpoint);


   return ret;
}


// returns an optimistic estimate of the cost to
// travel from 'from' to 'to'.
// this estimate may never be longer than the actual
// path or the found route is not guaranteed to be the shortest
// though larger etimates push the search more in the 'right' direction
int Routeplanner::dist_estimate(Atom *from, Atom *to)
{
   ASSERT(from);
   ASSERT(to);

   // manhattan distance
   // geeft rechthoekige paden
   return (abs(from->x - to->x) + abs(from->y - to->y));
   // pythagoras dist
   // geeft zichzag paden door voorkeur voor schuin oversteken
//    return (int)(1.4*sqrt((from->x - to->x)*(from->x - to->x) + (from->y - to->y)*(from->y - to->y)));
}

void Routeplanner::set()
{
}

void Routeplanner::clear()
{
}


#ifdef DEBUG_ROUTEPLANNER

#define SCALE 5
void Routeplanner::draw_route(BITMAP *onto)
{
   unsigned int d;
   List *l;
   Atom *a;
   acquire_bitmap(onto);
   for (int i=0;i<w;i++)
   {
      for (int j=0;j<h;j++)
      {
         d = work_array[i][j] >> 4;

         if (d > longesttried + 1)
         {
            if (lattice->get(i,j))
            {
            rectfill(onto, i * SCALE, j * SCALE,
                           i*SCALE+SCALE, j*SCALE+SCALE,
                           makecol(255,0,0));
            }
            else
            {
            rectfill(onto, i * SCALE, j * SCALE,
                           i*SCALE+SCALE, j*SCALE+SCALE,
                           makecol(255,255,255));
            }
         }
         else
         {
            rectfill(onto, i * SCALE, j * SCALE,
                           i*SCALE+SCALE, j*SCALE+SCALE,
                           makecol(0,d * 255 / (longesttried + 1),0));
         }
      }
   }

   if (found)
   {
      l = last_route();
      if (l)
      {
         l->reset();
         while((a = (Atom *)l->get()))
         {
           circlefill(onto, SCALE * a->x+5, SCALE * a->y+5,3, makecol(0,0,255));
           l->next();
         }
         delete l;
      }
      release_bitmap(onto);
      textprintf(onto, font,0,0,makecol(0,255,255),"iterations %d", iterations);
      
      save_bitmap("findrout.pcx",onto,desktop_palette);
      readkey();
      acquire_bitmap(onto);
   }
   circlefill(onto, SCALE * startpoint->x + 1, SCALE * startpoint->y + 1,2, makecol(255,0,255));
   circlefill(onto, SCALE * target->x + 2, SCALE * target->y + 2,2, makecol(255,255,0));



   release_bitmap(onto);

}
#endif

