/*
 *
 *   ^   |    sssss p   ddddd  fff  ggggg hhhh   iii  j   j    |   ^
 *  /|\  |    s     p   d     f   f   g   h   h i   i jj  j    |  /|\
 *   |   |    sss   p   ddd   f       g   hhhh  i   i j j j    |   |
 *   |  \|/   s     p   d     f   f   g   h   h i   i j  jj   \|/  |
 *   |   v    sssss ppp ddddd  fff    g   h   h  iii  j   j    v   |
 *
 *                           copyright 1999
 *                  Martijn Versteegh & Hein Zelle
 *
 */

/*
 * packet.h
 *
 * add all classes derived from 'packet' in this file.
 * at the bottom of the file is a single function which reads
 * any kind of packet from a Plug.
 *
 * unknown packets can be identified with the get_type function.
 *
 * structure of a packet:
 * __________________________________________________________
 * |1 byte packet type | 2 byte actor_type| 4 byte actor_id|
 *                    ______________________________________
 *  Packet_command     | 1 byte command | 4 * 2 bytes data |
 *                    ________________________________
 *  Packet_info_num    | 1 byte var_id | 2 byte data |
 *                    ________________________________________________________________
 *  Packet_info_string | 1 byte var_id | 2 byte string length | variable length data |
 *                    _____________________________________
 *  Packet_info_vect   | 1 byte var_id | 2 * 2 bytes data |
 */


#ifndef _PACKET_H_
#define _PACKET_H_

#include "object.h"
#include "net/plug.h"
#include "const.h"
#include "freelist.h"
#include "electron.h"
#include "utils.h"
#include <malloc.h>
#include <stdio.h>

// The first three constants (info packets) are used as
// array indices for the permissions array. These three have to
// be at the top of the List in the order num = 0, string = 1, vect = 2

enum packet_types
{
    TYPE_INFO_NUM = 0,
    TYPE_INFO_STRING,
    TYPE_INFO_VECT,
    TYPE_COMMAND
};

class Packet
    : public Object
{
public:
    
    Packet();
    virtual int read_from(Plug *connection) = 0;
    virtual int write_to(Plug *connection) = 0;
    virtual int write_to_bogus(Plug *connection, int abort) = 0;
    int get_type() const;

    
    int actor_type;
    int actor_id;

    // to report invalid packets from their constructors etc.
    int err;
    static char error[1024];

    
protected:
    int type;
    int read_actor(char const *buffer);
    int write_actor(char *buffer);
};

// read all possible commands and directions for packet_command
#include "commands.h"

#define PACKET_COMMAND_SIZE (2 + 4 + 1 + 5 * 2) // actor_type, actor_id, command + 4* 2 data
class Packet_command
: public Packet
{
public:

    Packet_command();
    Packet_command(int actor_type, int actor_id, int command, int arg1, int arg2, int arg3, int arg4, int arg5);
    Packet_command(Plug *connection);
    ~Packet_command();

    void set(int actor_type, int actor_id, int command, int arg1, int arg2, int arg3, int arg4, int arg5);
    void set(Plug *connection);
    void set(Packet_command const *other);
    void clear();
    
    int read_from(Plug *connection);
    int write_to(Plug *connection);
    int write_to_bogus(Plug *connection, int abort= -1);
    static int write_to(Plug *connection, int actor_type, int actor_id, int command, int arg1 = 0, int arg2 = 0, int arg3 = 0, int arg4 = 0, int arg5 = 0);
    static int write_to_ssl(Plug *connection, int actor_type, int actor_id, int command, int arg1 = 0, int arg2 = 0, int longarg2 = 0, int time = 0);
    static int write_to_ll(Plug *connection, int actor_type, int actor_id, int command, int longarg1 = 0, int longarg2 = 0, int time = 0);
    /* this function checks if p!= NULL, and if it's the right type, and then checks against all arguments, which can be
     *'NA' if you don't care (NULL for string arguments) */
    static int expect(Packet  const *p, int actor_type = NA, int actor_id = NA, int command = NA, int arg1 = NA, int arg2 = NA, int arg3 = NA, int arg4 = NA, int arg5 = NA);


    void set_larg1(unsigned int arg);// uses arg1, arg2 to store 1 4 byte unsigned int
    void set_larg2(unsigned int arg);// uses arg3, arg4 to store 1 4 byte unsigned int
    unsigned int larg1() const; // uses arg1, arg2 to store 1 4 byte unsigned int
    unsigned int larg2() const; // uses arg3, arg4 to store 1 4 byte unsigned int

    int command; // the type of commando
    int arg1;    // playernr for MOVE_TO_PLAYER else x coordinate
    int arg2;    // y coordinate
    int arg3;    // x coordinate 2
    int arg4;    // y coordinate 2
    int arg5;    // times
    static unsigned int shorts2long(int s1, int s2);
    static void long2shorts(int l1,int *s1, int *s2);
private:
    
    static char buffer[PACKET_COMMAND_SIZE + 1];
};


#define PACKET_INFO_NUM_SIZE  (2 + 4 + 1 + 4) // actor_type, actor_id, var_id, var_value

class Packet_info_num
: public Packet
{
public:
    Packet_info_num();
    Packet_info_num(int actor_type, int actor_id, int var_id, int var_value);
    Packet_info_num(Plug *connection);
    ~Packet_info_num();

    void set(int actor_type, int actor_id, int var_id, int var_value);
    void set(Plug *connection);
    void set(Packet_info_num const *other);
    void clear();
    
    int read_from(Plug *connection);
    int write_to(Plug *connection);
    int write_to_bogus(Plug *connection, int abort= -1);
    static int write_to(Plug *connection, int actor_type, int actor_id, int var_id, int var_value);
    /* this function checks if p!= NULL, and if it's the right type, and then checks against all arguments, which can be
     *'NA' if you don't care (NULL for string arguments) */
    static int expect(Packet const *p, int actor_type = NA, int actor_id = NA, int var_id = NA, int var_value = NA);

    int var_id;       // which variable is this about
    int var_value;    // what value is it

private:

    static char buffer[PACKET_INFO_NUM_SIZE + 1];

};


#define PACKET_INFO_VECT_SIZE  (2 + 4 + 1 + 2 + 2) // actor_type, actor_id, var_id, var_x, var_y
class Packet_info_vect
: public Packet
{
public:
    Packet_info_vect();
    Packet_info_vect(int actor_type, int actor_id, int var_id, int x, int y);
    Packet_info_vect(Plug *connection);
    ~Packet_info_vect();

    void set(int actor_type, int actor_id, int var_id, int x, int y) ;
    void set(Plug *connection);
    void set(Packet_info_vect const *other);
    void clear();
    
    int read_from(Plug *connection);
    int write_to(Plug *connection);
    int write_to_bogus(Plug *connection, int abort= -1);
    static int write_to(Plug *connection, int actor_type, int actor_id, int var_id, int x, int y);
    /* this function checks if p!= NULL, and if it's the right type, and then checks against all arguments, which can be
     *'NA' if you don't care (NULL for string arguments) */
    static int expect(Packet const *p, int actor_type = NA, int actor_id = NA, int var_id = NA, int x = NA, int y = NA);

    int var_id;    // whcih vector is this about
    int var_x;
    int var_y;

private:

    static char buffer[PACKET_INFO_VECT_SIZE + 1];

};



#define PACKET_INFO_STRING_SIZE (2 + 4 + 1 + 2) //actor_type, actor_id, var_id, string_length

class Packet_info_string
: public Packet
{
public:
    Packet_info_string();
    Packet_info_string(int actor_type, int actor_id, int var_id, char const *value);
    Packet_info_string(Plug *connection);
    ~Packet_info_string();

    void set(int actor_type, int actor_id, int var_id, char const *value);
    void set(Plug *connection);
    void set(Packet_info_string const *other);
    void clear();
    
    int read_from(Plug *connection);
    int write_to_bogus(Plug *connection, int abort= -1);
    int write_to(Plug *connection);
    static int write_to(Plug *connection, int actor_type, int actor_id, int var_id, char const *value);
    /* this function checks if p!= NULL, and if it's the right type, and then checks against all arguments, which can be
     *'NA' if you don't care (NULL for string arguments) */
    static int expect(Packet const *p, int actor_type = NA, int actor_id = NA, int var_id = NA, char const *var_value = NULL);

    char const *get_string() const;
    void set_string(char const *string);
    void clear_string();    // deletes the string and sets it to NULL

    int var_id;

    
private:
    
    int string_length;
    int max_string_length;
    char *string_value;
    static char buffer[PACKET_INFO_STRING_SIZE + 1];

};

/* read_packet reads a packet from plug, determining the type from
 * the type identifyer in the bytestream, and delegating to the
 * different type classes
 * a new packet will be
 */
Packet *read_packet(Plug *connection);

/* this one also reads a pcket, but it returns it in a temporary
 * static packet which is re-used ech time
 * so it is faster than the above, and no memory fragmentation
 * but the result is only valid till the next call
 * it retuirn NULL if no data is waiting
 */
Packet const *read_packet_static(Plug *connection);

void print_packet(Packet const *p);


/* some int/string conversion utilities
 * LSB/MSB independent
 */
long int int4_from_string(char const *string);
short int int2_from_string(char const *string);
void int4_to_string(long int val, char *dest);
void int2_to_string(int val, char *dest);

/* freelist functions */
Packet_info_num *new_packet_info_num(int _actor_type, int _actor_id, int _var_id, int _var_value);
Packet_info_string *new_packet_info_string(int _actor_type, int _actor_id, int _var_id, char const *_value);
Packet_info_vect *new_packet_info_vect(int _actor_type, int _actor_id, int _var_id, int _x, int _y);
Packet_command *new_packet_command(int _actor_type, int _actor_id, int _command, int _arg1, int _arg2, int _arg3, int _arg4, int _arg5);

/* copy constructor freelist functions */
Packet_info_num *new_packet_info_num(Packet_info_num const *other);
Packet_info_string *new_packet_info_string(Packet_info_string const *other);
Packet_info_vect *new_packet_info_vect(Packet_info_vect const *other);
Packet_command *new_packet_command(Packet_command const *other);
Packet *new_packet(Packet const *other);

/* the same functions with a Electron * as first argument */
Packet_info_num *new_packet_info_num(Electron *actor, int _var_id, int _var_value);
Packet_info_string *new_packet_info_string(Electron *actor, int _var_id, char const *_value);
Packet_info_vect *new_packet_info_vect(Electron *actor, int _var_id, int _x, int _y);


extern Packet_info_num *new_packet_info_num();
extern Packet_info_num *new_packet_info_num(int actor_type, int actor_id, int var_id, int var_value);
extern Packet_info_num *new_packet_info_num(Plug *connection);

extern Packet_info_string *new_packet_info_string();
extern Packet_info_string *new_packet_info_string(int actor_type, int actor_id, int var_id, char const *value);
extern Packet_info_string *new_packet_info_string(Plug *connection);

extern Packet_info_vect *new_packet_info_vect();
extern Packet_info_vect *new_packet_info_vect(int actor_type, int actor_id, int var_id, int x, int y);
extern Packet_info_vect *new_packet_info_vect(Plug *connection);
extern Packet_info_vect *new_packet_info_vect(int actor_type, int actor_id, int var_id, Vector v);
extern Packet_info_vect *new_packet_info_vect(Electron *actor, int var_id, Vector v);

extern Packet_command *new_packet_command();
extern Packet_command *new_packet_command(int actor_type, int actor_id, int command, int arg1, int arg2, int arg3, int arg4, int arg5);
extern Packet_command *new_packet_command(int actor_type, int actor_id, int command, int arg1, int arg2, int longarg2, int arg5);
extern Packet_command *new_packet_command(Plug *connection);
extern Packet_command *new_packet_command(Electron *actor, int _command, int _arg1, int _arg2, int _arg3, int _arg4, int _arg5);
extern Packet_command *new_packet_command(Electron *actor, int _command, int _arg1, int arg2,int _larg2, int _arg5);
extern Packet_command *new_packet_command(Electron *actor, int _command, Electron *actor2, int _arg5);

// callback for Electron::write_vars versions
// (see the define in electron.h)
// because we need a flexible write_vars function in
// all Electron-derived objects which can output
// the packets it generates either to a queue , or write them
// directly to a plug, or wahtever, we need a callback
extern void packet_to_plug(Packet *p, Object *o);

// set freelist indices of global packets to -1
void init_global_packets();
void shutdown_global_packets();

#endif
