/*
 *
 *   ^   |    sssss p   ddddd  fff  ggggg hhhh   iii  j   j    |   ^
 *  /|\  |    s     p   d     f   f   g   h   h i   i jj  j    |  /|\
 *   |   |    sss   p   ddd   f       g   hhhh  i   i j j j    |   |
 *   |  \|/   s     p   d     f   f   g   h   h i   i j  jj   \|/  |
 *   |   v    sssss ppp ddddd  fff    g   h   h  iii  j   j    v   |
 *
 *                           copyright 1999
 *                  Martijn Versteegh & Hein Zelle
 *
 */

/*
 *  packet.cc: class implementation of the base Packet and some helper functions
 */

#include "packet.h"
#include "utils.h"

Packet::Packet()
{
    err = 0;
}

int Packet::get_type() const
{
    return type;
}


int Packet::read_actor(char const *buffer)
{
    actor_type = int2_from_string(buffer);
    actor_id = int4_from_string(buffer + 2);

    return 0;
}


int Packet::write_actor(char *buffer)
{
    int2_to_string(actor_type, buffer);
    int4_to_string(actor_id, buffer + 2);

    return 0;
}


char Packet::error[1024];

Packet_command *global_packet_command = 0;
Packet_info_string *global_packet_info_string = 0;
Packet_info_num *global_packet_info_num = 0;
Packet_info_vect *global_packet_info_vect = 0;

Packet_command *error_packet()
{
    Packet_command *pc;

    pc = new_packet_command();
    pc->err = -1;
    
    return pc;
}

Packet_command *error_packet_static()
{
    Packet_command *pc;

    pc = global_packet_command;
    pc->err = -1;
    
    return pc;
}

Packet *read_packet(Plug *connection)
{
    char type;
    Packet *ret;
    int retval;

    retval = connection->read(&type, 1);

    if (retval < 0)
    {
	// error in read: this means the connection is broken.
	// return a dummy packet with the error flag set
        ssprintf(Packet::error, 1024,
           "read_packet : Read error in Plug. Error code: %d\n"
	   "              Look backwards in the log for a plug read error.\n",
	   retval);

	return error_packet();
    }
    
    if (retval != 1)
    {
	return NULL;
    }

    switch(type)
    {
        case TYPE_COMMAND:
            ret = new_packet_command(connection);
            break;
        case TYPE_INFO_STRING:
            ret = new_packet_info_string(connection);
            break;
        case TYPE_INFO_NUM:
            ret = new_packet_info_num(connection);
            break;
        case TYPE_INFO_VECT:
            ret = new_packet_info_vect(connection);
            break;
        default:
            warning("read_packet: Unknown packet received, discarding.");
	    warning("yes, really!");
	    ssprintf(Packet::error, 1024,
		     "read_packet : received a packet with unknown type %d",
		     type);
	    ret = error_packet();
            break;
    }

    return ret;
}

void init_global_packets()
{
    // set freelist indices to -1
    // these never end up in a freelist, and therefore will
    // cause a failed ASSERT if they are destroyed at the end of
    // the program while their freelist index is set to something valid

    global_packet_command = new_packet_command();
    global_packet_info_num = new_packet_info_num();
    global_packet_info_vect = new_packet_info_vect();
    global_packet_info_string = new_packet_info_string();
}

void shutdown_global_packets()
{
    if (global_packet_command)
    {
	delete_object(global_packet_command);
	global_packet_command = 0;
    }

    if (global_packet_info_num)
    {
	delete_object(global_packet_info_num);
	global_packet_info_num = 0;
    }

    if (global_packet_info_vect)
    {
	delete_object(global_packet_info_vect);
	global_packet_info_vect = 0;
    }

    if (global_packet_info_string)
    {
	delete_object(global_packet_info_string);
	global_packet_info_string = 0;
    }
}

Packet const *read_packet_static(Plug *connection)
{
    char type;
    Packet *ret;
    int r;

    r = connection->read(&type, 1);

    if (r == 0)
        return NULL;

    if (r < 0)
    {
	// error in read: this means the connection is broken.
	// return a dummy packet with the error flag set
        ssprintf(Packet::error, 1024,
           "read_packet : Read error in Plug. Error code: %d\n"
	   "              Look backwards in the log for a plug read error.\n",
	   r);

	return error_packet_static();
    }

    switch(type)
    {
        case TYPE_COMMAND:
            ret = global_packet_command;
	    ret->err = 0;
            break;
        case TYPE_INFO_STRING:
            ret = global_packet_info_string;
	    ret->err = 0;
            break;
        case TYPE_INFO_NUM:
            ret = global_packet_info_num;
	    ret->err = 0;
            break;
        case TYPE_INFO_VECT:
            ret = global_packet_info_vect;
	    ret->err = 0;
            break;
        default:
            warning("read_packet_static: Unknown packet received, discarding.");
	    ssprintf(Packet::error, 1024,
		 "read_packet_static : received a packet with unknown type %d",
		 type);
	    ret = error_packet_static();
	    return ret;
            break;
    }

    ret->read_from(connection);
    return ret;
}




void print_packet(Packet const *p)
{
   if (!p)
   {
       warning("NULL packet printed");
       return;
   }
       
   warning("");
   warning("");
   warning("Packet:");
   switch(p->get_type())
   {
       case TYPE_COMMAND:
	   warning
	       (
                "type: command.  command: %d   args: %d %d %d %d",
                ((Packet_command *)p)->command,
                ((Packet_command *)p)->arg1,
                ((Packet_command *)p)->arg2,
                ((Packet_command *)p)->arg3,
                ((Packet_command *)p)->arg4
	       );
	   break;
       case TYPE_INFO_STRING:
	   warning
	       (
                "type: info_string.  var id: %d   string: %s",
                ((Packet_info_string *)p)->var_id,
                ((Packet_info_string *)p)->get_string()
	       );
	   break;
       case TYPE_INFO_NUM:
	   warning
	       (
                "type: info_num.  var id: %d   num: %d",
                ((Packet_info_num *)p)->var_id,
                ((Packet_info_num *)p)->var_value
	       );
	   break;
       case TYPE_INFO_VECT:
	   warning
	       (
		"type: info_vect. var id: %d   vect: (%d, %d)",
                ((Packet_info_vect *)p)->var_id,
                ((Packet_info_vect *)p)->var_x,
                ((Packet_info_vect *)p)->var_y
	       );
	   break;
       default:
	   warning("print_packet : type %d UNKNOWN !!!", p->get_type());
	   return;
	   break;
   }
   
   warning("actor_type: %d  actor_id: %d\n\n",p->actor_type, p->actor_id);
}



long int int4_from_string(char const *string)
{
    unsigned char *tmp = (unsigned char *)string;
    return ((long int)(tmp[0]) << 24) +
          ((long int)(tmp[1]) << 16) +
          ((long int)(tmp[2]) << 8) +
          ((long int)(tmp[3]));

}


short int int2_from_string(char const *string)
{
    unsigned char *tmp = (unsigned char *)string;
    return ((short int)(tmp[0]) << 8) +
          ((short int)(tmp[1]));
}


void int4_to_string(long int val, char *dest)
{
    unsigned char *tmp = (unsigned char *)dest;


    tmp[0] = (int)((val & 0xFF000000L) >> 24);
    tmp[1] = (int)((val & 0x00FF0000L) >> 16);
    tmp[2] = (int)((val & 0x0000FF00L) >> 8);
    tmp[3] = (int)((val & 0x000000FFL));
}

void int2_to_string(int val, char *dest)
{
    short tmp = (short )val;
    unsigned char *tmps = (unsigned char *)dest;

    tmps[0] = (int)((tmp & 0xFF00) >> 8);
    tmps[1] = (int)((tmp & 0x00FF));
}




// callback for Electron::write_vars versions
// (see the define in electron.h)
// because we need a flexible write_vars function in
// all Electron-derived objects which can output
// the packets it generates either to a queue , or write them
// directly to a plug, or wahtever, we need a callback
extern void packet_to_plug(Packet *p, Object *o)
{
  //  now if this is called, we assume o is a Plug *.
  // if it isn't this will priobably give a HARD_CRASH!!!!
    Plug *plug = (Plug *)o;
 
    p->write_to(plug);

    // now throw the packet into our freelist
    delete_object(p);
    
}

Packet *new_packet(Packet const *other)
{
    switch (other->get_type())
    {
	case TYPE_INFO_NUM:
	    return new_packet_info_num((Packet_info_num *)other);
	    break;
	case TYPE_INFO_STRING:
	    return new_packet_info_string((Packet_info_string *)other);
	    break;
	case TYPE_INFO_VECT:
	    return new_packet_info_vect((Packet_info_vect *)other);
	    break;
	case TYPE_COMMAND:
	    return new_packet_command((Packet_command *)other);
	    break;
	default:
	    fatal("new_packet : invalid type %d", other->get_type());
	    break;
    }

    return 0;
}
