/*
 *
 *   ^   |    sssss p   ddddd  fff  ggggg hhhh   iii  j   j    |   ^
 *  /|\  |    s     p   d     f   f   g   h   h i   i jj  j    |  /|\
 *   |   |    sss   p   ddd   f       g   hhhh  i   i j j j    |   |
 *   |  \|/   s     p   d     f   f   g   h   h i   i j  jj   \|/  |
 *   |   v    sssss ppp ddddd  fff    g   h   h  iii  j   j    v   |
 *
 *                           copyright 1999
 *                  Martijn Versteegh & Hein Zelle
 *
 */
#include <typeinfo>
#include <stdlib.h>
#include "list.h"
#include "utils.h"
#include "freelist.h"

// implementation of the List_element and List classes

List_element::List_element()
{
    next = 0;
    prev = 0;
    data = 0;

}

List_element::~List_element()
{
    next = 0;
    prev = 0;
    data = 0;
}

// only assign the data pointer. the next pointer should never
// be duplicated to prevent horrible pointer spaghetti.
void List_element::set_data(Object *other)
{
    data = other;
}


// implementation of the actual List class
List_element *List::recycled = 0;

List::List()
{
    head = 0;
    tail = 0;
    current = 0;
    freelist_index = -1; // lists cannot be put inside a freelist at the moment (no valid index defined)
    #ifdef LISTDEBUG
     double_push_allowed = 0;
    #endif

    list_name = 0;
}

List::List(char const *_list_name)
{
    head = 0;
    tail = 0;
    current = 0;
    freelist_index = -1; // lists cannot be put inside a freelist at the moment (no valid index defined)
    #ifdef LISTDEBUG
     double_push_allowed = 0;
    #endif

    list_name = _list_name;
}


// remove all remaining List_element's, but do not delete the data pointers.
List::~List()
{
    clear();
}

// careful: this one does NOT delete the objects
void List::clear()
{
    List_element
        *tmp = head;
    
    while (tmp)
    {
        #ifdef DEBUGMODE
        dec_refcount(tmp->data);
        #endif
    
        head = tmp->next;
        del_list_element(tmp);
        tmp = head;
    }

    current = tail = 0;
}

// careful: this one actually deletes the objects
void List::destroy()
{
    List_element
        *tmp = head;
    
    while (tmp)
    {
        #ifdef DEBUGMODE
        dec_refcount(tmp->data);
        #endif
    
        delete tmp->data;
        head = tmp->next;
        del_list_element(tmp);
        tmp = head;
    }

    current = tail = 0;
}

Object *List::get_head()
{
    if (head)
        return head->data;
    else
        return 0;
}

/*
Object *List::get_next()
{
    if (current)
    {
        List_element
            *tmp = current;

        current = current->next;
        return tmp->data;
    }
    else
        return 0;
}
*/

Object *List::get()
{
    if (current)
    {
        return current->data;
    }
    else
    {
        return 0;
    }
}

int List::next()
{
    if (current)
    {
        current = current->next;
        return 1;
    }
    else
    {
        return 0;
    }
}

int List::prev()
{
    if (current)
    {
        current = current->prev;
        return 1;
    }
    else
    {
        return 0;
    }
}

Object *List::pop()
{
    List_element
        *tmp_e = head;
    Object
        *tmp;

    if (!tmp_e)
        return 0;

    tmp = tmp_e->data;
    head = tmp_e->next;
    current = head;



    if (head)
        head->prev = 0;
    else
        tail = 0;
    
    del_list_element(tmp_e);
    #ifdef DEBUGMODE
    dec_refcount(tmp);
    #endif
    return tmp;
}

// remove the specified Object from the List
Object *List::pop(Object *remove)
{
    List_element
        *tmp = head;
    Object
        *d;
        
    if (!remove)
        return 0;
    
    while (tmp)
    {
        if (tmp->data == remove)
        {
            // found the element to remove

            // see if we have to change the next pointer of the
            // previous element
            if (tmp->prev)
            {
                tmp->prev->next = tmp->next;
            }
            else
            {
                head = tmp->next;
            }

            // see if we have to change the prev pointer of the next element
            if (tmp->next)
            {
                tmp->next->prev = tmp->prev;
            }
            else
            {
                tail = tmp->prev;
            }
            
            current = tmp->next;
            d = tmp->data;
            del_list_element(tmp);
            #ifdef DEBUGMODE
            dec_refcount(d);
            #endif
            return d;
        }

        tmp = tmp->next;
    }

    return 0;
}

void List::push(Object *other)
{
    List_element
        *tmp = new_list_element();

    ASSERT(other); // null pointers are not allowed in the List
    
    #ifdef LISTDEBUG
    if (!double_push_allowed)
    {
      List_element *i;
      i = head;
    
      while (i)
      {
         ASSERT( i->data != other);
         i = i->next;
      }
    }
    
    #endif
    #ifdef DEBUGMODE
    inc_refcount(other);
    #endif


    
    tmp->data = other;
    tmp->next = head;
    tmp->prev = 0;
    
    if (tmp->next)
    {
        tmp->next->prev = tmp;
    }
    else
    {
        tail = tmp;
    }
    
    head = tmp;
    current = head;
}

int List::size() const
{
    int
        s = 0;
    List_element
        *tmp = head;

    while (tmp)
    {
        s++;
        tmp = tmp->next;
    }

    return s;
}

void List::reset()
{
    current = head;
}


// insert the Object *before* the current pointer

void List::insert(Object *other)
{
    List_element
        *tmp = new_list_element();


    tmp->data = other;

    #ifdef DEBUGMODE
    inc_refcount(other);
    #endif


    if (!head)
    {
        tmp->next = tmp->prev = 0;
        head = tail = current = tmp;
        return;
    }
    
    if (current == head)
    {
        tmp->next = head;
        tmp->prev = 0;
        head = tmp;
        current = tmp;

        if (tmp->next)
        {
            tmp->next->prev = tmp;
        }
        
        return;
    }

    tmp->next = current;

    if (current)
    {
        tmp->prev = current->prev;
        tmp->prev->next = tmp;
        current->prev = tmp;
    }
    else
    {
        // List is not empty, and we're at the end
        tail->next = tmp;
        tmp->prev = tail;

        tail = tmp;
    }
        
    current = tmp;
}

/*

void List::insert_before(Object *other)
{
    List_element
        *tmp = new_list_element(),
        *find = head,
        *lfind = head;

    // insert the Object before the last Object returned by get_next()
    // that means *before* the element *before* the current pointer

    tmp->data = other;

    if (current == head || (head && current == head->next))
    {
        tmp->next = head;
        head = tmp;
        current = tmp;
        return;
    }

    // find the element before the element before current
    // if current = 0, that will be the element before the last element
    while (find && find->next != current)
    {
        lfind = find;
        find = find->next;
    }

    tmp->next = find;
    current = tmp;

    // it's possible that lfind is still equal to find
    if (lfind != find)
        lfind->next = tmp;
}

*/

Object *List::get_tail()
{
    if (tail)
    {
	return tail->data;
    }
    
    return 0;
}

Object *List::pop_tail()                  // get and remove the last element
{
    List_element
        *tmp_e = tail;
    Object
        *tmp;

    if (!tmp_e)
        return 0;

    tmp = tmp_e->data;
    tail = tmp_e->prev;

    if (tail)
        tail->next = 0;
    else
        head = 0;
    
    del_list_element(tmp_e);

    #ifdef DEBUGMODE
    dec_refcount(tmp);
    #endif
    
    return tmp;
}

void List::push_tail(Object *other)     // add a new element at the tail
{
    List_element
        *tmp = new_list_element();


    ASSERT(other); // null pointers are not allowed in the List

    #ifdef LISTDEBUG
    if (!double_push_allowed)
    {
      List_element *i;
      i = head;
    
      while (i)
      {
         ASSERT( i->data != other);
         i = i->next;
      }

    }
    #endif
    #ifdef DEBUGMODE
    inc_refcount(other);
    #endif

    
    tmp->data = other;
    
    if (!tail)
    {
        tail = head = tmp;
    }
    else
    {
        tmp->prev = tail;
        tail->next = tmp;
        tail = tmp;
    }

    current = tmp;
}

/*
Object *List::get_prev()
{
    if (current)
    {
        List_element
            *tmp = current;

        current = current->prev;
        return tmp->data;
    }
    else
        return 0;
}
*/

void List::reset_tail()
{
    current = tail;
}

// free List functions
List_element *List::new_list_element()
{
    List_element *ret = 0;
    
    if (recycled)
    {
        // pop the first element of recycled
        ret = recycled;
        recycled = recycled->next;

        ret->next = ret->prev = 0;
        ret->data = 0;
    }
    else
    {
        ret = new List_element;
    }

    return ret;
}

void List::del_list_element(List_element *le)
{
    // add the element to the recycled List (push)
    le->next = recycled;
    le->prev = 0;
    le->data = 0;

    recycled = le;
}

void List::cleanup_freelist()
{
    List_element *tmp;
    
    while (recycled)
    {
        tmp = recycled;
        recycled = recycled->next;

        delete tmp;
    }
}

Object *List::find(Object const *search)
{
    List_element *tmp = head;

    while (tmp)
    {
        if (tmp->data == search)
        {
            current = tmp;
            return tmp->data;
        }

        tmp = tmp->next;
    }

    return 0;
}

void List::ifnotfoundpush(Object *other)
{
    if (!other)
    {
	warning("List::ifnotfoundpush : NULL pointer being pushed!");
	abort();
    }

    
    if (!find(other))
    {
        push(other);
    }
}


void List::filter(int (*criterium)(Object *), List *to)
{

    List_element *i, *tmp;
    int ret;
    
    i = head;
    
    while (i)
    {
	ret = criterium(i->data);
	switch(ret)
	{
	    case FILTER_MOVE:
	    case FILTER_DELETE:
	    case FILTER_DELETE_OBJECT:

		// these have in common that they remove the list element
		// from the original list. another switch follows at the
		// bottom for the differences.
		
		tmp = i;
		if (tmp->prev)
		    tmp->prev->next = tmp->next;
		
		if (tmp->next)
		    tmp->next->prev = tmp->prev;
		
		i = i->next;
		
		if (tmp == head)
		    head = tmp->next;
		
		if (tmp == tail)
		    tail = tmp->prev;

		if (tmp == current)
		    current = head;

#ifdef DEBUGMODE
		dec_refcount(tmp->data);
#endif

		// switch to select action for MOVE, DELETE and DELETE_OBJECT
		switch(ret)
		{
		    case FILTER_MOVE:
			if (to)
			    to->push(tmp->data);
			break;
			
		    case FILTER_DELETE:
			if (tmp->data)
			{
			    delete tmp->data;
			    tmp->data = 0;
			}
			break;
			
		    case FILTER_DELETE_OBJECT:
			if (tmp->data)
			{
			    delete_object(tmp->data);
			    tmp->data = 0;
			}
			break;
			
		    default:
			ASSERT(0);
			break;
		}
		    
		// recycle the list element
                del_list_element(tmp);

		break;    // next element

	    case FILTER_COPY:
		if (to)
		    to->push(i->data);
		
		i = i->next;

		break;    // next element

	    case FILTER_LEAVE:
		i = i->next;

		break;    // next element
		
	    default:
		ASSERT(0);
	}
    }
}

// this pushes this List onto the tail
// of the target List. If you want to actually copy the List,
// call List::clearcopy
void List::copy(List *target)
{
    List_element *tmp = head;
 
    while (tmp)
    {
	target->push_tail(tmp->data);
	tmp = tmp->next;
    }
}

void List::clearcopy(List *target)
{
    target->clear();
    copy(target);
}

#ifdef DEBUGMODE
void List::inc_refcount(Object *o)
{
    #ifdef LISTINOUT
     warning("object of type %s added to list %s\n",typeid(*o).name(), list_name ? list_name : "(unknown list)");
    #endif

//  Atom *a = dynamic_cast<Atom *>o;
//  if (a) warning("object of type %s added to list %s\n",typeid(*o).name(), list_name ? list_name : "(unknown list)");
    o->list_refcount++;
}

void List::dec_refcount(Object *o)
{
    #ifdef LISTINOUT
     warning("object of type %s removed from list %s\n",typeid(*o).name(), list_name ? list_name : "(unknown list)");
    #endif
//  Atom *a = dynamic_cast<Atom *>o;
//  if (a) warning("object of type %s removed from list %s\n",typeid(*o).name(), list_name ? list_name : "(unknown list)");
    o->list_refcount--;
}

#endif
