/*
 *
 *   ^   |    sssss p   ddddd  fff  ggggg hhhh   iii  j   j    |   ^
 *  /|\  |    s     p   d     f   f   g   h   h i   i jj  j    |  /|\
 *   |   |    sss   p   ddd   f       g   hhhh  i   i j j j    |   |
 *   |  \|/   s     p   d     f   f   g   h   h i   i j  jj   \|/  |
 *   |   v    sssss ppp ddddd  fff    g   h   h  iii  j   j    v   |
 *
 *                           copyright 1999
 *                  Martijn Versteegh & Hein Zelle
 *
 */

/*
 * class implementation of Lattice
 */

#include <stdlib.h>
#include "lattice.h"
#include "utils.h"


/*
 * one global lattice pointer defined, always 
 * you have to initialise it yourself after you make your lattice
 */

Lattice *lattice = 0;


Lattice::Lattice(int width, int height)
{
    lattice = new (Atom **)[width];
    for (int i = 0; i < width; i++)
    {
        lattice[i] = new (Atom *)[height];
        for (int j = 0; j < height; j++)
            lattice[i][j] = NULL;
    }
    w = width;
    h = height;

    reset_x = 0;
    reset_y = 0;

    vote_x = 0;
    vote_y = 0;
}

Lattice::~Lattice()
{
    for (int i = 0; i < w; i++)
    {
        for (int j = 0; j < h; j++)
        {
            if (lattice[i][j])
                delete_object(lattice[i][j]);
        }
        delete [] lattice[i];
    }
    delete [] lattice;
}

Atom *Lattice::get(int x, int y)
{
    if (x >= 0 && x < w && y >= 0 && y < h)
       return lattice[x][y];

    return NULL;
}

// 0 = failure, 1 = succes
int Lattice::set(Atom *a, int x, int y)
{
    if (x >= 0 && x < w && y >= 0 && y < h)
    {
        if (lattice[x][y])
        {
           warning("Atom put in lattice at position (%d, %d)\n"
                   "    Where an atom was already present.\n"
                   "    (Bose-Einstein condensate not supported.)", x, y);
        }
        else
        {
            lattice[x][y] = a;
            a->x = x;
            a->y = y;
            return 1;
        }
    }
    else
    {
        warning("Lattice::set : invalid coordinates [%d,%d]", x, y);
    }
    
    // failure
    return 0;
}


void Lattice::clear(int x, int y)
{
    if (x >= 0 && x < w && y >= 0 && y < h)
    {
        if (lattice[x][y])
           delete_object(lattice[x][y]);

           
        lattice[x][y] = NULL;
    }
}


int Lattice::add(Electron *e, int x, int y)
{

    if (x < 0 && y < 0 && x >= w && y >= h)
       return -1;

    if (!(lattice[x][y]))
       set(new_atom(), x, y);

    lattice[x][y]->add(e);
//    warning("adding Object to lattice (%d, %d) %s", x, y, e->type_name);

        
    return 0;
}

int Lattice::remove(Electron *e, int x, int y)
{
    int ret;

    if (x < 0 && y < 0 && x >= w && y >= h)
       return -1;

    if (!(lattice[x][y]))
       return -1;

    ret = (lattice[x][y]->remove(e));




    return ret;
}

Electron *Lattice::find(int type,  unsigned int id)
{
    Electron *ret = NULL;
    static Electron *cached = NULL;

    if (cached && cached->actor_id == id)
        return cached;
    
    for (int i = 0; i < w; i++)
    {
        for (int j = 0; j < h; j++)
        {
            if (lattice[i][j])
            {
                ret = lattice[i][j]->find(type, id);
                if (ret)
                {
                   cached = ret;
                   return ret;
                }
            }
        }
    }
    return NULL;
}


// this could be optimized a little:
// we can use e->get_atom() directly to get to the atom
// no need to use gx() and gy() anymore
// pay attention: lattice is the 2d member Atom array,
// NOT the global Lattice * !!
void Lattice::move_electron(Electron *e, Electron *new_parent)
{
//    warning("lattice::move_electron : %p [%d, %d], %p [%d, %d], %p [%d, %d]",
//          e, e->gx(), e->gy(),
//          e->parent, e->parent->gx(), e->parent->gy(),
//          new_parent, new_parent->gx(), new_parent->gy()); 

    // simple case, no current parent;
    if (!e->parent)
    {
	Atom *a = e->get_atom();
	if (!a)
	{
	    warning("Lattice::move_electron() : moving an electron that was"
		    "not in the lattice yet! (%s, %d)",
		    e->type_name, e->actor_id);
	}

	a->remove(e);
        lattice[new_parent->gx()][new_parent->gy()]->add(e, new_parent);
        return;
    }

    // we have a parent electron
    if (e->get_atom() == new_parent->get_atom())
    {
        // both parent & new_parent are in the same atom, so just rewire
        // inventories
        e->parent->inv_del(e);
        new_parent->inv_add(e);
        return;
    }

    // do the long way
//    warning("the long way");
//    warning("lattice: e at [%d, %d]", e->gx(), e->gy());

    e->parent->get_atom()->remove(e);
//    lattice[e->parent->gx()][e->parent->gy()]->remove(e);

//    warning("lattice: e at [%d, %d]", e->gx(), e->gy());

    new_parent->get_atom()->add(e, new_parent);
//    lattice[new_parent->gx()][new_parent->gy()]->add(e, new_parent);

//    warning("lattice: e at [%d, %d]", e->gx(), e->gy());
}


void Lattice::set_reset(int x, int y)
{
    reset_x = x;
    reset_y = y;
}

void Lattice::get_reset(int *x, int *y)
{
    *x = reset_x;
    *y = reset_y;
}

void Lattice::set_vote(int x, int y)
{
    vote_x = x;
    vote_y = y;
}

void Lattice::get_vote(int *x, int *y)
{
    *x = vote_x;
    *y = vote_y;
}

Atom *Lattice::get_random_atom(Electron *entrant)
{
    Atom *a = 0;
    int
	max_tries = 1000,
	nr_tries = 0,
	x, y;

    // try until we find a valid room, and no more than 1000 times.
    while (!a && nr_tries < max_tries)
    {
	x = (int) ((double)w * rand() / (RAND_MAX + 1.0));
	y = (int) ((double)h * rand() / (RAND_MAX + 1.0));

	a = get(x, y);

	if (a && entrant)
	{
	    Electron *room = a->top_level_object;
	    ASSERT(room);

	    if (!room->can_be_entered(entrant))
	    {
		a = 0;
	    }
	}
    }

    if (!a)
    {
	fatal("Lattice::get_random_room : maximum tries (%d) exceeded.",
	      max_tries);
    }
    
    return a;
}
