/*
 *
 *   ^   |    sssss p   ddddd  fff  ggggg hhhh   iii  j   j    |   ^
 *  /|\  |    s     p   d     f   f   g   h   h i   i jj  j    |  /|\
 *   |   |    sss   p   ddd   f       g   hhhh  i   i j j j    |   |
 *   |  \|/   s     p   d     f   f   g   h   h i   i j  jj   \|/  |
 *   |   v    sssss ppp ddddd  fff    g   h   h  iii  j   j    v   |
 *
 *                           copyright 1999
 *                  Martijn Versteegh & Hein Zelle
 *
 */

/*
 *      freelist.cc
 *
 *      implementation of the freelist system.
 *      this could be placed in a class, but i'm lazy so instead we
 *      make the global variables static.
 */

/*
 *      example:
 *
 *      OBJ *new_OBJ()
 *      {
 *          OBJ *tmp = (OBJ *)new_object(COMMON_FREELIST_OBJ);
 *          if (!tmp)
 *          {
 *               tmp = new OBJ(args);
 *          }
 *          else
 *          {
 *               tmp->destroy();
 *               tmp->create(args);
 *          }
 *
 *          return tmp;
 *      }
 */

#include "freelist.h"
#include "utils.h"
#include <typeinfo>

#ifdef DEBUGMODE
#include "common.h"
#endif


static List *recycled_lists = 0;
static int total_lists = 0;
int nr_freelists;
Freelist_info *freelist_info = 0;

void init_freelists(int number_lists)
{
    total_lists = COMMON_FREELIST_LAST + number_lists;
    recycled_lists = new List[total_lists];

    freelist_info = new Freelist_info[total_lists];
        
    nr_freelists = total_lists;
}

static int for_all_callback(Object *o)
{
    o->freelist_index = -1;
    return List::FILTER_LEAVE;
}

void cleanup_freelists()
{
    
    // empty all the lists and DELETE all the objects in them!
    for (int i = 0; i < total_lists; i++)
    {
      // filter can also be used as a for_all function, byt selecting a
      // null 'to' List and letting the criterium return FILER_LEAVE for everything
      recycled_lists[i].filter(for_all_callback, 0);
      recycled_lists[i].destroy();
    }

    // delete the lists, and set the counter to 0
    delete [] recycled_lists;
       
    delete [] freelist_info;


    freelist_info = 0;
    nr_freelists = 0;
    recycled_lists = 0;
    total_lists = 0;
}

// functions to actually recycle objects

// this function can RETURN 0 !!! that means the recycle List was empty
Object *new_object(int list_nr)
{

    ASSERT(list_nr >=0);
    ASSERT(list_nr < total_lists);
    
    Object *ret = recycled_lists[list_nr].pop();
    if (ret)
    {
      freelist_info[list_nr].nr_hits++;
      freelist_info[list_nr].list_size--;
    }
    else
    {
      freelist_info[list_nr].nr_misses++;
    }
    return ret;
}

void delete_object(Object *obj)
{
    if (obj->freelist_index < 0)
       fatal("tried to put an illegal Object in a freelist");

    #ifdef DEBUGMODE
    if (!obj->allowed_in_freelist())
    {
	fatal("object of type %s put in freelist, but it's not allowed",
	      typeid(*obj).name());
    }
    #endif

    ASSERT(obj->allowed_in_freelist());

    obj->clear();


    #ifdef DEBUGMODE
    if (!going_down)
    {
        if (obj->list_refcount)
            fatal("object of type %s put in freelist, but it's refcount = %d\n",typeid(*obj).name(),obj->list_refcount);
    }
    #endif
    
    freelist_info[obj->freelist_index].nr_frees++;
    freelist_info[obj->freelist_index].list_size++;

    
    recycled_lists[obj->freelist_index].push(obj);
}

Freelist_info::Freelist_info()
{
    nr_hits = nr_misses = nr_frees = list_size = 0;
}


void dump_freelist_stats()
{
    warning("Freelist stats:");

/*
    warning("/===========================================================");
    warning("| number of freelists: %d", nr_freelists);
    warning("|===========================================================");
    for (int i = 0;i < nr_freelists;i++)
    {
        warning("| List %d", i);
        warning("|   hits=%d misses=%d frees=%d    current size=%d",
                     freelist_info[i].nr_hits,
                     freelist_info[i].nr_misses,
                     freelist_info[i].nr_frees,
                     freelist_info[i].list_size
                     );
        warning("|     nr of objects in the game=%d", freelist_info[i].nr_misses - freelist_info[i].nr_frees + freelist_info[i].nr_hits);
                     
        if ( freelist_info[i].list_size != freelist_info[i].nr_frees - freelist_info[i].nr_hits)
        {
             warning("|  ERROR!!! list_size != nr_frees - nr_hits !!!!!!!!!!!!!!!!");
             warning("|  this probably means an Object was created without the freelist");
             warning("|  system first being asked");
        }
        warning("|---------------------------------------------");
                     
    }
    warning("| the total number of objects allocated is equal to 'misses'");
    warning("\\===========================================================");
*/    
}


void delete_objects(List *l)
{
   Object *o;

   l->reset();

   while((o = l->pop()))
   {
       delete_object(o);
   }
}
