/*
 *
 *   ^   |    sssss p   ddddd  fff  ggggg hhhh   iii  j   j    |   ^
 *  /|\  |    s     p   d     f   f   g   h   h i   i jj  j    |  /|\
 *   |   |    sss   p   ddd   f       g   hhhh  i   i j j j    |   |
 *   |  \|/   s     p   d     f   f   g   h   h i   i j  jj   \|/  |
 *   |   v    sssss ppp ddddd  fff    g   h   h  iii  j   j    v   |
 *
 *                           copyright 1999
 *                  Martijn Versteegh & Hein Zelle
 *
 */

/*
 *    electron.h
 *
 *    class definition of the Electron
 *
 *    this is *not* a global header file for the electron program,
 *    Electron is the base class for all objects (weapons, rooms,
 *    players, etc) in the electron program.
 *
 *    all unique actor_types are to be defined in the file act_type.h
 */

#ifndef _ELECTRON_H_
#define _ELECTRON_H_

#include "object.h"
#include "trigger.h"
#include "act_type.h"
#include "vector.h"
#include "list.h"
#include "packet.h"

class Atom;

class Electron : public Object, public Trigger
{
public:
    Electron();
    Electron(int actor_type, int actor_id);
    ~Electron();
    
    // public constructors /destructor    
    void set(int _actor_type, int _actor_id);   
    virtual void clear();

    static unsigned int fid(); // fresh id 

    virtual void set_int_var(int var_id, int val);
    virtual void set_vector_var(int var_id, Vector val);
    virtual void set_vector_var(int var_id, int x, int y);
    virtual void set_string_var(int var_id, char const *string);
    virtual char const *get_string_var(int var_id) const;
    virtual int  get_int_var(int var_id) const;
    virtual Vector get_vector_var(int var_id) const;
    
    // because an electron can be inside another electron
    // it's x and y are accessed with these memberfunctions
    // which recurse to the parent if there is one
    int gx();
    int gy();
    void get_pos(int *x, int *y);

    //returns this electron's inventory, if it has one
    virtual List *inv();

    // add an electron into the inventory
    void inv_add(Electron *e);
    
    // returns true (!0) if the inventory contains this Object
    // nice hack: start at e, and follow parents to see if we find
    // this electron
    int inv_contains(Electron *e);
    
    // returns true (!0) if the inventory contains this Object,
    // any Object in the inventory contains it int its inventory
    int inv_contains_recursive(Electron *e);

    // removes an Object from the inventory (does not delete the thing itself)
    int inv_del(Electron *e);

    // if this electron has a parent, this function will
    // move the specified electron to that parent's inventory
    // it returns TRUE if it dropped anything
    // if this Object does not have a parent, it does not do anything
    // but printing a warning
    int inv_drop(Electron *e);
    // the same as above, but it searches the whole inventory tree for the Object
    // to drop (it does drop it to this electron's parent however)
    int inv_drop_recursive(Electron *e);

    virtual int is_group(int group) const;
    virtual int can_be_entered(Electron const *other) const;

    // hmd functions: health, damage_multiplier, damage
    // values are in promille  (1 / 1000)
    virtual int has_hmd() { return 0; }

    virtual int get_h() { return 1000; }
    virtual int get_m() { return 0; }
    virtual int get_d() { return 0; }

    virtual void set_h(int new_h) { return; }
    virtual void set_m(int new_m) { return; }
    virtual void set_d(int new_d) { return; }
    
    void get_hm(int *h, int *m);
    void get_hmd(int *h, int *m, int *d);
    void set_hm(int new_h, int new_m);
    void set_hmd(int new_h, int new_m, int new_d);

    // do damage to another electron
    void do_damage(Electron *bokje, Electron *killer);

    // receive damage. apply multiplier first
    // virtual because splayer overloads it to notify
    // the game system of what's happening
    virtual void rec_damage(int hoeveel, Electron *killer);

    // receive bare damage, do not apply multiplyer
    void rec_damage_bare(int promille, Electron *killer);
    
    virtual void die(Electron *killer);

    Atom *get_atom(); //returns the atom the electron is in, checking it's parent if it has one

    
    // network functions
    // this one writes all variables of the electron to a callback
    // function, in the form of info packets. the Object *to will be
    // passed on to the callback function. (internally, to is either
    // a List or a Plug)
    // when overloading, use
    // WRITE_VARS_PROTO;
    // WRITE_VARS_FUNCTION()
    // (see below)
    virtual void write_vars(
                            void (*callback)(Packet *p, Object *to),
                            Object *to
                           ) const;
    virtual void write_health(
                              void (*callback)(Packet *p, Object *to),
                              Object *to
                              ) const;

    // firing functions
    virtual int fire_to(Atom *a, Electron *pl);
    virtual int fire_dir(int dir, Electron *pl);
    virtual int fire_at(Electron *target, Electron *pl);

    // map functions
    virtual void map_finish() { return; } // called after electron is created
    
    // state machine for client electrons
    virtual int get_state() { return 0; }

     // tells the electron to re-examine it's data for a new state
    // optionally with some argument
    virtual void change_state(int argument = -1) { return; }
    
    // public member variables
    
    // this electron's parent, if it is contained inside another Object.
    Electron *parent;     //! mot dit wel public???

    Atom *atom;

    unsigned int actor_id;
    unsigned int actor_type;
    unsigned int list_id;
    char const *type_name;
    int dead;                  // are we dead?
    
    Object *extra_data; /* any extra data needed by server or client , but not both */

//    we don't understand why this was set to 0 for electrons anymore.
//    removed ALLOW_FREELIST macro, object creates virtual function with
//    result 1.
    
//    int allowed_in_freelist() { return 0;};

protected:


    // this electron can contain other electrons (for example a player has an inventory)
    List *inventory;
};


// definition of the write_vars function body
// and WE don't mind this .. HAHAHAHA


#define VARS_PROTO                                                 \
virtual void set_int_var(int var_id, int val);                     \
virtual void set_vector_var(int var_id, Vector val);               \
virtual void set_vector_var(int var_id, int x, int y);             \
virtual void set_string_var(int var_id, char const *string);       \
virtual char const *get_string_var(int var_id) const;              \
virtual int  get_int_var(int var_id) const;                        \
virtual Vector get_vector_var(int var_id) const;                   \
virtual void write_vars(void (*callback)(Packet *p, Object *to),   \
                Object *to) const;


#define VARS_FUNCTION( t )                                         \
int t :: get_int_var(int var_id) const                             \
{                                                                  \
    if (var_id >= 0 && var_id < INTVARS_LAST)                      \
        return int_vars[var_id];                                   \
                                                                   \
    warning(" var index should be in the range  0 - %d "           \
            " for Object type %s", INTVARS_LAST -1, type_name);    \
    return -1;                                                     \
}                                                                  \
                                                                   \
char const * t :: get_string_var(int var_id) const                 \
{                                                                  \
    if (var_id >= 0 && var_id < STRINGVARS_LAST)                   \
        return string_vars[var_id];                                \
                                                                   \
    warning(" var index should be in the range  0 - %d "           \
            " for Object type %s", STRINGVARS_LAST -1, type_name); \
    return "--";                                                   \
}                                                                  \
                                                                   \
Vector t :: get_vector_var(int var_id) const                       \
{                                                                  \
    if (var_id >= 0 && var_id < VECTORVARS_LAST)                   \
        return vector_vars[var_id];                                \
                                                                   \
    warning(" var index should be in the range  0 - %d "           \
            " for Object type %s", VECTORVARS_LAST -1, type_name); \
                                                                   \
    Vector v(-1, -1);                                              \
    return v;                                                      \
}                                                                  \
                                                                   \
void t :: set_int_var(int var_id, int val)                 \
{                                                                  \
    if (var_id >= 0 && var_id < INTVARS_LAST)                      \
        int_vars[var_id] = val;                                    \
}                                                                  \
                                                                   \
void t :: set_string_var(int var_id, char const * val)     \
{                                                                  \
    if (var_id >= 0 && var_id < STRINGVARS_LAST)                   \
    {                                                              \
        if (string_vars[var_id])                                   \
            delete string_vars[var_id];                            \
        string_vars[var_id] = sstrdup(val);                        \
    }                                                              \
}                                                                  \
                                                                   \
void t :: set_vector_var(int var_id, Vector val)           \
{                                                                  \
    if (var_id >= 0 && var_id < VECTORVARS_LAST)                   \
        vector_vars[var_id] = val;                                 \
}                                                                  \
                                                                   \
void t :: set_vector_var(int var_id, int val1, int val2)      \
{                                                                  \
    if (var_id >= 0 && var_id < INTVARS_LAST)                      \
    {                                                              \
        Vector v(val1, val2);                                      \
        vector_vars[var_id] = v;                                   \
    }                                                              \
}                                                                  \
                                                                   \
void t :: write_vars(void (*callback)(Packet *pak, Object *tok),  \
                          Object *to) const                        \
{ \
    Packet *p; \
    \
    for (int i = 0; i < INTVARS_LAST; i++) \
    { \
        p = new_packet_info_num(actor_type, actor_id, i, int_vars[i]); \
        callback(p, to); \
    } \
    \
    for (int i = 0; i < STRINGVARS_LAST; i++) \
    { \
        p = new_packet_info_string(actor_type, actor_id, i, string_vars[i]); \
        callback(p, to); \
    } \
    \
    for (int i = 0; i < VECTORVARS_LAST; i++) \
    { \
        p = new_packet_info_vect(actor_type, actor_id, i, vector_vars[i]); \
        callback(p, to); \
    } \
}


#define HMD_ENUM     \
        HEALTH = 0,  \
        MULT,        \
        DAMAGE   


#define HMD_PROTO                                                  \
        int has_hmd() { return 1; }                                \
        int get_h() { return int_vars[HEALTH]; }                   \
        int get_m() { return int_vars[MULT]; }                     \
        int get_d() { return int_vars[DAMAGE]; }                   \
        void set_h(int new_h) { int_vars[HEALTH] = new_h; }        \
        void set_m(int new_m) { int_vars[MULT] = new_m; }          \
        void set_d(int new_d) { int_vars[DAMAGE] = new_d; }        

#define WRITE_HMD_PROTO                                            \
        void write_health(void (*callback)(Packet *p, Object *to), \
                          Object *to) const

#define WRITE_HMD_FUNCTION( t )                                    \
void t :: write_health(void (*callback)(Packet *pak, Object *tok), \
                       Object *to) const                           \
{                                                                  \
    Packet *p;                                                     \
                                                                   \
    p = new_packet_info_num(actor_type, actor_id,                  \
                            t :: HEALTH,                           \
                            int_vars[t :: HEALTH]);                \
    callback(p, to);                                               \
}

// no longer needed: Object already defines a virtual function
// that returns 1 (allow in freelist)
// overload function allowed_in_freelist to return 0 for exceptions
// that should *not* go in the freelist.

// #define ALLOW_FREELIST int allowed_in_freelist() { return 1;}

#endif  
