/*
 *
 *   ^   |    sssss p   ddddd  fff  ggggg hhhh   iii  j   j    |   ^
 *  /|\  |    s     p   d     f   f   g   h   h i   i jj  j    |  /|\
 *   |   |    sss   p   ddd   f       g   hhhh  i   i j j j    |   |
 *   |  \|/   s     p   d     f   f   g   h   h i   i j  jj   \|/  |
 *   |   v    sssss ppp ddddd  fff    g   h   h  iii  j   j    v   |
 *
 *                           copyright 1999
 *                  Martijn Versteegh & Hein Zelle
 *
 */

/*
 *      electron.cc
 *
 *      class implementation of the Electron
 */

#include <stdlib.h> // NULL
#include "electron.h"
#include "atom.h"
#include "utils.h"

// initialize cur_actor_id with ACTOR_LAST to allow the client to do some
// dirty tricks in it's command parser: it wants to distinguish Object
// types from Object id's
static unsigned int cur_actor_id = ACTOR_LAST;

Electron::Electron()
{
    actor_type = ACTOR_SYSTEM;     // these defaults are NOT FOR SENSIBLE USE!
    actor_id = 0;                  // Derived electrons should ALWAYS have
    list_id = 0;                   // a (default) constructor.
    parent = NULL;
    inventory = NULL;
    atom = NULL;
    type_name = "Unnamed Electron";
    extra_data = NULL;
    dead = 0;                      // we start alive
}

Electron::Electron(int _actor_type, int _actor_id)
{
    extra_data = NULL;
    type_name = "Unnamed Electron";
    set(_actor_type, _actor_id);
}

void Electron::set(int _actor_type, int _actor_id)
{
    actor_type = _actor_type;
    actor_id = _actor_id;
    list_id = translate_actor_type[_actor_type];
    parent = NULL;
    atom = NULL;
    inventory = NULL;
    dead = 0;

    // DO NOT SET extra_data to NULL here, it is meant to be kept in thew freelist and
    // only deleted once at program end
}

void Electron::clear()
{
    parent = NULL;
    atom = NULL;

    if (inventory)
       inventory->clear();
       
    dead = 1;
}

unsigned int Electron::fid()
{
    if (cur_actor_id >= 0xFFFFFFFF)
	fatal("Too many objects created: server ran too long.\n"
	      "Nag Electron developers to implement garbage collection.");
    
    return ++cur_actor_id;
}

void Electron::set_int_var(int var_id, int val)
{
    warning("set_int_var(): Object of type %s has no intvars", type_name);
}

void Electron::set_vector_var(int var_id, int _x, int _y)
{
    Vector v(_x, _y);
    set_vector_var(var_id, v);
}

void Electron::set_vector_var(int var_id, Vector val)
{
    warning("set_vector_var(): Object of type %s has no vectorvars", type_name);
}

void Electron::set_string_var(int var_id, char const *string)
{
    warning("set_string:var(): Object of type %s has no stringvars", type_name);
}

int Electron::get_int_var(int var_id) const
{
    warning("get_int_var() : Object of type %s has no intvars", type_name);
    return -1;
}

Vector Electron::get_vector_var(int var_id) const
{
    Vector v(-1, -1);
    
    warning("get_vector_var() : Object of type %s has no vectorvars", type_name);
    return v;
}

char const *Electron::get_string_var(int var_id) const
{
   warning("get_string_var() : Object of type %s has no stringvars", type_name);
   return 0;
}

int Electron::gx()
{
    Atom *a = get_atom();

    if (!a)
    {
//        warning("Electron::gx() electron (%s, %d) has no atom yet, so it does not yet have an x", type_name, actor_id);
        return -1;
    }

    return a->x;

}


int Electron::gy()
{
    Atom *a = get_atom();

    if (!a)
    {
//        warning("Electron::gy() electron (%s, %d) has no atom yet, so it does not yet have an y", type_name, actor_id);
        return -1;
    }

    return a->y;

}


void Electron::get_pos(int *x, int *y)
{
    *x = gx();
    *y = gy();
}




List *Electron::inv()
{
    return inventory;
}

void Electron::inv_add(Electron *e)
{
    if (!inventory)
    {
        inventory = new List;
    }

    e->parent = this;
    inventory->push(e);

    e->atom = NULL; // inf the elctron is now in this elctron's inventory, it can never be the top level objetc anymore
}

// nice hack: start at _e, and follow the parent pointer to see if
// it points to the current electron
int Electron::inv_contains(Electron *_e)
{
    if (_e->parent == this)
	return -1;
    else
	return 0;
}

// nice hack: start at _e, and follow the parent pointers recursively
// to see if they point to the current electron
int Electron::inv_contains_recursive(Electron *_e)
{
    if (!_e->parent)
	return 0;

    if (_e->parent == this)
	return -1;

    return inv_contains_recursive(_e->parent);
}

int Electron::inv_del(Electron *e)
{
    int ret;
    
    if (!inventory)
    {
        warning("asked for inventory of a %s, but that doesn't have one!", type_name);
        return 0;
    }
    ret = (inventory->pop(e) != NULL);

    if (ret)
        e->parent = NULL;
    return  ret;
}


int Electron::is_group(int group) const
{
    return 0;
}

// electrons can (by default) not be entered
// override this function to make it possible
int Electron::can_be_entered(Electron const *other) const
{
    return 0;
}

// if this electron has a parent, this function will
// move the specified electron to that parent's inventory
// it returns TRUE if it dropped anything
// if this Object does not have a parent, it does not do anything
// but printing a warning
int Electron::inv_drop(Electron *e)
{
    if (!inventory || !parent)
       return 0;

    if (!inv_del(e))
       return 0;

    parent->inv_add(e);

    return -1;
}

// the same as above, but it searches the whole inventory tree for the Object
// to drop (it does drop it to this electron's parent however)
int Electron::inv_drop_recursive(Electron *_e)
{
    // are we in the current electrons inventory, somewhere?
    if (!inv_contains_recursive(_e))
	return 0;

    // yes? remove from it's parent, and insert into our parent
    _e->parent->inv_del(_e);
    parent->inv_add(_e);

    return -1;
}

Electron::~Electron()
{
    if (inventory)
        delete inventory;

    if (extra_data)
        delete extra_data;
}

// health, multiplier and damage functions
// most of these are inline in the header file

void Electron::get_hm(int *h, int *m)
{
    *h = get_h();
    *m = get_m();
}

void Electron::get_hmd(int *h, int *m, int *d)
{
    *h = get_h();
    *m = get_m();
    *d = get_d();
}

void Electron::set_hm(int new_h, int new_m)
{
    set_h(new_h);
    set_m(new_m);
}

void Electron::set_hmd(int new_h, int new_m, int new_d)
{
    set_h(new_h);
    set_m(new_m);
    set_d(new_d);
}

void Electron::do_damage(Electron *bokje, Electron *killer)
{
    Atom *a;
    Electron
	*room;

    // no damage in safe rooms, nor from safe rooms
    a = bokje->get_atom();
    ASSERT(a);
    room = a->top_level_object;
    ASSERT(room);

    if (room->actor_type == ACTOR_SAFE_ROOM)
	return;
    
    if (killer)
    {
	a = killer->get_atom();
	ASSERT(a);
	room = a->top_level_object;
	ASSERT(room);

	if (room->actor_type == ACTOR_SAFE_ROOM)
	    return;
    }

    bokje->rec_damage(get_d(), killer);
}

void Electron::rec_damage(int hoeveel, Electron *killer)
{
    int mult = get_m();

    // if we're a player, and if killer is also a player,
    // notify the game
    
    rec_damage_bare(mult * hoeveel / 1000, killer);
}

void Electron::rec_damage_bare(int promille, Electron *killer)
{
    int health = get_h();
    
    if (promille < 0)
    {
	warning("Electron::rec_damage_bare : negative damage %d", promille);
    }
    else
    {
	warning("Electron::rec_damage_bare : %d damage received", promille);
    }
    
    health -= promille;
    if (health < 0)
	health = 0;
    
    set_h(health);

    warning("health = %d", health);
    
    if (!health)
    {
	die(killer);
    }
}

void Electron::die(Electron *killer)
{
    warning("Electron %d (%s) died!", actor_id, type_name);
}

// recepy for overloading write_vars:
// create an info packet for each internal variable
// call the callback function as follows:
// callback(p, to);
// for each created packet.

// use the WRITE_VARS_PROTO and WRITE_VARS_FUNCTION macros for this

void Electron::write_vars(void (*callback)(Packet *p, Object *to),
			  Object *to) const
{
    return;   // not overloaded? no variables sent
}

// use the HMD_ENUM, HMD_PROTO and WRITE_HMD_FUNCTION macros for this

void Electron::write_health(void (*callback)(Packet *p, Object *to),
			  Object *to) const
{
    return;   // not overloaded? no variables sent
}


Atom *Electron::get_atom()
{
    if (parent)
        return parent->get_atom();

    return atom;
}

// firing functions
// plain electrons cannot fire, so the default functions
// don't do anything. overload to make them functional
int Electron::fire_to(Atom *a, Electron *pl)
{
    return 0;
}

int Electron::fire_dir(int dir, Electron *pl)
{
    return 0;
}

int Electron::fire_at(Electron *target, Electron *pl)
{
    // default implementation: find the coordinates
    // of target and fire at those. If fire_to is not defined,
    // this will simply do nothing. It makes it so that
    // vehicles like a tank automatically have a working
    // fire_at function if they already have a fire_to function.

    TRACE("Electron::fire_at");
    
    Atom *a = target->get_atom();

    if (!a)
    {
	return 0;
    }

    warning("electron::fire_at : located atom %d, %d", a->x, a->y);
    
    return fire_to(a, pl);
}
