/*
 *
 *   ^   |    sssss p   ddddd  fff  ggggg hhhh   iii  j   j    |   ^
 *  /|\  |    s     p   d     f   f   g   h   h i   i jj  j    |  /|\
 *   |   |    sss   p   ddd   f       g   hhhh  i   i j j j    |   |
 *   |  \|/   s     p   d     f   f   g   h   h i   i j  jj   \|/  |
 *   |   v    sssss ppp ddddd  fff    g   h   h  iii  j   j    v   |
 *
 *                           copyright 1999
 *                  Martijn Versteegh & Hein Zelle
 *
 */

/*
 *     class implementation of the Atom
 */
#include <stdlib.h>
#include <stdio.h>
#include "atom.h"
#include "utils.h"

Atom::Atom()
{
    set();
}

void Atom::set()
{
    top_level_object = NULL;
    freelist_index = COMMON_FREELIST_ATOM;
    x = y = -1;
}

Atom::~Atom()
{
    clear();
}

void Atom::clear()
{
    // clear all the lists
    for (int i = 0; i < LIST_LAST; i++)
    {
        contents[i].clear();
    }

}
// careful: this one actually DELETES the objects that are stored in the
// lists. this means you cannot delete_object() two atoms that both contain the
// same Object!
void Atom::delete_objects()
{
    // delete_object all data in the lists
    for (int i = 0; i < LIST_LAST; i++)
    {
        contents[i].reset();

        ::delete_objects(&(contents[i]));
    }
}

List *Atom::operator[](int index)
{
    return get(index);
}


List *Atom::get(int index)
{
    if (index < 0 || index >= LIST_LAST)
        fatal("Atom::operator[] : invalid index %d", index);
    
    return &contents[index];
}

/* recursively adds an electron and all its children to an Atom*/
int Atom::lowlevel_add(Electron *to_add)
{
    Electron *e;
    List *inventory;

    // add the electron to this atom's search lists
    contents[to_add->list_id].push(to_add);

    // if this electron has an inventory, add all electrons in it also to this atom
    inventory = to_add->inv();
    
    if (inventory)
    {
        for(inventory->reset();(e = (Electron *)(inventory->get()));inventory->next())
        {
            lowlevel_add(e);
        }
    }

    return 0;
}

// returns -1 on failure (parent niet in dit atom)
// adds an electron to an atom, specifying it's parent electron,
// it will be put in it's parent's inventory
int Atom::add(Electron *e, Electron *parent = NULL)
{
    // check if this atom already contains a top_level_object
    if (!top_level_object) // there is no top_level_object yet
    {
        if (parent)  // error: there is nothing here yet , so this electron's parent won't be either
        {
           warning("Atom::Add : Atom contains no top_level_object");
           return -1;
        }
        else // ok, we'll make this the top_level_object
        {
            if (!(e->is_group(GROUP_ROOM)))
               warning("Atom::Add : top_level_object should be a room");
            top_level_object = e;
            e->atom = this; // make sure the top_levekl_object's atom pointer points to the atom
        }
    }
    else  // there is a top_level_object
    {
        if (parent) // ok, we have a parent, check if it's here.
        {
            if ((parent != top_level_object)
                && !(top_level_object->inv_contains_recursive(parent))
               )
            {
                warning("Atom::add : Atom does not contain the parent to add thisd Object to");
                return -1;
            }
        }
        else // parent not specified, use the top_level_object
        {
            parent = top_level_object;
        }
    }

    // add the elctron and all it's inventory to this atom
    lowlevel_add(e);

    // add this electron to the inventory of it's parent
    if (parent)
        parent->inv_add(e);

    return 0;
}


// careful: 0 = success, -1 = failure
/* recursively remove an electron
 */
int Atom::lowlevel_remove(Electron *to_remove)
{
    Electron *e;
    List *inventory;


    // remove it from the searchlist
    
    if (contents[to_remove->list_id].pop(to_remove))
    {
        inventory = to_remove->inv();
        if (inventory)
        {
            // also remove the inventory
            for(inventory->reset();(e = (Electron *)(inventory->get()));inventory->next())
            {
                lowlevel_remove(e);
            }
        }
        
    
        return 0;
    }
    else
    {
        warning("Atom::lowlevel_remove :Object type %s with id %d is not in this atom",
                 to_remove->type_name, to_remove->actor_id);
        return -1;
    }
}


// warning , returns -1 on failure
int Atom::remove(Electron *to_remove)
{
    if (!lowlevel_remove(to_remove))
    {
        if (to_remove->parent) // also remove it from it's parent's inventory
            to_remove->parent->inv_del(to_remove);

        if (to_remove == top_level_object)
        {
            to_remove->atom = NULL;
            top_level_object = NULL;
        }

        return 0;
    }

    return -1;

}

// atom::find should expect invalid arguments.
// a client can send a lot of bullshit arguments to the server,
// in commands such as 'pickup'. this will cause the server
// to call a 'find' on the atom with invalid arguments. If the
// numbers are large, they may even be wrapped to negative numbers.

Electron *Atom::find(unsigned int actor_type, unsigned int id)
{
    Electron *ret;
    int list_type;

    if (actor_type < 0 || actor_type >= ACTOR_LAST)
    {
        warning("Atom::find() : actor_type %d does not exist", actor_type);
        return NULL;
    }
    
    list_type = translate_actor_type[actor_type];
//    warning("finding (%d, %d), list_type = %d VEHICLE=%d", actor_type, id, list_type, LIST_VEHICLE);


    for (contents[list_type].reset();
         (ret = (Electron *)(contents[list_type].get()));
         contents[list_type].next())
    {
//        warning(" checking %d", ret->actor_id);
        
        if (ret->actor_id == id && ret->actor_type == actor_type)
        {
//           warning("found it");
           return ret;
        }
    }
//    warning("not found it");
    return NULL;
}

// same as find, except this one returns the first electron of the
// given type that is found in the atom. Useful for checking if a
// certain type of electron exists in a room.
Electron *Atom::find_type(unsigned int actor_type)
{
    Electron *ret;
    int list_type;

    if (actor_type < 0 || actor_type >= ACTOR_LAST)
    {
        warning("Atom::find() : actor_type %d does not exist", actor_type);
        return NULL;
    }
    
    list_type = translate_actor_type[actor_type];

    for (contents[list_type].reset();
         (ret = (Electron *)(contents[list_type].get()));
         contents[list_type].next())
    {
	// if it has the right type, return it
        if (ret->actor_type == actor_type)
        {
           return ret;
        }
    }

    return NULL;
}

Atom *new_atom()
{
    Atom *a;

    a = (Atom *)new_object(COMMON_FREELIST_ATOM);

    if (a)
    {
        a->set();
    }
    else
    {
        a = new Atom();
    }

    return a;
}

Atom *storage;
