/*
 *
 *   ^   |    sssss p   ddddd  fff  ggggg hhhh   iii  j   j    |   ^
 *  /|\  |    s     p   d     f   f   g   h   h i   i jj  j    |  /|\
 *   |   |    sss   p   ddd   f       g   hhhh  i   i j j j    |   |
 *   |  \|/   s     p   d     f   f   g   h   h i   i j  jj   \|/  |
 *   |   v    sssss ppp ddddd  fff    g   h   h  iii  j   j    v   |
 *
 *                           copyright 1999
 *                  Martijn Versteegh & Hein Zelle
 *
 */

/*
 *     act_type.h
 *
 *     definition of all actor_types
 *     actor types must be unique unsigned integers, ranging from
 *     0 to 65535. define a new actor_type for each new type of Object
 *     that is created in the game. the enum will guarantee continuous
 *     numbers, now we just can't create more than 65536 types :)
 */

#ifndef _ACT_TYPE_H_
#define _ACT_TYPE_H_

#include "utils.h"

// all the actor types
// if you add an actor_type here, UPDATE the translate_types array
// in the file act_type.cc !!!!!
enum actor_types
{
    ACTOR_SYSTEM = 0,
    ACTOR_SIMPLE_ROOM,
    ACTOR_PLAYER,
    ACTOR_LIGHTCYCLE,
    ACTOR_KEY,
    ACTOR_CYCLEWALL,
    ACTOR_TANK,
    ACTOR_GRENADE,
    ACTOR_EXPLOSION,
    ACTOR_LAUNCHER,
    ACTOR_BLUE_BALL,
    ACTOR_YELLOW_BALL,
    ACTOR_RED_BALL,
    ACTOR_HOVER,
    ACTOR_LASERBEAM,
    ACTOR_MINE,
    ACTOR_DOOR,
    ACTOR_WATER,
    ACTOR_DOORKEY,
    ACTOR_AMBULANCE,
    ACTOR_MEDICAL_PAD,
    ACTOR_AMMO_PAD,
    ACTOR_REPAIR_PAD,
    ACTOR_FIETSENMAKER,
    ACTOR_WEAPONSMITH,
    ACTOR_VOTE_ROOM,
    ACTOR_VOTEMACHINE,
    ACTOR_WEAPONMACHINE,
    ACTOR_SAFE_ROOM,
    ACTOR_LAST
};

enum actor_groups
{
    GROUP_VEHICLE = 0,
    GROUP_WEAPON,
    GROUP_ROOM,
    GROUP_BULLET,       // bullets, grenades, laser beams, etc
    GROUP_PAD,
    GROUP_LAST
};

// forward declaration of Electron to prevent circular inclusion troubles
class Electron;
class Packet;

extern void init_translate_types();
extern void shutdown_translate_types();
extern void init_global_actor_statics();
extern void shutdown_global_actor_statics();
    
extern void set_electron_var(Electron *e, Packet const *p);

struct actor_entry
{
    int actor_type;
    int list_id;
    char const *permissions[3];
};

extern int *translate_actor_type;
extern actor_entry translate_types[];

// new_actor function(s)
// these should be used as follows:
// new_actor<Lightcycle>(Electron::fid())
// new_actor<Simple_room>(Electron::fid())
// etcetera

template <class T>
int get_correct_freelist_index()
{
   
   T *init = new T;     // force the freelist_index to be set for this type
   int ret = init->freelist_index;
   init->freelist_index = -1;
   
   delete init;

   return ret;
}

template <class T>
T *new_actor(int _actor_id)
{
    // we want to initialise correct_freelist_index to the freelist index of
    // T, but only once. we cannot use a static Object of type T, because
    // that will give an error at program exit, because the freelist index
    // would be valid, though it would not be in a freelist.
    // we don't want to create such an Object once and copy the
    // freelist_index to correct_freelist_index, and then deleting it again
    // since this would involve an extra check for each call.
    // hence we use another template function , to get this done in a static
    // initialisation (only once).
    // we know this is over-optimisation, but we couldn't resist ;-)
    static int correct_freelist_index = get_correct_freelist_index<T>();

    T *ret;

//    standard cast doesn't work with virtual baseclass
//    ret = (T *)new_object(correct_freelist_index);

//    new insight (and bugs) has led us to believe dynamic_cast is better
//    than reinterpret_cast. time will tell.
    
    ret = dynamic_cast<T *>(new_object(correct_freelist_index));
    
    if (!ret)
    {
	ret = new T;
    }

    ASSERT(ret->allowed_in_freelist());

    ret->set(_actor_id);

    return ret;
}

extern Electron *global_actor_statics[ACTOR_LAST];

#endif
