#include <allegro.h>
#include "srend.h"
#include "client/clfrlist.h"

#include "sampmap.inc"


Voice::Voice()
{
    freelist_index = CLIENT_FREELIST_VOICE;

    last_freq = 0;


    volume_sweep_deltatime = frequency_sweep_deltatime = 0;
    volume_sweep_starttime = frequency_sweep_starttime = 0;
    is_running = 0;
}


int Voice::dist()
{
    return (ear_x - x) * (ear_x - x) + (ear_y - y) * (ear_y - y) + (ear_z - 3) * (ear_z - 3);
}


void Voice::set(int _x,int _y)
{


   x = _x;
   y = _y;
   next = prev = NULL;
   owner = NULL;
   sample = NULL;
   is_running = 0;
   allegro_voice = -1;
   stop_time = -1;
   volume = 255;

   priority = 1000;
}

void Voice::set(Electron *_owner)
{

   owner = _owner;
   x = y = 0;
   next = prev = NULL;
   sample = NULL;
   is_running = 0;
   allegro_voice = -1;
   stop_time = -1;
   volume = 255;

   priority = 1000;
   
}


Voice::~Voice()
{
   ASSERT(!prev);
   ASSERT(!next);
}


void Voice::set_sample(Sound_renderer::SAMPLES sampindex)
{
   if (!srenderer->do_sound)
      return;
      
   int was_running = is_running;

   if (is_running)
       stop();


   sample = srenderer->get_sample(sampindex);

   if (allegro_voice >=0)
       reallocate_voice(allegro_voice, sample);


   frequency = base_frequency = sample->freq;
   
   if (was_running)
       start();

}

void Voice::set_volume(int _volume, int delta)
{

    if (delta)
    {
         volume_sweep_starttime = gtime;
         volume_sweep_deltatime = delta;
         volume_sweep_start = volume;
         volume_sweep_delta = _volume - volume;
    }
    
    volume = _volume;


    do_pan_volume_frequency();

}


void Voice::set_speed(double relative_speed, int delta)
{

    if (delta)
    {
         frequency_sweep_starttime = gtime;
         frequency_sweep_deltatime = delta;
         frequency_sweep_start = frequency;
         frequency_sweep_delta = (int)(base_frequency * relative_speed) - frequency;
//         severe("sweep start %d  %d   %d %d", frequency_sweep_starttime , frequency_sweep_deltatime,frequency_sweep_start,frequency_sweep_delta);
    }
    else
    {
      frequency = (int)(base_frequency * relative_speed);
    }
//   severe("freq set to %d", frequency);

   do_pan_volume_frequency();

}

void Voice::set_priority(int p)
{
   priority = p;
}


int Voice::has_finished()
{
     if (!is_running || allegro_voice < 0)
         return TRUE;

     if (voice_get_position(allegro_voice) < 0)
         return TRUE;

     return FALSE;



}


void Voice::start()
{
   if (is_running)
        return;
        
   is_running = TRUE;
   

   if (allegro_voice >=0)
       voice_start(allegro_voice);

   stop_time = -1;

   srenderer->notify_start(this);
}

void Voice::start_once()
{

   if (allegro_voice <=0)
        return;
   
   is_running = TRUE;
   voice_set_playmode(allegro_voice, PLAYMODE_PLAY);
   voice_start(allegro_voice);

}


void Voice::stop()
{
   is_running = FALSE;
   if (allegro_voice >=0)
       voice_stop(allegro_voice);
   srenderer->notify_stop(this);
       
}

void Voice::stop(int in_time)
{
   stop_time = in_time;
}


void Voice::update_logic(int _ear_x, int _ear_y, int _ear_z)
{
   ear_x = _ear_x;
   ear_y = _ear_y;
   ear_z = _ear_z;


   if (owner)
   {
      x = owner->gx();
      y = owner->gy();
   }

   if (is_running && allegro_voice >=0)
       do_pan_volume_frequency();


   compensated_priority = calc_priority(); // combine with do_pan_volume to do distc calc only once


   if (stop_time >0)
   {
      stop_time--;
      if (!stop_time)
         stop();
   }

   // do new lookup for priority? or let soundrenderer decide?
}



int Voice::calc_priority()
{
     if (!is_running)
        return 0;

   int adjprior = 25*priority / (1+dist());

   return MIN(adjprior, priority);
}


int Voice::needs_phys_voice()
{
      if (is_running && sample)
          return TRUE;

      return FALSE;
}

int Voice::has_phys_voice()
{
      if (allegro_voice>=0)
          return TRUE;

      return FALSE;
}


void Voice::give_phys_voice()
{
//   severe("%p voice got voice, has %d", this, allegro_voice);
   ASSERT(allegro_voice <0);

   if (!sample)
      return;

   allegro_voice = allocate_voice(sample);

   voice_set_playmode(allegro_voice, PLAYMODE_LOOP);

   do_pan_volume_frequency();

   if (is_running)
       voice_start(allegro_voice);

}

void Voice::take_phys_voice()
{
   if (allegro_voice >=0 && is_running)
       voice_stop(allegro_voice);


   if (allegro_voice>=0)
       deallocate_voice(allegro_voice);
       
   allegro_voice = -1;

   
}


void Voice::do_pan_volume_frequency()
{
   int v;
   int pan;
   int t;

   if (volume_sweep_deltatime && (gtime < volume_sweep_starttime + volume_sweep_deltatime))
   {
       t = gtime - volume_sweep_starttime;
       volume = volume_sweep_start + t * volume_sweep_delta / (int)volume_sweep_deltatime;
//       message(" gtime %d  sws %d delt %d", gtime, volume_sweep_starttime, volume_sweep_deltatime);
   }
   
   if (frequency_sweep_deltatime && (gtime < frequency_sweep_starttime + frequency_sweep_deltatime))
   {
       t = gtime - frequency_sweep_starttime;
       frequency = frequency_sweep_start + t * frequency_sweep_delta / (int)frequency_sweep_deltatime;
   }

   v = volume * 20;
   v /= (dist() + 1);
   voice_set_volume(allegro_voice, MIN(v, 255));


   pan  =  (15 * (x - ear_x)/(abs(ear_y - y) + 1)) + 128;

   pan = MID(0, pan, 255);

   voice_set_pan(allegro_voice, pan);
//   warning("volume %d pan %d", v, pan);
   
   if (last_freq != frequency)
      voice_set_frequency(allegro_voice, frequency);

   last_freq = frequency;

}


Voicelist::Voicelist()
{
   head = NULL;
}

void Voicelist::add(Voice *v)
{
   ASSERT(!v->prev);
   ASSERT(!v->next);


   v->next = head;

   if (head)
      head->prev = v;

   head = v;
}

void Voicelist::remove(Voice *v)
{
    if (v->prev)
        v->prev->next = v->next;

    if (v->next)
        v->next->prev = v->prev;

    if (v == head)
       head = v->next;

    v->next = v->prev = NULL;

}


Sound_renderer::Sound_renderer(char const *datafile_name)
{
   DATAFILE *f;
   char const *n;
   do_sound = -1;
   int sounds_unknown = 0;

   if (install_sound(DIGI_AUTODETECT, MIDI_AUTODETECT, NULL))
   {
      do_sound = 0;
      message(" Could not install sound driver\n");
      install_sound(DIGI_NONE, MIDI_NONE, NULL);
      num_physical_voices = max_physical_voices = 0;

      return;
   }


   if (!digi_driver || digi_driver->voices < 4)
   {
      do_sound = 0;
      message("cannot allocate enough sound voices to do any sound effects.");
      max_physical_voices = 0;
   }
   else
   {
      max_physical_voices = digi_driver->voices;
   }


   samples_datafile = load_datafile(datafile_name);

   if (samples_datafile)
   {
      message("loaded %s", datafile_name);

      for (int i = 0; i < SAMP_LAST; i++)
      {
         samples[i].s = NULL;
         samples[i].name = NULL;
      }

      for (int i = 0; i< (int)(sizeof(sample_map) / sizeof(Sample_map_info));i++)
      {

         f = find_datafile_object(samples_datafile, sample_map[i].name);
         n = sample_map[i].name;

         if ((!f) && sample_map[i].fallback)
         {
            f = find_datafile_object(samples_datafile, sample_map[i].fallback);

            if (f)
               n = sample_map[i].fallback;
         }

         samples[sample_map[i].samp].name = n;
         
         if ((!f))
         {
            if (sample_map[i].fallback)
            {
               message("could not load sample %s , nor the fallback %s, ignoring sound %d",
                        sample_map[i].name,
                        sample_map[i].fallback,
                        sample_map[i].samp);
            }
            else
            {
               message("could not load sample %s , ignoring sound %d",
                        sample_map[i].name,
                        sample_map[i].samp);
            }


            
         }
         else
         {
            samples[sample_map[i].samp].s = (SAMPLE *)(f->dat);
         }
      }


      // now check what we have loaded
      
      for (int i = 0; i < SAMP_LAST; i++)
      {
         if (!samples[i].s && !samples[i].name)
         {
            warning("the soundrenderer doesn't know which sample to use for sound %d");
            sounds_unknown = -1;
         }
      }
      if (sounds_unknown)
         warning("please add the unknown sounds to the table in client/srender/samplemap.cc");
      
      
   }
   else
   {
      do_sound = 0;
      max_physical_voices = 0;
      message("could not %s NO SOUND", datafile_name);
      
   }



   num_physical_voices = 0;
}

SAMPLE *Sound_renderer::get_sample(SAMPLES s)
{
   return samples[s].s;
}


Sound_renderer::~Sound_renderer()
{

}

Voice *Sound_renderer::get_voice(Electron *for_who)
{
   Voice *ret;

   ret = (Voice *)(new_object(CLIENT_FREELIST_VOICE));

   if (!ret)
      ret = new Voice();

   ret->set(for_who);


   voices.push(ret);
   stopped.add(ret);

   return ret;
}


void Voice::clear()
{
}



Voice *Sound_renderer::get_voice(int x, int y)
{
   Voice *ret;

   ret = (Voice *)(new_object(CLIENT_FREELIST_VOICE));

   if (!ret)
      ret = new Voice();

   ret->set(x,y);


   voices.push(ret);

   stopped.add(ret);

   return ret;

}


void Sound_renderer::free_voice(Voice *v)
{

   voices.pop(v);

   if (v->is_running)
      v->stop();
   

   // it should now be in stopped
   stopped.remove(v);

   delete_object(v);

}


int Sound_renderer::voice_available(int priority_of_triggered)
{
   if (!do_sound)
       return FALSE;

   if (num_physical_voices < max_physical_voices)
      return TRUE;

   // else find a voice with a lower priority and kick it out
   Voice *v = has_physical.head;

   Voice *lowest = v;

   ASSERT(v);

   int lowest_priority = v->compensated_priority;

   while(v->next)
   {
      v = v->next;
      if (v->compensated_priority < lowest_priority)
      {
         lowest = v;
         lowest_priority = v->compensated_priority;
      }
   }

   if (lowest_priority < priority_of_triggered)
   {
      take_phys_voice(lowest);

      return TRUE;
   }


   return FALSE;
}


void Sound_renderer::take_phys_voice(Voice *v)
{
   v->take_phys_voice();

   has_physical.remove(v);
   no_physical.add(v);

   num_physical_voices--;
   
}

void Sound_renderer::give_phys_voice(Voice *v)
{
   v->give_phys_voice();

   no_physical.remove(v);
   has_physical.add(v);

   num_physical_voices++;
   
}


void Sound_renderer::one_shot_effect(int priority, int x, int y, SAMPLES s)
{
   Voice *v;

   v = get_voice(x, y);
   v->set_sample(s);
   v->set_priority(priority);

   if (voice_available(v->compensated_priority))
   {
      give_phys_voice(v);
      one_shots.push(v);
      v->start_once();
   }
   else
   {
      free_voice(v); // it didn't get to play, so discard it immediately
   }
}

void Sound_renderer::one_shot_effect(int priority, Electron *for_who, SAMPLES s)
{
   Voice *v;

   v = get_voice(for_who);
   v->set_sample(s);
   v->set_priority(priority);

   if (voice_available(v->compensated_priority))
   {
      give_phys_voice(v);
      one_shots.push(v);
      v->start_once();
   }
   else
   {
      free_voice(v); // it didn't get to play, so discard it immediately
   }
}

static int oneshot_voice_has_finished(Object *o)
{
   return ((Voice *)o)->has_finished();
}

void Sound_renderer::render_sounds(int ear_x, int ear_y, int ear_height)
{
   Voice *v;
   int highest;
   Voice *found;

   List tmp;

   // first let all voices update their volumes and priorities
   for(voices.reset();(v = (Voice *)(voices.get())); voices.next())
   {
      if (v->is_running)
         v->update_logic(ear_x, ear_y, ear_height);

      if (v->has_phys_voice() && !v->needs_phys_voice())
      {
           take_phys_voice(v);
      }
   }

   // next see which one_shot voices can be deleted
   one_shots.filter(oneshot_voice_has_finished,&tmp);


   for(tmp.reset();(v = (Voice *)(tmp.get()));tmp.next())
   {
      free_voice(v);
   }

   // if we now have N unused physical voices left
   // we'd like to promote the N voices from the no_physical list
   // with the highest priority to the has_physical list.
   // since this would mean sorting the complete no_physical list
   // (and we don't have time for that), we only try to promote the
   // voice with the highest priority, and leave
   // the rest to the next game loop



   highest = 0;
   found = NULL;
   
   for (v = no_physical.head; v ; v = v->next)
   {
      if (v->needs_phys_voice() && v->compensated_priority > highest)
      {
         highest = v->compensated_priority;
         found = v;
      }
   }

   if (found && voice_available(highest))
   {
      give_phys_voice(found);
   }



}


void Sound_renderer::notify_stop(Voice *who)
{
    if (who->has_phys_voice())
        take_phys_voice(who);

    no_physical.remove(who);
    stopped.add(who);
}

void Sound_renderer::notify_start(Voice *who)
{
    stopped.remove(who);
    no_physical.add(who);
}



Sound_renderer *srenderer;
