#include <stdio.h>
#include "p_intern.h"


int ppe_animation_frame;

void ppe_texture_fill_bitmap(PPE_TEX *t, BITMAP *b)
{
   t->b = b;
   t->w = b->w;
   t->h = b->h;
   t->nr_frames = t->mipmap = 0;
}



PPE_TEX *ppe_create_texture(BITMAP *b)
{
   PPE_TEX *t = malloc(sizeof(PPE_TEX));

   if (!t)
      return NULL;

   t->nr_frames = 0;
   t->mipmap = 0;
   ppe_texture_fill_bitmap(t, b);
   t->mean_color = -1;
   return t;

}

/* creates an animated texture from all bitmaps in a datafile */

PPE_TEX *ppe_create_animated_texture(DATAFILE *d, int speed)
{
   PPE_TEX *t = malloc(sizeof(PPE_TEX));
   int count = 0;
   int i;
   t->mipmap = 0;

   if (!t)
      return NULL;

   t->multiplier = speed;
   t->mean_color = -1;

   for (i = 0; d[i].type != DAT_END; i++)
   {
      if (d[i].type == DAT_BITMAP)
      {
         count++;
      }
   }
   if (!count)
   {
      free(t);
      return NULL;
   }

   t->frames = (BITMAP **)malloc(sizeof(BITMAP *) * count);

   if (!t->frames)
   {
      free(t);
      return NULL;
   }
   
   count = 0;
   for (i = 0; d[i].type != DAT_END; i++)
   {
      if (d[i].type == DAT_BITMAP)
      {
         t->frames[count] = (BITMAP *)(d[i].dat);

         // use the first picture for b and the width/height
         if (!count)
            ppe_texture_fill_bitmap(t,t->frames[count]);
            
         count++;
      }
   }
   
   t->nr_frames = count;
      
   return t;

}


static int nr_mipmap_levels(int size)
{
int i;

   for (i=0;i<16;i++)
       if (size == (1 <<i)) //! table?
           return i;

   sprintf(ppe_error,"nr_mipmap_levels: texture should have a width which is a power of 2");

   return -1;
}


/*
static BITMAP *bitmap_halfsize(BITMAP *src)
{
   BITMAP *n;
   int i,j;
   int c,c0,c1,c2,c3;

   n = create_bitmap(src->w/2, src->h/2);

   if (!n)
      return NULL;

   for (i=0;i<n->w;i++)
   {
      for (j=0;j<n->h;j++)
      {
         c0 = getpixel(src, 2*i, 2*j);
         c1 = getpixel(src, 2*i+1, 2*j);
         c2 = getpixel(src, 2*i+1, 2*j+1);
         c3 = getpixel(src, 2*i, 2*j+1);

         c = makecol((getr(c0)+getr(c1)+getr(c2)+getr(c3))/4,
                     (getg(c0)+getg(c1)+getg(c2)+getg(c3))/4,
                     (getb(c0)+getb(c1)+getb(c2)+getb(c3))/4);

         putpixel(n,i,j,c);
      }
   }

   return n;
}
*/
static BITMAP *bitmap_blur(BITMAP *src, int blur)
{
   BITMAP *n;
   int i,j, k,l;
   int r,g,b,c;
   int count;

   n = create_bitmap(src->w, src->h);

   if (!n)
      return NULL;

   for (i=0;i<n->w/blur;i++)
   {
      for (j=0;j<n->h/blur;j++)
      {
         r = g = b = count = 0;
         for (k=0;k<blur;k++)
         {
            for (l=0;l<blur;l++)
            {
               c = getpixel(src, i *blur + k, j * blur +l);
               if (c != bitmap_mask_color(src))
               {
                   r += getr(c);
                   g += getg(c);
                   b += getb(c);
                   count++;
               }
            }
         }

         if (!count)
         {
            r = 255;
            g = 0;
            b = 255;
            count = 1;
         }
         c = makecol(r/count,g/count,b/count);
         for (k=0;k<blur;k++)
         {
            for (l=0;l<blur;l++)
            {
               putpixel(n,i* blur + k,j * blur + l,c);
            }
         }
      }
   }

   return n;
}

static int get_mipmap_index(PPE_TEX *tex, double scale)
{
   int inv_scale;

   int index = 0;


   if (scale < PPE_EPSILON)
      return -1;


   if (tex->mip_scaleshift > 0)
       inv_scale = (1 << tex->mip_scaleshift) / scale;
   else if (tex->mip_scaleshift < 0)
       inv_scale = 1 / (scale * (1<< (-tex->mip_scaleshift)));
   else
       inv_scale = 1 / scale;
   
   while(inv_scale)
   {
      inv_scale>>=1;
      index++;
   }

   if (index > tex->mipmap)
       index = tex->mipmap;

   return index - 1;
}


void ppe_tex_add_mipmap(PPE_TEX *tex, int max_camheight)
{
   int i,j;
   int tsz_shift = 0, img_shift = 0;
   int maxlevel;

   #ifdef DEBUGMIPMAP
          BITMAP *tmp = create_bitmap(16,8);
          ASSERT(tmp);
   #endif
   if (!max_camheight)
       max_camheight = ppe_nr_layers - 1;

   max_camheight = MIN(max_camheight, ppe_nr_layers - 1);

   if (tex->nr_frames) // animated textures cannot be mipmapped
      return;

   if (tex->mipmap)
      return;

   while(ppe_tile_size >> tsz_shift)
       tsz_shift++;

   while((tex->w)>>img_shift)
       img_shift++;
       
   tex->mip_scaleshift = img_shift - tsz_shift;
   
   tex->mipmap = nr_mipmap_levels(tex->w);


   maxlevel = get_mipmap_index(tex, PPE_SCALE(max_camheight,0));

   #ifdef DEBUGMIPMAP
     fprintf(stderr,
              "PPE mipmap generation: \n"
              "      texture (%dx%d) \n"
              "      levels %d, clipped by %d (max camheight %d) to %d\n"
              "       extra mem %d bytes (%d / level)\n",
              tex->w, tex->h,
              tex->mipmap, maxlevel+1,max_camheight,MIN(maxlevel+1, tex->mipmap),
              MIN(maxlevel+1, tex->mipmap) * tex->w * tex->h * (bitmap_color_depth(tex->b) / 8),
              tex->w * tex->h * (bitmap_color_depth(tex->b) / 8)
              );
   #endif

   tex->mipmap = MIN(maxlevel+1, tex->mipmap);

   if (tex->mipmap <= 0)
   {
      tex->mipmap = 0;
      return;
   }
   tex->frames = malloc(sizeof(BITMAP *) * tex->mipmap);

   if (!tex->frames)
   {
      tex->mipmap = 0;
      return;
   }

   
   for (i=0;i<tex->mipmap;i++)
   {

      tex->frames[i] = bitmap_blur(tex->b, 2<<i);

      
      if (!tex->frames[i])
      {
         for (j=0;j<i;j++)
             destroy_bitmap(tex->frames[i]);

         free(tex->frames);

         tex->frames = NULL;
         tex->mipmap = 0;
         return;
      }

      #ifdef DEBUGMIPMAP
             clear_to_color(tmp,1);
             textprintf(tmp,font,0,0,makecol(255,0,0),"%d",i);
             stretch_blit(tmp,tex->frames[i],0,0,tmp->w,tmp->h,0,0,tex->frames[i]->w/2,tex->frames[i]->h/2);
      #endif

   
   }

   

   
}

static BITMAP *ppe_animated_texture_get_bitmap(PPE_TEX *tex, double scale, int anim_offset)
{
       return tex->frames[((ppe_animation_frame >> tex->multiplier) + anim_offset) % tex->nr_frames];
}


static BITMAP *ppe_mipmap_texture_get_bitmap(PPE_TEX *tex, double scale )
{

   int index = get_mipmap_index(tex, scale);

   if (index < 0)
      return tex->b;

   if (index >= tex->mipmap)
   {
      abort();
   }

   return tex->frames[index];
}

BITMAP *ppe_texture_get_bitmap(PPE_TEX *tex, double scale, int anim_offset)
{

   if (tex->nr_frames)
       return ppe_animated_texture_get_bitmap(tex, scale, anim_offset);
   else if (tex->mipmap)
       return ppe_mipmap_texture_get_bitmap(tex, scale);
   else
       return tex->b;
}


void ppe_destroy_texture(PPE_TEX *t)
{

   if (t->nr_frames)
      free(t->frames);
      
   free(t);

}


extern int ppe_animated_texture_speed(PPE_TEX *t, int speed)
{
   int ret;
   
   if (!(t->nr_frames))
   {
      return -1;
   }
   
   ret = t->multiplier;
   t->multiplier = speed;

   return ret;
}







static DATAFILE *bitmaps;





/* loads a datafile with textures, the *first* entry should be
 * a palette, and all folowing should be bitmaps, doesn't work for animated textures
 */
int ppe_load_textures(char *datafilename, int load_names)
{
   int i;
   DATAFILE *tmp;

   if (ppe_textures)
      ppe_unload_textures();


   if (screen && bitmap_color_depth(screen))
   {
      tmp = load_datafile_object(datafilename, "0000_PAL");

      if (tmp)
      {
         select_palette(tmp->dat);
         unload_datafile_object(tmp);
      }
   }

   bitmaps = load_datafile(datafilename);

   if (!bitmaps)
   {
      sprintf(ppe_error,"could not load datafile %s",datafilename);
      return TRUE;
   }
   
   ppe_nr_textures = 0;
   while(bitmaps[ppe_nr_textures].type != DAT_END)
   {
       if (!ppe_nr_textures)
       {
         if (bitmaps[ppe_nr_textures].type != DAT_PALETTE)
         {
            unload_datafile(bitmaps);
            sprintf(ppe_error, "Invalid texture file, first entry should be a palette");
            return TRUE;
         }
       }
       else if (bitmaps[ppe_nr_textures].type != DAT_BITMAP)
       {

         unload_datafile(bitmaps);
         sprintf(ppe_error, "Invalid texture file");

         return TRUE;
       }

       ppe_nr_textures++;
   }
   ppe_nr_textures--; /* the first is the palette */
   ppe_textures = malloc(sizeof(PPE_TEX) * ppe_nr_textures);

   if (load_names)
   {
      ppe_texture_names = malloc(sizeof(char *) * ppe_nr_textures);
      if (!ppe_texture_names)
      {
         sprintf(ppe_error,"out of memory allocating space for texture names");
         return TRUE;
      }
      for (i = 0; i < ppe_nr_textures; i++)
      {
         ppe_texture_names[i] = get_datafile_property(bitmaps + i + 1, AL_ID('N','A','M','E'));
      }
      
   }



   if (!ppe_textures)
   {
      unload_datafile(bitmaps);
      if (load_names)
         free(ppe_texture_names);
         
      ppe_texture_names = NULL;
      sprintf(ppe_error,"out of memory allocating space for textures");
      return TRUE;
   }

   ppe_texture_palette = bitmaps[0].dat;

   for (i=0; i<ppe_nr_textures;i++)
   {
      ppe_texture_fill_bitmap(ppe_textures + i, bitmaps[i + 1].dat);
   }
   
   return FALSE;
}


void ppe_unload_textures()
{
   free(ppe_textures);
   if (ppe_texture_names)
      free(ppe_texture_names);
   ppe_textures = NULL;
   ppe_nr_textures = 0;
   ppe_texture_names = NULL;
   unload_datafile(bitmaps);
}

void ppe_tex_is_sprite(PPE_TEX *t)
{
   t->mip_scaleshift = 0;
}



