/*
 *
 *   ^   |    sssss p   ddddd  fff  ggggg hhhh   iii  j   j    |   ^
 *  /|\  |    s     p   d     f   f   g   h   h i   i jj  j    |  /|\
 *   |   |    sss   p   ddd   f       g   hhhh  i   i j j j    |   |
 *   |  \|/   s     p   d     f   f   g   h   h i   i j  jj   \|/  |
 *   |   v    sssss ppp ddddd  fff    g   h   h  iii  j   j    v   |
 *
 *                           copyright 1999
 *                  Martijn Versteegh & Hein Zelle
 *
 */
/*
 * implementation of a simple scrolling text console
 */
#include <stdio.h>
#include <ctype.h>

#include "renderer.h"

#define BUFSIZE 1024

void Renderer::cons_setup(int _w, int _h, int _num_streams)
{
    cons_numstreams = MAX(1, _num_streams);
    cons_w = _w;
    cons_h = _h;
    cons_smooth_scroll = 0;
    cons_gray = makecol(100,100,100);
    cons_white = makecol(0,255,255);
    cons_vidmem = (char *)malloc( sizeof(char) * (cons_w + 1) * cons_h * cons_numstreams);
    cons_lines = (char **)malloc(sizeof(char *) * cons_h * cons_numstreams);
    cons_color = (int *)malloc(sizeof(int) * cons_h * cons_numstreams);

    cons_x = new int[cons_numstreams];
    cons_y = new int[cons_numstreams];
    cons_curline = new double [cons_numstreams];
    cons_requested_curline = new int [cons_numstreams];
    
    if (!cons_vidmem || !cons_lines || !cons_color)
       fatal("out of memory");
       
    for (int i = 0; i < cons_h * cons_numstreams; i++)
    {
        cons_lines[i] = cons_vidmem + (cons_w + 1) * i;
    }
    cons_clear();


    cons_curstream = -1;
    cons_select_stream(0);

    cons_bgcol = -1; //makecol(0,40,0);
    cons_wwrap = TRUE;
}

void Renderer::cons_cleanup()
{
    delete [] cons_requested_curline;
    delete [] cons_curline;
    delete [] cons_x;
    delete [] cons_y;

    free(cons_vidmem);
    free(cons_lines);
    free(cons_color);
}


void Renderer::cons_printf(int col, char const *format, ...)
{

    char buf[BUFSIZE];

    va_list args;
    va_start( args, format );
    vsprintf( buf, format, args );
    va_end(args);
    fprintf(stderr, "%s",buf);
    cons_print(col, buf);
}

void Renderer::cons_print(int col, char const *string)
{
    cons_color[cons_y[cons_curstream] + cons_stream_offset] = col;

    int is_space;

    const char *i;

    
    for (char const *s = string;*s;s++)
    {
        if (*s == 13)
           continue;


        if (*s == 10)
        {
            cons_newline(col);
        }
        else
        {
            is_space = *s ? isspace(*s) : 0;

            // if we are at the start of a line, and the current char is a space
            // erase any contiuation signs in the last line,last column
            if (!cons_x[cons_curstream] && is_space)
               cons_lines[((cons_y[cons_curstream] + cons_h -1) % cons_h)+ cons_stream_offset][cons_width()-1] = '\0';
               
            cons_lines[cons_y[cons_curstream] + cons_stream_offset][cons_x[cons_curstream]] = *s;
            cons_x[cons_curstream]++;
            if (!cons_wwrap)
            {
                if (cons_x[cons_curstream] >= cons_width()-1)
                {
                    cons_lines[cons_y[cons_curstream] + cons_stream_offset][cons_width()-1] = CONTINUATION_CHAR;
                    cons_x[cons_curstream] = cons_width();
                    cons_newline(col);
                }
            }
            else
            {
                if (cons_x[cons_curstream] >= cons_width()-1)
                {
                    cons_lines[cons_y[cons_curstream] + cons_stream_offset][cons_width()-1] = CONTINUATION_CHAR;
                    cons_x[cons_curstream] = cons_width();
                    cons_newline(col);
                }
                    
                if (is_space)
                {
                    i = s + 1;
                    while( *i && !isspace(*i))
                           i++;
                           
                    if ((i - s) < (cons_width() - 1) && cons_x[cons_curstream] +  (i - s) >= (cons_width() - 1))
                    {
                       cons_newline(col);
                    }
                }
            }

            
        }
    }
    cons_lines[cons_y[cons_curstream] + cons_stream_offset][cons_x[cons_curstream]] = 0;
    
    cons_requested_curline[cons_curstream] = cons_y[cons_curstream];
}

void Renderer::cons_newline(int col)
{
    cons_lines[cons_y[cons_curstream] + cons_stream_offset][cons_x[cons_curstream]] = 0;
    cons_x[cons_curstream]=0;
    cons_y[cons_curstream]++;
    cons_y[cons_curstream] %= cons_h;
    cons_color[cons_y[cons_curstream] + cons_stream_offset] = col;
    cons_mark_end();
}

void Renderer::cons_backspace()
{
    cons_lines[cons_y[cons_curstream] + cons_stream_offset][cons_x[cons_curstream]] = 0;
    cons_x[cons_curstream]--;

    if (cons_x[cons_curstream] <0)
    {
        cons_x[cons_curstream] = cons_w - 1;
        cons_y[cons_curstream]--;
        if (cons_y[cons_curstream] < 0)
         cons_y[cons_curstream] += cons_h;
        
    }
    cons_lines[cons_y[cons_curstream] + cons_stream_offset][cons_x[cons_curstream]] = 0;
    cons_requested_curline[cons_curstream] = cons_y[cons_curstream];
}

void Renderer::cons_clear()
{
    for (int i = 0; i < cons_h * cons_numstreams; i++)
    {
        cons_lines[i][cons_w] = 0;
        cons_lines[i][0] = 0;
        cons_color[i] = 0;
    }

    for (int i=0;i<cons_numstreams;i++)
    {
      cons_x[i] = cons_y[i] = 0;
      cons_curline[i] = 0;
      cons_requested_curline[i] = 0;
    }
}



void Renderer::cons_scroll_back(int lines)
{
   cons_requested_curline[cons_curstream] -= lines;
   
   while (cons_requested_curline[cons_curstream] <0)
      cons_requested_curline[cons_curstream] += cons_h;
      
   while (cons_requested_curline[cons_curstream] >= cons_h)
      cons_requested_curline[cons_curstream] -= cons_h;

}


void Renderer::cons_set_smooth_scroll(int s)
{
   cons_smooth_scroll = s;
}


void Renderer::cons_mark_end()
{
   int i;
   int last = (cons_y[cons_curstream] + 1) % cons_h;
   last += cons_stream_offset;

   for (i=0;i<cons_w;i++)
   {
      cons_lines[last][i] = '=';
   }

   cons_lines[last][cons_w] = 0;
   cons_color[last] = cons_gray;
}


int Renderer::cons_getline()
{
   return cons_y[cons_curstream];
}


int Renderer::cons_getcol()
{
   return cons_color[cons_y[cons_curstream] + cons_stream_offset];
}

int Renderer::cons_getcolumn()
{
   return cons_x[cons_curstream];
}


int Renderer::cons_select_stream(int s)
{
   int oldstream = cons_curstream;

   if (s == oldstream)
       return oldstream;

   s = MAX(0, s);
   s = MIN(s, cons_numstreams - 1);

   cons_curstream = s;

   cons_stream_offset = cons_h * s;
   cons_requested_curline[cons_curstream] = cons_y[cons_curstream];

   cons_curline[cons_curstream] = cons_y[cons_curstream];

   return oldstream;
}

int Renderer::cons_get_stream()
{
   return cons_curstream;
}



int Renderer::cons_height()
{
   return cons_h;
}
