/*
 *
 *   ^   |    sssss p   ddddd  fff  ggggg hhhh   iii  j   j    |   ^
 *  /|\  |    s     p   d     f   f   g   h   h i   i jj  j    |  /|\
 *   |   |    sss   p   ddd   f       g   hhhh  i   i j j j    |   |
 *   |  \|/   s     p   d     f   f   g   h   h i   i j  jj   \|/  |
 *   |   v    sssss ppp ddddd  fff    g   h   h  iii  j   j    v   |
 *
 *                           copyright 1999
 *                  Martijn Versteegh & Hein Zelle
 *
 */
#include "common/defs.h"
#include "parse.h"
#include "client/clutils.h"
#include <ctype.h>
#include <string.h>
#include "client/commbuf.h"
#include "parshlpr.h"
#include "alias.h"
#include "history.h"
#include "vars.h"
#include "common/act_type.h"
#include "client/configfile.h"
#include "client/keys.h"
#include "client/renderer/renderer.h"

#define _CMD_INCLUDE_CODE_
#include "cmds.h"
#undef _CMD_INCLUDE_CODE_



static void execute(int command, int num_args);



char command_buffer[MAX_COMMAND_LENGTH] = "";
int command_pos = 0;

enum
{
   SHOW_CONFIG = 1,
   SHOW_SYSTEM,
   SHOW_VERSION,
   SHOW_KEYBIND
} show_args;


static char *parts[MAXPARTS];

static void check_commands()
{
   if ((sizeof(parser_commands) / sizeof(PARSER_CMD)) != P_LAST)
      fatal(" size of parser command table doesn't match number of commands in client/cmds.h");
}


static void add_system_vars()
{
   add_sys_var("north",DIR_N);
   add_sys_var("south",DIR_S);
   add_sys_var("east",DIR_E);
   add_sys_var("west",DIR_W);
   add_sys_var("config", SHOW_CONFIG);
   add_sys_var("system", SHOW_SYSTEM);
   add_sys_var("version", SHOW_VERSION);
   add_sys_var("keybindings", SHOW_KEYBIND);

   for (int i=1;i<ACTOR_LAST;i++)
   {
      add_sys_var(global_actor_statics[i]->type_name, i);
   }
}

void parser_init()
{
   check_commands();
   aliases_init();
   hist_init();
   vars_init();

   add_system_aliases();
   add_system_vars();

   command_buffer[0] = 0;
}



void parser_exit()
{
   aliases_exit();
   vars_exit();
   hist_exit();

}

void hprint(char *string)
{
   renderer->cons_print(lightblue, string);
   renderer->cons_print(lightblue, "\n");
}

void hprintf(char *fmt, ...)
{
    char buf[1024];//? groot genoeg voor help strings?
    va_list args;
    va_start( args, fmt );
    #ifdef vsnprintf
    vsnprintf( buf, 1024, fmt, args );
    #else
    vsprintf( buf, fmt, args );
    #endif
    va_end(args);
    hprint(buf);
}




static int is_numerical(char *string)
{
   int isnonumber = FALSE;
   int len =0;
   

   for (int i=0;string[i];i++)
   {
      if ((string[i] < '0' || string[i] > '9') && !isspace(string[i]))
          isnonumber = TRUE;
      len++;
   }

   if (!len)
   {
      warning("empty string checked");
      return FALSE;
   }

   if (!isnonumber)
      return TRUE;

   // it might be a variable
   return checkvar(string, NULL);
}


static int numerical_value(char *string)
{
   int isnonumber = FALSE;
   int val = 0;
//   Electron *e;
//   Atom *a;

   for (int i=0;string[i];i++)
   {
      if ((string[i] < '0' || string[i] > '9') && !isspace(string[i]))
      {
//          warning("%s is no number", string);
          isnonumber = TRUE;
      }
   }

   if (!isnonumber)
   {
      val =  atoi(string);
   }
   else if (!checkvar(string, &val))
   {
      severe("value neither numerical nor a variable: '%s'",string);
      return 0;
   }
/*
   if (val < ACTOR_LAST)
   {
     a = lattice->get(me->gx(), me->gy());
     ASSERT(a);

     e = find_with_type(a, val);

     if (!e)
     {
         warning("no Object of this type found at location (%d, %d)", me->gx(), me->gy());
         return 0;
     }

     val = e->actor_id;
   }
  */
   return val;
}




static void print_help()
{
   hprint("help:");
   for (int i = 0;i < P_LAST;i++)
       hprintf("  %s",parser_commands[i].command);

   alias_print_help(NULL);

   hprint("type 'help <command>' for more info about a command");
   hprint("commands can be abbreviated");
   
}


static int split(char *command) // replaces all whitespace in the string with 0's and setsup the parts[] array to point ot each part, returns the nr of parts
{
   char *s;
   int p = 0;


   s = command;

   if (*s && !isspace(*s) && !(*s == '"'))
   {
      p = 1;
      parts[0] = s;
   }
   
   while (*s)
   {
      if (*s == '"')
      {
         p++;
         parts[p-1] = s + 1;
         while(*(++s) && *s != '"');

         if (*s != '"')
         {
            warning("unmatched \" ");
            return 0;
         }

         *s = 0;
      }
      else if (isspace(*s))
      {
          *s = 0;

          if (p == MAXPARTS)
             return p;
          
          if (*(s+1) && (!isspace(*(s+1)) ) && (*(s+1) != '"'))
          {
            p++;
            parts[p-1] = s + 1;
          }
      }
      s++;
   }

//   warning("split: %d",p);
//   for (int i=0;i<p;i++)
//       warning(">%s<",parts[i]);

   return p;
}


static int find_command(char const *cmd)
{
   for (int i=0;i<P_LAST;i++)
   {
      if (!strncmp(cmd, parser_commands[i].command, strlen(cmd)))
      {
         return i;
      }
   }

   return -1;
}

int find_command_exact(char const *cmd)
{
   for (int i=0;i<P_LAST;i++)
   {
      if (!strcmp(cmd, parser_commands[i].command))
      {
         return i;
      }
   }

   return -1;
}


static int check_command(int num_parts)
{
   int c = find_command(parts[0]);

   if (c<0)
   {
      response("unknown command: '%s' ", parts[0]);
      return -1;
   }
   
   if (num_parts -1 > parser_commands[c].num_args)
   {
      response("too many arguments: '%s' takes at most %d arguments",parser_commands[c].command,parser_commands[c].num_args);
      return -1;
   }

   if (num_parts -1 < parser_commands[c].num_needed_args)
   {
      response("too few arguments: '%s' takes at least %d arguments",parser_commands[c].command,parser_commands[c].num_needed_args);
      return -1;
   }

   // need to check type of args
   for (int i=0;i<num_parts-1;i++)
   {
      if (parser_commands[c].type_args[i] == 'n')
      {
         if (!is_numerical(parts[i+1]))
         {
            response("'%s' should be a number or a variable", parts[i+1]);
            return -1;
         }
      }
   
   }


   return c;
}


// translates a number lower than ACTOR_LAST to an id
// of an actor at the current location or int he electron if it is not NULL
static int n2id(int val, Electron *in_inv = NULL, int recursive = FALSE)
{
   Electron *e;
   Atom *a;

   if (val >= ACTOR_LAST)
       return val;

   if (!in_inv)
   {
        a = lattice->get(me->gx(), me->gy());
        ASSERT(a);
   
        e = find_with_type(a, val);
   
        if (!e)
        {
            warning("no Object of this type found at location (%d, %d)", me->gx(), me->gy());
            return 0;
        }
   
   }
   else
   {
        e = find_with_type(in_inv, val, recursive);
        if (!e)
        {
            warning("no Object of that type found");
            return 0;
        }
        
   }
   val = e->actor_id;
   
   return val;
}





static void execute_command(char *command)
{
   int num_parts = split(command);
   int c;

   if (!me) // if the player is not yet known, the parser can do nothing
      return;

   if (!num_parts)
      return;

   if ((c = find_command_exact(command)) <0)
   {
      if (check_aliases(parts[0], num_parts-1, parts+1))
          return;
   }
   if ((c = check_command(num_parts))<0)
      return;

   execute(c, num_parts - 1);
}

int parse()
{

//   if (me->gx() < 0)
//       return 0; //! moet uiteindelijk alleen sommige commando's blokkeren

   if (command_buffer[0] == '!')
   {
       hist_find(command_buffer);
       history_add(command_buffer); //?
   }
   else
   {
       history_add(command_buffer);
   }
   
   lowlevel_parse();

   command_buffer[0] = 0;
   command_pos = 0;

   return 0;
}

void lowlevel_parse()
{
   char *next_cmd;
   char local_cmd_buf[MAX_COMMAND_LENGTH];
   char *command = local_cmd_buf;
   // copy the command into our own processing buffer
   sstrncpy(command, command_buffer, MAX_COMMAND_LENGTH);

   
   while((next_cmd = find_separator(command, ';')))
   {
       *next_cmd = 0;
       execute_command(command);

       command = next_cmd + 1;
   }

   if (*command)
        execute_command(command);

}


void parser_help(char *command)
{
   int c = find_command(command);

   if (c <0)
   {
      if (alias_print_help(command))
         warning("unkown command : %s", command);
         
      return;
   }

   if (parser_commands[c].help)
   {
      hprintf("%s:",parser_commands[c].command);
      hprintf(" %s",parser_commands[c].help);
   }
   else
      hprintf("no help for command %s (sorry)",parser_commands[c].command);
   
}


static void execute(int command, int num_args)
{
   // whaargrbl , nasty switch code, now we really ought
   // to write a decent bison/flex parser ;-)
   Electron *e;
   int what;

   int args[MAXPARTS - 1];

   for (int i=0;i<MAXPARTS-1;i++)
       args[i] = 0;

   switch(command)
   {
    case P_NORTH:
      cbuf_add_command(C_MOVEPLAYER, DIR_N);
      break;
    case P_SOUTH:
      cbuf_add_command(C_MOVEPLAYER, DIR_S);
      break;
    case P_EAST:
      cbuf_add_command(C_MOVEPLAYER, DIR_E);
      break;
    case P_WEST:
      cbuf_add_command(C_MOVEPLAYER, DIR_W);
      break;
    case P_DO_COMMAND:
      for (int i=0;i< num_args;i++)
          args[i] = numerical_value(parts[i+1]);
          
      cbuf_add_generic_commandpacket(args[0],-1,-1,args[1],args[2],args[3],args[4],args[5], args[6]);
      break;
    case P_HELP:
      if (!num_args)
         print_help();
      else
         parser_help(parts[1]);
      break;
    case P_LOOK:
      if (num_args)
      {
          // first look in our parent's inventory
          what = n2id(numerical_value(parts[1]), me->parent, FALSE);
          // then look in our own's inventory
          if (!what)
              what = n2id(numerical_value(parts[1]), me, TRUE);
          // then look in the atom
          if (!what)
              what = n2id(numerical_value(parts[1]));
          
          look(what);
      }
      else
          look(-1);
      break;
    case P_INV:
      if (num_args)
          inventory(n2id(numerical_value(parts[1])));
      else
          inventory(-1);

      break;

    case P_PICKUP:
      if (num_args)
          pickup(n2id(numerical_value(parts[1]),me->parent, FALSE));
      else
          browse(me->get_atom(),pickup);

      break;
    case P_DROP:
      drop(n2id(numerical_value(parts[1]), me, TRUE));
      break;
    case P_PUT:
      put(n2id(numerical_value(parts[1]), me, TRUE), n2id(numerical_value(parts[2]), me, TRUE));
      break;
    case P_ENTER:
      if (num_args)
          do_enter(n2id(numerical_value(parts[1])));
      else
          do_enter(-1);
      break;
    case P_EXIT:
         do_exit();
      break;

    case P_ALIAS:
      if (num_args == 2)
      {
         add_alias(parts[1], parts[2]);
      }
      else if (num_args == 1)
      {
         print_alias(parts[1]);
      }
      else
         print_alias(NULL);
         
      break;
    case P_UNALIAS:
      remove_alias(parts[1]);
      break;
    case P_SET:
      if (num_args == 2)
      {
         add_var(parts[1], numerical_value(parts[2]));
      }
      else if (num_args == 1)
      {
         print_var(parts[1]);
      }
      else
         print_var(NULL);
         
      break;

    case P_SHOW:
      switch(numerical_value(parts[1]))
      {
       case SHOW_CONFIG:
        print_config();
        break;
       case SHOW_SYSTEM:
        print_sysinfo();
        break;
       case SHOW_VERSION:
        print_version();
        break;
       case SHOW_KEYBIND:
        print_key(-1,key_shifts);
        break;
       default:
        warning("show: don't recognise argument: %s",parts[1]);
        break;
      }

      break;
    case P_SETNUMVAR:
      args[0] = n2id(numerical_value(parts[1]));
      args[1] = numerical_value(parts[2]);
      args[2] = numerical_value(parts[3]);

      e = find_with_id(storage, args[0]);
      if (!e)
      {
           warning("Object %d does not exist", args[0]);
      }
      else
      {
           cbuf_add_generic_numpacket(e->actor_type,e->actor_id,args[1], args[2]);
      }
      break;
    case P_GETNUMVAR:
      args[0] = n2id(numerical_value(parts[1]));
      args[1] = numerical_value(parts[2]);

      e = find_with_id(storage, args[0]);
      
      if (!e)
      {
           warning("Object %d does not exist", args[0]);
      }
      else
      {
           message("Object %d (%s): var %d has value %d",e->actor_id, e->type_name, args[1], e->get_int_var(args[1]));
      }
      break;
    case P_GETSTRINGVAR:
      args[0] = n2id(numerical_value(parts[1]));
      args[1] = numerical_value(parts[2]);

      e = find_with_id(storage, args[0]);
      
      if (!e)
      {
           warning("Object %d does not exist", args[0]);
      }
      else
      {
           message("Object %d (%s): var %d has value %s",e->actor_id, e->type_name, args[1], e->get_string_var(args[1]));
      }
      break;
    case P_HISTORY:
      hist_print();
      break;

    case P_BIND:
        bind_key();
      break;
    case P_UNBIND:
        unbind_key();
      break;
    case P_QUIT:
      playing = FALSE;
      break;

    case P_SCREENSHOT:
      if (num_args)
          renderer->screenshot(parts[1]);
      else
          renderer->screenshot("");
      break;
    case P_ZOOM:
      what = numerical_value(parts[1]);
      if (what > 2)
          requested_camheight = what;
      break;
    case P_DUMPLIST:
      dump_freelist_stats();
      break;
    case P_FPS:
      message("fps : %g, avg. fps.: %g, last 5 secs: %g",fpscount, avfpscount, smoothfpscount);
      break;
    case P_SHUTDOWN_SERVER:
      cbuf_add_command(C_NUMPACK, 0, ACTOR_SYSTEM, SYS_MESSAGE, System::CTS_SHUTDOWN);
      break;
    case P_SWITCHFONT:
      renderer->cons_switch_font();
      break;
    case P_SHOOT:
      if (!num_args)
      {
         target(me, shoot_at_player_if_there_else_location);
      }
      else if (num_args == 1)
      {
         args[0] = numerical_value(parts[1]);
         what = args[0];
         if (what < ACTOR_LAST)
         {
            message("%s should be a player", parts[1]);
            return;
         }
         e = find_with_id(storage, what);

         if (!e)
         {
            message("No such player : %s",parts[1]);
         }
//         args[0] = e->gx();
//         args[1] = e->gy();
         cbuf_add_generic_commandpacket(COMMAND_FIRE_AT,-1, -1,e->actor_type, e->actor_id);
      }
      else
      {
        args[0] = numerical_value(parts[1]);
        args[1] = numerical_value(parts[2]);
        cbuf_add_generic_commandpacket(COMMAND_FIRE_TO,-1,-1,args[0],args[1]);
      }

        
      break;
    case P_DEBUG:
     if (!num_args)
        debuggingcommand(0);
     else
        debuggingcommand(numerical_value(parts[1]));

      break;
    default:
      warning("bug: command %d not implemented in funcion 'void execute(int cmd)' file 'parse.cc'", command);
      break;
   }

}


