/*
 *
 *   ^   |    sssss p   ddddd  fff  ggggg hhhh   iii  j   j    |   ^
 *  /|\  |    s     p   d     f   f   g   h   h i   i jj  j    |  /|\
 *   |   |    sss   p   ddd   f       g   hhhh  i   i j j j    |   |
 *   |  \|/   s     p   d     f   f   g   h   h i   i j  jj   \|/  |
 *   |   v    sssss ppp ddddd  fff    g   h   h  iii  j   j    v   |
 *
 *                           copyright 1999
 *                  Martijn Versteegh & Hein Zelle
 *
 */
#include <string.h>
#include "client/client.h"
#include "alias.h"
#include "parse.h"
#include "client/renderer/renderer.h"

typedef struct ALIAS
{
   char *alias;
   char *commands;
   int num_args;
   int system; // system aliases are read only and have help
   char const *help;
   int currently_executed; // used to check for recursive aliases
} ALIAS;


static int num_aliases;
static int max_num_aliases;
static ALIAS *aliases;

static void save_aliases()
{
   int count = 0;
   for (int i=0;i < num_aliases;i++)
   {
      if (!(aliases[i].system))
         count++;
   }
   set_config_int("aliases","num_aliases",count);

   count = 0;
   for (int i=0;i < num_aliases;i++)
   {
      if (!(aliases[i].system))
      {
         sprintf(command_buffer,"alias%03d",count);
         set_config_string("aliases",command_buffer,aliases[i].alias);
         set_config_string("aliases",aliases[i].alias,aliases[i].commands);
         count++;
      }
   }

   command_buffer[0] = 0;
}

static void load_aliases()
{
   int num_aliases_to_load = get_config_int("aliases","num_aliases",0);
   char *tmp_string;

   for (int i=0;i<num_aliases_to_load;i++)
   {
       sprintf(command_buffer,"alias%03d",i);
       tmp_string = sstrdup(get_config_string("aliases", command_buffer, "notfound"));
       if (strcmp(tmp_string, "notfound"))
       {
            if (strcmp(get_config_string("aliases",tmp_string,"couldnotloadalias"),"couldnotloadalias"))
                add_alias(tmp_string, get_config_string("aliases",tmp_string,"couldnotloadalias"));
       }
       else
       {
            warning("alias %s not found in configfile",i);
       }
       free(tmp_string);
   }

}

void aliases_init()
{
    num_aliases = max_num_aliases = 0;
    aliases = NULL;

    load_aliases();
}

void aliases_exit()
{
  //! cleanup
    save_aliases();
}


static int alias_count_args(char const *c)
{
   int l = strlen(c);
   int a, max;

   max = 0;

   for (int i=0;i<l-1;i++)
   {
      if (c[i] == '%')
      {
           a = c[i+1] - '0' + 1;

           if (a <1 || a >10)
           {
               message("illegal argument:\%%c",c[i+1]);
               return -1;
           }

           if (a > max)
              max = a;
      }

   }

   return max;
}

static int find_alias(char const *alias)
{
   for (int i=0;i<num_aliases;i++)
   {
      if (!strncmp(alias, aliases[i].alias, strlen(alias)))
      {
         return i;
      }

   }
   return -1;
}

static int find_alias_exact(char const *alias)
{
   for (int i=0;i<num_aliases;i++)
   {
      if (!strcmp(alias, aliases[i].alias))
      {
         return i;
      }

   }
   return -1;
}

// WARNING, help MUST STAY a valid string during the program's lifetime
void add_system_alias(char const *alias, char const *commands, char const *help)
{
   int n_args = alias_count_args(commands);

   if (n_args <0)
      return;

   int a = find_alias_exact(alias);


   if (a>=0)
      fatal("cannot redefine system alias");

   if (strchr(commands, ';'))
       fatal("add_system_alias: system aliases cannot contain ;'s");

   ALIAS *tmp;
   num_aliases++;
   if (num_aliases > max_num_aliases)
   {
      max_num_aliases+=10;
      tmp = (ALIAS *)realloc(aliases, max_num_aliases * sizeof(ALIAS));

      if (!tmp)
      {
         warning("not enough memory to add system aliases");
         max_num_aliases-=10;
         num_aliases--;
         return;
      }

      aliases = tmp;
   }

//   message(" add_system_alias %d: %s", num_aliases - 1, alias);
   aliases[num_aliases - 1].alias = sstrdup(alias);
   aliases[num_aliases - 1].commands = sstrdup(commands);
   aliases[num_aliases - 1].num_args = n_args;
   aliases[num_aliases - 1].system = TRUE;
   aliases[num_aliases - 1].help = help;
   aliases[num_aliases - 1].currently_executed = FALSE;
}

void add_alias(char const *alias, char const *commands)
{
   char buffer[256];
   int n_args = alias_count_args(commands);

   sstrncpy(buffer, commands, 256);


   if (n_args <0)
      return;

   if (!n_args)
   {
      sstrncat(buffer," %0 %1 %2 %3 %4 %5 %6 %7 %8 %9", 256);
      n_args = 10;
   }

   int a = find_alias_exact(alias);
//   message("add alias : found %d",a);
   
   if (a>=0)
   {
      if (aliases[a].system)
      {
         message("cannot redefine built in commands");
         return;
      }
      free(aliases[a].commands);
      
      aliases[a].commands = sstrdup(buffer);
      aliases[a].num_args = n_args;
      return;
   }


   a = find_command_exact(alias);

   if (a>=0)
   {
      message("cannot redefine built in commands");
      return;
   }

   ALIAS *tmp;
   num_aliases++;
   if (num_aliases > max_num_aliases)
   {
      max_num_aliases+=10;
      tmp = (ALIAS *)realloc(aliases, max_num_aliases * sizeof(ALIAS));

      if (!tmp)
      {
         warning("not enough memory to add aliases");
         max_num_aliases-=10;
         num_aliases--;
         return;
      }

      aliases = tmp;
   }

   aliases[num_aliases - 1].alias = sstrdup(alias);
   aliases[num_aliases - 1].commands = sstrdup(buffer);
   aliases[num_aliases - 1].num_args = n_args;
   aliases[num_aliases - 1].system = FALSE;
   aliases[num_aliases - 1].help = NULL;
   aliases[num_aliases - 1].currently_executed = FALSE;
   
}

static void del_alias(int alias)
{
   free(aliases[alias].alias);
   free(aliases[alias].commands);

   num_aliases--;
   for (int i=alias;i<num_aliases;i++)
   {
      aliases[i] = aliases[i+1];
   }
}

void remove_alias(char const *alias)
{
      int a = find_alias_exact(alias);
      if (a < 0)
      {
         message("alias not found : %s", alias);
         return;
      }

      if (aliases[a].system)
      {
          message("cannot delete built in command %s", alias);
          return;
      }

      del_alias(a);
}


static int write_cmd_string(char *into, char *cmd,char **args, int num_args,int maxlen)
{
   int fp = 0, tp=0;
   int a;
   maxlen--;

//   message("write_cmd_string:%s",cmd);

   while(cmd[fp] && cmd[fp] != ';' && tp < maxlen)
   {
      if (cmd[fp] == '%')
      {
         fp++;
         a = cmd[fp] - '0';

         if (a < num_args)
         {
             strncpy(into+tp, args[a], MIN(maxlen - tp, (int)strlen(args[a])));
             
             tp += MIN(maxlen - tp, (int)strlen(args[a])) - 1; // max length should generate a warning
         }
         else
         {
            into[tp] = ' ';
         }
         
      }
      else
      {
         into[tp] = cmd[fp];
      }

      fp++;
      tp++;
   }
   into[tp] = 0;

//   message("result:%s",into);

   // in case of half used commands, wind to the end of the command
   while( cmd[fp] && cmd[fp] != ';')
       fp++;

   return fp;
}

static void execute_alias(int alias, char **args, int num_args)
{

   int i = 0;
   int l = strlen(aliases[alias].commands);

   if (aliases[alias].currently_executed)
   {
       warning(" recursive command detected, ignoring comand %s",aliases[alias].commands);
       return;
   }
   aliases[alias].currently_executed = TRUE;

   while(i < l)
   {
//      ("wcmd for %s at pos %d", aliases[alias].commands, i);
      i += write_cmd_string(command_buffer, aliases[alias].commands + i, args,num_args, 1024);

      if (!aliases[alias].system)
      {
          renderer->cons_print(blue, command_buffer);
          renderer->cons_print(blue, "\n");
      }
      lowlevel_parse(); /* feed back our newly created alias string to the parser */

      i++;
   }
   aliases[alias].currently_executed = FALSE;
}

int check_aliases(char const *command, int num_args, char **args)
{
   int a;

   a = find_alias_exact(command);
   if (a < 0)
       a = find_alias(command);

   if (a<0)
      return FALSE;

   if (num_args > aliases[a].num_args)
   {
      message("command %s takes at most %d arguments",aliases[a].alias, aliases[a].num_args);
      return TRUE;
   }


   if (aliases[a].system && num_args != aliases[a].num_args)
   {
      message("command %s takes %d arguments",aliases[a].alias, aliases[a].num_args);
      return TRUE;
   }


   execute_alias(a, args, num_args);


   return TRUE;
}


void print_alias(char const *which)
{
   int a;
   if (!which)
   {
      for (int i=0;i<num_aliases;i++)
      {
         if (!aliases[i].system)
            renderer->cons_printf(lightblue,"%s: %s\n", aliases[i].alias,aliases[i].commands);
      }
      return;
   }

   a = find_alias_exact(which);
   if (a < 0)
       a = find_alias(which);

   if (a<0)
      message("no such alias : %s",which);
   else
      renderer->cons_printf(lightblue,"%s: %s\n", aliases[a].alias,aliases[a].commands);

}


int alias_print_help(char const *alias)
{

   if (!alias)
   {
      for (int i=0;i<num_aliases;i++)
      {
         hprintf("  %s",aliases[i].alias);
      }
      return 0;
   }

   int v;

   v = find_alias_exact(alias);

   if (v<0)
      v = find_alias(alias);

   if (v <0)
      return -1;

   if (aliases[v].help)
   {
      hprintf("%s:",aliases[v].alias);
      hprintf(" %s",aliases[v].help);
   }
   else
      hprintf("no help for '%s', sorry.",aliases[v].alias);
   
   return 0;
}
